package vazkii.patchouli.common.util;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import org.apache.commons.lang3.tuple.ImmutableTriple;
import org.apache.commons.lang3.tuple.Triple;

import vazkii.patchouli.common.book.Book;
import vazkii.patchouli.common.book.BookRegistry;
import vazkii.patchouli.common.item.ItemModBook;

import org.jetbrains.annotations.Nullable;

import java.util.*;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2487;
import net.minecraft.class_2522;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_6862;
import net.minecraft.class_7923;
import net.minecraft.class_7924;

public final class ItemStackUtil {
	private static final Gson GSON = new GsonBuilder().create();

	private ItemStackUtil() {}

	public static Triple<class_2960, Integer, class_2487> parseItemStackString(String res) {
		String nbt = "";
		int nbtStart = res.indexOf("{");
		if (nbtStart > 0) {
			nbt = res.substring(nbtStart).replaceAll("([^\\\\])'", "$1\"").replaceAll("\\\\'", "'");
			res = res.substring(0, nbtStart);
		}

		String[] upper = res.split("#");
		String count = "1";
		if (upper.length > 1) {
			res = upper[0];
			count = upper[1];
		}

		String[] tokens = res.split(":");
		if (tokens.length < 2) {
			throw new RuntimeException("Malformed item ID " + res);
		}

		class_2960 key = new class_2960(tokens[0], tokens[1]);
		int countn = Integer.parseInt(count);
		class_2487 tag = null;

		if (!nbt.isEmpty()) {
			try {
				tag = class_2522.method_10718(nbt);
			} catch (CommandSyntaxException e) {
				throw new RuntimeException("Failed to parse ItemStack JSON", e);
			}
		}

		return ImmutableTriple.of(key, countn, tag);
	}

	public static class_1799 loadFromParsed(Triple<class_2960, Integer, class_2487> parsed) {
		var key = parsed.getLeft();
		var count = parsed.getMiddle();
		var nbt = parsed.getRight();
		Optional<class_1792> maybeItem = class_7923.field_41178.method_17966(key);
		if (maybeItem.isEmpty()) {
			throw new RuntimeException("Unknown item ID: " + key);
		}
		class_1792 item = maybeItem.get();
		class_1799 stack = new class_1799(item, count);

		if (nbt != null) {
			stack.method_7980(nbt);
		}
		return stack;
	}

	public static class_1799 loadStackFromString(String res) {
		return loadFromParsed(parseItemStackString(res));
	}

	public static class_1856 loadIngredientFromString(String ingredientString) {
		return class_1856.method_8101(loadStackListFromString(ingredientString).toArray(new class_1799[0]));
	}

	public static List<class_1799> loadStackListFromString(String ingredientString) {
		String[] stacksSerialized = splitStacksFromSerializedIngredient(ingredientString);
		List<class_1799> stacks = new ArrayList<>();
		for (String s : stacksSerialized) {
			if (s.startsWith("tag:")) {
				var key = class_6862.method_40092(class_7924.field_41197, new class_2960(s.substring(4)));
				class_7923.field_41178.method_40266(key).ifPresent(tag -> tag.method_40239().forEach(item -> stacks.add(new class_1799(item))));
			} else {
				stacks.add(loadStackFromString(s));
			}
		}
		return stacks;
	}

	public static StackWrapper wrapStack(class_1799 stack) {
		return stack.method_7960() ? StackWrapper.EMPTY_WRAPPER : new StackWrapper(stack);
	}

	@Nullable
	public static Book getBookFromStack(class_1799 stack) {
		if (stack.method_7909() instanceof ItemModBook) {
			return ItemModBook.getBook(stack);
		}

		Collection<Book> books = BookRegistry.INSTANCE.books.values();
		for (Book b : books) {
			if (class_1799.method_7984(b.getBookItem(), stack)) {
				return b;
			}
		}

		return null;
	}

	public static class StackWrapper {

		public static final StackWrapper EMPTY_WRAPPER = new StackWrapper(class_1799.field_8037);

		public final class_1799 stack;

		public StackWrapper(class_1799 stack) {
			this.stack = stack;
		}

		@Override
		public boolean equals(Object obj) {
			return obj == this || (obj instanceof StackWrapper && class_1799.method_7984(stack, ((StackWrapper) obj).stack));
		}

		@Override
		public int hashCode() {
			return stack.method_7909().hashCode();
		}

		@Override
		public String toString() {
			return "Wrapper[" + stack.toString() + "]";
		}

	}

	private static String[] splitStacksFromSerializedIngredient(String ingredientSerialized) {
		final List<String> result = new ArrayList<>();

		int lastIndex = 0;
		int braces = 0;
		Character insideString = null;
		for (int i = 0; i < ingredientSerialized.length(); i++) {
			switch (ingredientSerialized.charAt(i)) {
			case '{':
				if (insideString == null) {
					braces++;
				}
				break;
			case '}':
				if (insideString == null) {
					braces--;
				}
				break;
			case '\'':
				insideString = insideString == null ? '\'' : null;
				break;
			case '"':
				insideString = insideString == null ? '"' : null;
				break;
			case ',':
				if (braces <= 0) {
					result.add(ingredientSerialized.substring(lastIndex, i));
					lastIndex = i + 1;
					break;
				}
			}
		}

		result.add(ingredientSerialized.substring(lastIndex));

		return result.toArray(new String[0]);
	}

	public static class_1799 loadStackFromJson(JsonObject json) {
		// Adapted from net.minecraftforge.common.crafting.CraftingHelper::getItemStack
		String itemName = json.get("item").getAsString();

		class_1792 item = class_7923.field_41178.method_17966(new class_2960(itemName)).orElseThrow(() -> new IllegalArgumentException("Unknown item '" + itemName + "'")
		);

		class_1799 stack = new class_1799(item, class_3518.method_15282(json, "count", 1));

		if (json.has("nbt")) {
			try {
				JsonElement element = json.get("nbt");
				class_2487 nbt;
				if (element.isJsonObject()) {
					nbt = class_2522.method_10718(GSON.toJson(element));
				} else {
					nbt = class_2522.method_10718(element.getAsString());
				}
				stack.method_7980(nbt);
			} catch (CommandSyntaxException e) {
				throw new IllegalArgumentException("Invalid NBT Entry: " + e, e);
			}
		}

		return stack;
	}
}
