package vazkii.patchouli.client.book.gui;

import vazkii.patchouli.api.IComponentRenderContext;
import vazkii.patchouli.client.base.PersistentData;
import vazkii.patchouli.client.base.PersistentData.BookData;
import vazkii.patchouli.client.base.PersistentData.Bookmark;
import vazkii.patchouli.client.book.BookEntry;
import vazkii.patchouli.client.book.BookPage;
import vazkii.patchouli.common.book.Book;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_4185;
import net.minecraft.class_437;

public class GuiBookEntry extends GuiBook implements IComponentRenderContext {

	protected final BookEntry entry;
	@Nullable private BookPage leftPage;
	@Nullable private BookPage rightPage;

	public GuiBookEntry(Book book, BookEntry entry) {
		this(book, entry, 0);
	}

	public GuiBookEntry(Book book, BookEntry entry, int spread) {
		super(book, entry.getName());
		this.entry = entry;
		this.spread = spread;
	}

	@Override
	public void method_25426() {
		super.method_25426();

		maxSpreads = (int) Math.ceil((float) entry.getPages().size() / 2);
		setupPages();
	}

	@Override
	public void onFirstOpened() {
		super.onFirstOpened();

		boolean dirty = false;
		var key = entry.getId();

		BookData data = PersistentData.data.getBookData(book);

		if (!data.viewedEntries.contains(key)) {
			data.viewedEntries.add(key);
			dirty = true;
			entry.markReadStateDirty();
		}

		int index = data.history.indexOf(key);
		if (index != 0) {
			if (index > 0) {
				data.history.remove(key);
			}

			data.history.add(0, key);
			while (data.history.size() > GuiBookEntryList.ENTRIES_PER_PAGE) {
				data.history.remove(GuiBookEntryList.ENTRIES_PER_PAGE);
			}

			dirty = true;
		}

		if (dirty) {
			PersistentData.save();
		}
	}

	@Override
	void drawForegroundElements(class_332 graphics, int mouseX, int mouseY, float partialTicks) {
		drawPage(graphics, leftPage, mouseX, mouseY, partialTicks);
		drawPage(graphics, rightPage, mouseX, mouseY, partialTicks);

		if (rightPage == null) {
			drawPageFiller(graphics, entry.getBook());
		}
	}

	@Override
	public boolean mouseClickedScaled(double mouseX, double mouseY, int mouseButton) {
		return clickPage(leftPage, mouseX, mouseY, mouseButton)
				|| clickPage(rightPage, mouseX, mouseY, mouseButton)
				|| super.mouseClickedScaled(mouseX, mouseY, mouseButton);
	}

	void drawPage(class_332 graphics, @Nullable BookPage page, int mouseX, int mouseY, float pticks) {
		if (page == null) {
			return;
		}

		graphics.method_51448().method_22903();
		graphics.method_51448().method_46416(page.left, page.top, 0);
		page.render(graphics, mouseX - page.left, mouseY - page.top, pticks);
		graphics.method_51448().method_22909();
	}

	private boolean clickPage(@Nullable BookPage page, double mouseX, double mouseY, int mouseButton) {
		if (page != null) {
			return page.mouseClicked(mouseX - page.left, mouseY - page.top, mouseButton);
		}

		return false;
	}

	@Override
	void onPageChanged() {
		setupPages();
		needsBookmarkUpdate = true;
	}

	private void setupPages() {
		if (leftPage != null) {
			leftPage.onHidden(this);
		}
		if (rightPage != null) {
			rightPage.onHidden(this);
		}

		List<BookPage> pages = entry.getPages();
		int leftNum = spread * 2;
		int rightNum = (spread * 2) + 1;

		leftPage = leftNum < pages.size() ? pages.get(leftNum) : null;
		rightPage = rightNum < pages.size() ? pages.get(rightNum) : null;

		if (leftPage != null) {
			leftPage.onDisplayed(this, LEFT_PAGE_X, TOP_PADDING);
		}
		if (rightPage != null) {
			rightPage.onDisplayed(this, RIGHT_PAGE_X, TOP_PADDING);
		}
	}

	public BookEntry getEntry() {
		return entry;
	}

	@Override
	public boolean equals(Object obj) {
		return obj == this || (obj instanceof GuiBookEntry && ((GuiBookEntry) obj).entry == entry && ((GuiBookEntry) obj).spread == spread);
	}

	@Override
	public int hashCode() {
		return Objects.hashCode(entry) * 31 + Objects.hashCode(spread);
	}

	@Override
	public boolean canBeOpened() {
		return !entry.isLocked() && !equals(class_310.method_1551().field_1755);
	}

	@Override
	protected boolean shouldAddAddBookmarkButton() {
		return !isBookmarkedAlready();
	}

	boolean isBookmarkedAlready() {
		if (entry == null || entry.getId() == null) {
			return false;
		}

		String entryKey = entry.getId().toString();
		BookData data = PersistentData.data.getBookData(book);

		for (Bookmark bookmark : data.bookmarks) {
			if (bookmark.entry.equals(entryKey) && bookmark.spread == spread) {
				return true;
			}
		}

		return false;
	}

	@Override
	public void bookmarkThis() {
		var entryKey = entry.getId();
		BookData data = PersistentData.data.getBookData(book);
		data.bookmarks.add(new Bookmark(entryKey, spread));
		PersistentData.save();
		needsBookmarkUpdate = true;
	}

	public static void displayOrBookmark(GuiBook currGui, BookEntry entry) {
		Book book = currGui.book;
		GuiBookEntry gui = new GuiBookEntry(currGui.book, entry);

		if (class_437.method_25442()) {
			BookData data = PersistentData.data.getBookData(book);

			if (gui.isBookmarkedAlready()) {
				String key = entry.getId().toString();
				data.bookmarks.removeIf((bm) -> bm.entry.equals(key) && bm.spread == 0);
				PersistentData.save();
				currGui.needsBookmarkUpdate = true;
				return;
			} else if (data.bookmarks.size() < MAX_BOOKMARKS) {
				gui.bookmarkThis();
				currGui.needsBookmarkUpdate = true;
				return;
			}
		}

		book.getContents().openLexiconGui(gui, true);
	}

	@Override
	public class_437 getGui() {
		return this;
	}

	@Override
	public class_2583 getFont() {
		return book.getFontStyle();
	}

	@Override
	public void renderItemStack(class_332 graphics, int x, int y, int mouseX, int mouseY, class_1799 stack) {
		if (stack.method_7960()) {
			return;
		}

		graphics.method_51427(stack, x, y);
		graphics.method_51431(field_22793, stack, x, y);

		if (isMouseInRelativeRange(mouseX, mouseY, x, y, 16, 16)) {
			setTooltipStack(stack);
		}
	}

	@Override
	public void renderIngredient(class_332 graphics, int x, int y, int mouseX, int mouseY, class_1856 ingr) {
		class_1799[] stacks = ingr.method_8105();
		if (stacks.length > 0) {
			renderItemStack(graphics, x, y, mouseX, mouseY, stacks[(ticksInBook / 20) % stacks.length]);
		}
	}

	@Override
	public void setHoverTooltip(List<String> tooltip) {
		setTooltip(tooltip.stream().map(class_2561::method_43470).collect(Collectors.toList()));
	}

	@Override
	public void setHoverTooltipComponents(@NotNull List<class_2561> tooltip) {
		setTooltip(tooltip);
	}

	@Override
	public boolean isAreaHovered(int mouseX, int mouseY, int x, int y, int w, int h) {
		return isMouseInRelativeRange(mouseX, mouseY, x, y, w, h);
	}

	@Override
	public boolean navigateToEntry(class_2960 entry, int page, boolean push) {
		BookEntry bookEntry = book.getContents().entries.get(entry);
		if (bookEntry != null && !bookEntry.isLocked()) {
			displayLexiconGui(new GuiBookEntry(book, bookEntry, page), push);
			return true;
		}
		return false;
	}

	@SuppressWarnings("removal")
	@Override
	public void registerButton(class_4185 button, int pageNum, Runnable onClick) {
		addWidget(button, pageNum);
	}

	@Override
	public void addWidget(class_339 widget, int pageNum) {
		widget.method_46421(widget.method_46426() + (bookLeft + ((pageNum % 2) == 0 ? LEFT_PAGE_X : RIGHT_PAGE_X)));
		widget.method_46419(widget.method_46427() + bookTop);
		method_37063(widget);
	}

	@Override
	public boolean method_25404(int keyCode, int scanCode, int modifiers) {
		if (class_310.method_1551().field_1690.field_1822.method_1417(keyCode, scanCode)) {
			this.method_25419();
			return true;
		}
		return super.method_25404(keyCode, scanCode, modifiers);
	}

	@Override
	protected boolean shouldAddMarkReadButton() {
		return false;
	}

	@Override
	public class_2960 getBookTexture() {
		return book.bookTexture;
	}

	@Override
	public class_2960 getCraftingTexture() {
		return book.craftingTexture;
	}

	@Override
	public int getTextColor() {
		return book.textColor;
	}

	@Override
	public int getHeaderColor() {
		return book.headerColor;
	}

	@Override
	public int getTicksInBook() {
		return ticksInBook;
	}
}
