package vazkii.patchouli.client.base;

import com.google.common.base.Charsets;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import vazkii.patchouli.api.PatchouliAPI;
import vazkii.patchouli.client.book.BookEntry;
import vazkii.patchouli.common.book.Book;
import vazkii.patchouli.common.util.SerializationUtil;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import net.minecraft.class_2960;
import net.minecraft.class_3518;

public final class PersistentData {

	private static final Path saveFile = Paths.get("patchouli_data.json");

	public static DataHolder data = new DataHolder(new JsonObject());

	public static void setup() {
		try (var r = Files.newBufferedReader(saveFile, Charsets.UTF_8)) {
			var root = SerializationUtil.RAW_GSON.fromJson(r, JsonObject.class);
			data = new DataHolder(root);
		} catch (IOException e) {
			if (!(e instanceof NoSuchFileException)) {
				PatchouliAPI.LOGGER.warn("Unable to load patchouli_data.json, replacing with default", e);
			}
			data = new DataHolder(new JsonObject());
			save();
		} catch (Exception e) {
			PatchouliAPI.LOGGER.warn("Corrupted patchouli_data.json, replacing with default", e);
			data = new DataHolder(new JsonObject());
			save();
		}
	}

	public static void save() {
		var json = data.serialize();
		try (var w = Files.newBufferedWriter(saveFile, Charsets.UTF_8)) {
			SerializationUtil.PRETTY_GSON.toJson(json, w);
		} catch (IOException e) {
			PatchouliAPI.LOGGER.warn("Unable to save patchouli_data.json", e);
		}
	}

	public static final class DataHolder {
		public int bookGuiScale;
		public boolean clickedVisualize;

		private final Map<class_2960, PersistentData.BookData> bookData = new HashMap<>();

		public DataHolder(JsonObject root) {
			this.bookGuiScale = class_3518.method_15282(root, "bookGuiScale", 0);
			this.clickedVisualize = class_3518.method_15258(root, "clickedVisualize", false);
			var obj = class_3518.method_15281(root, "bookData", new JsonObject());

			for (var e : obj.entrySet()) {
				this.bookData.put(new class_2960(e.getKey()), new BookData(e.getValue().getAsJsonObject()));
			}
		}

		public PersistentData.BookData getBookData(Book book) {
			return bookData.computeIfAbsent(book.id, k -> new BookData(new JsonObject()));
		}

		public JsonObject serialize() {
			var ret = new JsonObject();
			ret.addProperty("bookGuiScale", this.bookGuiScale);
			ret.addProperty("clickedVisualize", this.clickedVisualize);

			var books = new JsonObject();
			for (var e : bookData.entrySet()) {
				books.add(e.getKey().toString(), e.getValue().serialize());
			}
			ret.add("bookData", books);
			return ret;
		}
	}

	public static final class Bookmark {
		public final class_2960 entry;
		public final int spread;

		public Bookmark(class_2960 entry, int spread) {
			this.entry = entry;
			this.spread = spread;
		}

		public Bookmark(JsonObject root) {
			this.entry = new class_2960(class_3518.method_15265(root, "entry"));
			this.spread = class_3518.method_15260(root, "page"); // Serialized as page for legacy reasons
		}

		public BookEntry getEntry(Book book) {
			return book.getContents().entries.get(entry);
		}

		public JsonObject serialize() {
			var ret = new JsonObject();
			ret.addProperty("entry", this.entry.toString());
			ret.addProperty("page", this.spread); // Serialized as page for legacy reasons
			return ret;
		}
	}

	public static final class BookData {
		public final List<class_2960> viewedEntries = new ArrayList<>();
		public final List<Bookmark> bookmarks = new ArrayList<>();
		public final List<class_2960> history = new ArrayList<>();
		public final List<class_2960> completedManualQuests = new ArrayList<>();

		public BookData(JsonObject root) {
			var emptyArray = new JsonArray();
			for (var e : class_3518.method_15292(root, "viewedEntries", emptyArray)) {
				viewedEntries.add(new class_2960(e.getAsString()));
			}
			for (var e : class_3518.method_15292(root, "bookmarks", emptyArray)) {
				bookmarks.add(new Bookmark(e.getAsJsonObject()));
			}
			for (var e : class_3518.method_15292(root, "history", emptyArray)) {
				history.add(new class_2960(e.getAsString()));
			}
			for (var e : class_3518.method_15292(root, "completedManualQuests", emptyArray)) {
				completedManualQuests.add(new class_2960(e.getAsString()));
			}
		}

		public JsonObject serialize() {
			var ret = new JsonObject();
			var viewed = new JsonArray();
			this.viewedEntries.stream().map(Object::toString).forEach(viewed::add);
			ret.add("viewedEntries", viewed);

			var bookmarks = new JsonArray();
			this.bookmarks.stream().map(Bookmark::serialize).forEach(bookmarks::add);
			ret.add("bookmarks", bookmarks);

			var completed = new JsonArray();
			this.completedManualQuests.stream().map(Object::toString).forEach(completed::add);
			ret.add("completedManualQuests", completed);

			return ret;
		}
	}
}
