/**
 * This class was created by <Vazkii>. It's distributed as
 * part of the Botania Mod. Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 *
 * File Created @ [Jun 9, 2014, 8:51:55 PM (GMT)]
 */
package vazkii.botania.common.block.tile;

import net.minecraft.block.BlockState;
import net.minecraft.entity.item.ItemEntity;
import net.minecraft.item.ItemStack;
import net.minecraft.item.Items;
import net.minecraft.nbt.CompoundNBT;
import net.minecraft.tileentity.ITickableTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.tileentity.TileEntityType;
import net.minecraft.util.LazyLoadBase;
import net.minecraft.util.Rotation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.Explosion;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.registries.ObjectHolder;
import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.recipe.ElvenPortalUpdateEvent;
import vazkii.botania.api.recipe.IElvenItem;
import vazkii.botania.api.recipe.RecipeElvenTrade;
import vazkii.botania.api.state.BotaniaStateProps;
import vazkii.botania.api.state.enums.AlfPortalState;
import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.common.block.ModBlocks;
import vazkii.botania.common.block.mana.BlockPool;
import vazkii.botania.common.block.tile.mana.TilePool;
import vazkii.botania.common.core.handler.ConfigHandler;
import vazkii.botania.common.item.ItemLexicon;
import vazkii.botania.common.lib.LibBlockNames;
import vazkii.botania.common.lib.LibMisc;
import vazkii.patchouli.api.IMultiblock;
import vazkii.patchouli.api.PatchouliAPI;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class TileAlfPortal extends TileMod implements ITickableTileEntity {

	@ObjectHolder(LibMisc.MOD_ID + ":" + LibBlockNames.ALF_PORTAL)
	public static TileEntityType<TileAlfPortal> TYPE;

	public static final LazyLoadBase<IMultiblock> MULTIBLOCK = new LazyLoadBase<>(() -> PatchouliAPI.instance.makeMultiblock(
			new String[][] {
					{ "_", "W", "G", "W", "_" },
					{ "W", " ", " ", " ", "W" },
					{ "G", " ", " ", " ", "G" },
					{ "W", " ", " ", " ", "W" },
					{ "_", "W", "0", "W", "_" }
			},
			'W', ModBlocks.livingwood,
			'G', ModBlocks.livingwoodGlimmering,
			'0', ModBlocks.alfPortal
	));

	private static final String TAG_TICKS_OPEN = "ticksOpen";
	private static final String TAG_TICKS_SINCE_LAST_ITEM = "ticksSinceLastItem";
	private static final String TAG_STACK_COUNT = "stackCount";
	private static final String TAG_STACK = "portalStack";
	private static final String TAG_PORTAL_FLAG = "_elvenPortal";

	private final List<ItemStack> stacksIn = new ArrayList<>();

	public int ticksOpen = 0;
	private int ticksSinceLastItem = 0;
	private boolean closeNow = false;
	private boolean explode = false;

	public TileAlfPortal() {
		super(TYPE);
	}

	@Override
	public void tick() {
		BlockState iBlockState = world.getBlockState(getPos());
		if(iBlockState.get(BotaniaStateProps.ALFPORTAL_STATE) == AlfPortalState.OFF) {
			ticksOpen = 0;
			return;
		}
		AlfPortalState state = iBlockState.get(BotaniaStateProps.ALFPORTAL_STATE);
		AlfPortalState newState = getValidState();

		ticksOpen++;

		AxisAlignedBB aabb = getPortalAABB();
		boolean open = ticksOpen > 60;
		ElvenPortalUpdateEvent event = new ElvenPortalUpdateEvent(this, aabb, open, stacksIn);
		MinecraftForge.EVENT_BUS.post(event);

		if(ticksOpen > 60) {
			ticksSinceLastItem++;
			if(world.isRemote && ConfigHandler.CLIENT.elfPortalParticlesEnabled.get())
				blockParticle(state);

			List<ItemEntity> items = world.getEntitiesWithinAABB(ItemEntity.class, aabb);
			if(!world.isRemote)
				for(ItemEntity item : items) {
					if(!item.isAlive())
						continue;

					ItemStack stack = item.getItem();
					boolean consume;
					if (item.getPersistentData().contains(TAG_PORTAL_FLAG)) {
						consume = false;
					} else if (stack.getItem() instanceof ItemLexicon) {
						consume = true;
					} else if ((!(stack.getItem() instanceof IElvenItem) || !((IElvenItem) stack.getItem()).isElvenItem(stack))) {
						consume = true;
					} else {
						consume = false;
					}

					if (consume) {
						item.remove();
						if (validateItemUsage(stack))
							addItem(stack);
						ticksSinceLastItem = 0;
					}
				}

			if(!world.isRemote && !stacksIn.isEmpty() && ticksSinceLastItem >= 4) {
				resolveRecipes();
			}
		}

		if(closeNow) {
			world.setBlockState(getPos(), ModBlocks.alfPortal.getDefaultState(), 1 | 2);
			for(int i = 0; i < 36; i++)
				blockParticle(state);
			closeNow = false;
		} else if(newState != state) {
			if(newState == AlfPortalState.OFF)
				for(int i = 0; i < 36; i++)
					blockParticle(state);
			world.setBlockState(getPos(), world.getBlockState(getPos()).with(BotaniaStateProps.ALFPORTAL_STATE, newState), 1 | 2);
		} else if(explode) {
			world.createExplosion(null, pos.getX() + .5, pos.getY() + 2.0, pos.getZ() + .5,
					3f, Explosion.Mode.DESTROY);
			explode = false;
		}
	}

	private boolean validateItemUsage(ItemStack inputStack) {
		for(RecipeElvenTrade recipe : BotaniaAPI.elvenTradeRecipes.values()) {
			if(recipe.containsItem(inputStack)) {
				return true;
			}
		}
		if(inputStack.getItem() == Items.BREAD) //Don't teleport bread. (See also: #2403)
			explode = true;

		return false;
	}

	private void blockParticle(AlfPortalState state) {
		double dh, dy;

		// Pick one of the inner positions
		switch (world.rand.nextInt(9)) {
			default:
			case 0: dh = 0; dy = 1; break;
			case 1: dh = 0; dy = 2; break;
			case 2: dh = 0; dy = 3; break;
			case 3: dh = -1; dy = 1; break;
			case 4: dh = -1; dy = 2; break;
			case 5: dh = -1; dy = 3; break;
			case 6: dh = 1; dy = 1; break;
			case 7: dh = 1; dy = 2; break;
			case 8: dh = 1; dy = 3; break;
		}
		double dx = state == AlfPortalState.ON_X ? 0 : dh;
		double dz = state == AlfPortalState.ON_Z ? 0 : dh;

		float motionMul = 0.2F;
		WispParticleData data = WispParticleData.wisp((float) (Math.random() * 0.15F + 0.1F), (float) (Math.random() * 0.25F), (float) (Math.random() * 0.5F + 0.5F), (float) (Math.random() * 0.25F));
		world.addParticle(data, getPos().getX() + dx, getPos().getY() + dy, getPos().getZ() + dz, (float) (Math.random() - 0.5F) * motionMul, (float) (Math.random() - 0.5F) * motionMul, (float) (Math.random() - 0.5F) * motionMul);
	}

	public boolean onWanded() {
		AlfPortalState state = world.getBlockState(getPos()).get(BotaniaStateProps.ALFPORTAL_STATE);
		if(state == AlfPortalState.OFF) {
			AlfPortalState newState = getValidState();
			if(newState != AlfPortalState.OFF) {
				world.setBlockState(getPos(), world.getBlockState(getPos()).with(BotaniaStateProps.ALFPORTAL_STATE, newState), 1 | 2);
				return true;
			}
		}

		return false;
	}

	private AxisAlignedBB getPortalAABB() {
		AxisAlignedBB aabb = new AxisAlignedBB(pos.add(-1, 1, 0), pos.add(2, 4, 1));
		if(world.getBlockState(getPos()).get(BotaniaStateProps.ALFPORTAL_STATE) == AlfPortalState.ON_X)
			aabb = new AxisAlignedBB(pos.add(0, 1, -1), pos.add(1, 4, 2));

		return aabb;
	}

	private void addItem(ItemStack stack) {
		int size = stack.getCount();
		stack.setCount(1);
		for(int i = 0; i < size; i++)
			stacksIn.add(stack.copy());
	}

	private void resolveRecipes() {
		List<BlockPos> pylons = locatePylons();
		for(RecipeElvenTrade recipe : BotaniaAPI.elvenTradeRecipes.values()) {
			Optional<List<ItemStack>> match = recipe.match(stacksIn);
			if(match.isPresent()) {
				if(consumeMana(pylons, 500, false)) {
					List<ItemStack> inputs = match.get();
					for(ItemStack stack : inputs) {
						stacksIn.remove(stack);
					}
					for(ItemStack output : recipe.getOutputs(inputs))
						spawnItem(output.copy());
				}
				break;
			}
		}
	}

	private void spawnItem(ItemStack stack) {
		ItemEntity item = new ItemEntity(world, pos.getX() + 0.5, pos.getY() + 1.5, pos.getZ() + 0.5, stack);
		item.getPersistentData().putBoolean(TAG_PORTAL_FLAG, true);
		world.addEntity(item);
		ticksSinceLastItem = 0;
	}

	@Nonnull
	@Override
	public CompoundNBT write(CompoundNBT cmp) {
		CompoundNBT ret = super.write(cmp);

		cmp.putInt(TAG_STACK_COUNT, stacksIn.size());
		int i = 0;
		for(ItemStack stack : stacksIn) {
			CompoundNBT stackcmp = stack.write(new CompoundNBT());
			cmp.put(TAG_STACK + i, stackcmp);
			i++;
		}

		return ret;
	}

	@Override
	public void read(CompoundNBT cmp) {
		super.read(cmp);

		int count = cmp.getInt(TAG_STACK_COUNT);
		stacksIn.clear();
		for(int i = 0; i < count; i++) {
			CompoundNBT stackcmp = cmp.getCompound(TAG_STACK + i);
			ItemStack stack = ItemStack.read(stackcmp);
			stacksIn.add(stack);
		}
	}

	@Override
	public void writePacketNBT(CompoundNBT cmp) {
		cmp.putInt(TAG_TICKS_OPEN, ticksOpen);
		cmp.putInt(TAG_TICKS_SINCE_LAST_ITEM, ticksSinceLastItem);
	}

	@Override
	public void readPacketNBT(CompoundNBT cmp) {
		ticksOpen = cmp.getInt(TAG_TICKS_OPEN);
		ticksSinceLastItem = cmp.getInt(TAG_TICKS_SINCE_LAST_ITEM);
	}

	private AlfPortalState getValidState() {
		Rotation rot = MULTIBLOCK.getValue().validate(world, getPos());
		if (rot == null)
			return AlfPortalState.OFF;

		lightPylons();
		switch (rot) {
			default:
			case NONE:
			case CLOCKWISE_180: return AlfPortalState.ON_Z;
			case CLOCKWISE_90:
			case COUNTERCLOCKWISE_90: return AlfPortalState.ON_X;
		}
	}

	public List<BlockPos> locatePylons() {
		List<BlockPos> list = new ArrayList<>();
		int range = 5;

		BlockState pylonState = ModBlocks.naturaPylon.getDefaultState();

		for(int i = -range; i < range + 1; i++)
			for(int j = -range; j < range + 1; j++)
				for(int k = -range; k < range + 1; k++) {
					BlockPos pos = getPos().add(i, j, k);
					if(world.getBlockState(pos) == pylonState && world.getBlockState(pos.down()).getBlock() instanceof BlockPool)
						list.add(pos);
				}

		return list;
	}

	public void lightPylons() {
		if(ticksOpen < 50)
			return;

		List<BlockPos> pylons = locatePylons();
		for(BlockPos pos : pylons) {
			TileEntity tile = world.getTileEntity(pos);
			if(tile instanceof TilePylon) {
				TilePylon pylon = (TilePylon) tile;
				pylon.activated = true;
				pylon.centerPos = getPos();
			}
		}

		if(ticksOpen == 50)
			consumeMana(pylons, 200000, true);
	}

	public boolean consumeMana(List<BlockPos> pylons, int totalCost, boolean close) {
		List<TilePool> consumePools = new ArrayList<>();
		int consumed = 0;

		if(pylons.size() < 2) {
			closeNow = true;
			return false;
		}

		int costPer = Math.max(1, totalCost / pylons.size());
		int expectedConsumption = costPer * pylons.size();

		for(BlockPos pos : pylons) {
			TileEntity tile = world.getTileEntity(pos);
			if(tile instanceof TilePylon) {
				TilePylon pylon = (TilePylon) tile;
				pylon.activated = true;
				pylon.centerPos = getPos();
			}

			tile = world.getTileEntity(pos.down());
			if(tile instanceof TilePool) {
				TilePool pool = (TilePool) tile;

				if(pool.getCurrentMana() < costPer) {
					closeNow = closeNow || close;
					return false;
				} else if(!world.isRemote) {
					consumePools.add(pool);
					consumed += costPer;
				}
			}
		}

		if(consumed >= expectedConsumption) {
			for(TilePool pool : consumePools)
				pool.recieveMana(-costPer);
			return true;
		}

		return false;
	}

	@Nonnull
	@Override
	public AxisAlignedBB getRenderBoundingBox() {
		return INFINITE_EXTENT_AABB;
	}
}
