/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.flower.generating;

import vazkii.botania.api.block_entity.GeneratingFlowerBlockEntity;
import vazkii.botania.api.block_entity.RadiusDescriptor;
import vazkii.botania.common.block.BotaniaFlowerBlocks;
import vazkii.botania.common.helper.DelayHelper;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import net.minecraft.class_1542;
import net.minecraft.class_1799;
import net.minecraft.class_2338;
import net.minecraft.class_238;
import net.minecraft.class_2392;
import net.minecraft.class_2398;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_4174;
import net.minecraft.class_5712;

public class GourmaryllisBlockEntity extends GeneratingFlowerBlockEntity {
	private static final String TAG_COOLDOWN = "cooldown";
	private static final String TAG_DIGESTING_MANA = "digestingMana";
	public static final String TAG_LAST_FOODS = "lastFoods";
	public static final String TAG_LAST_FOOD_COUNT = "lastFoodCount";
	public static final String TAG_STREAK_LENGTH = "streakLength";
	private static final int RANGE = 1;
	private static final double[] STREAK_MULTIPLIERS = { 0, 1, 1.3, 1.5, 1.6, 1.7, 1.75, 1.8 };
	private static final int MAX_FOOD_VALUE = 12;
	private static final int FOOD_COOLDOWN_FACTOR = 10;
	private static final int FOOD_MANA_FACTOR = 70;
	private static final int MAX_MANA = getDigestingMana(MAX_FOOD_VALUE, STREAK_MULTIPLIERS[STREAK_MULTIPLIERS.length - 1]);

	private int cooldown = 0;
	private int digestingMana = 0;
	private final List<class_1799> lastFoods = new LinkedList<>();
	private int streakLength = -1;
	private int lastFoodCount = 0;

	public GourmaryllisBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaFlowerBlocks.GOURMARYLLIS, pos, state);
	}

	private int getMaxStreak() {
		return STREAK_MULTIPLIERS.length - 1;
	}

	private double getMultiplierForStreak(int index) {
		// special-case repeated first foods
		if (index == 0) {
			return 1.0 / ++lastFoodCount;
		} else {
			lastFoodCount = 1;
			return STREAK_MULTIPLIERS[index];
		}
	}

	/**
	 * Processes a food, placing it in the appropriate place in the history.
	 * 
	 * @return the last time the food showed up in history.
	 */
	private int processFood(class_1799 food) {
		for (ListIterator<class_1799> it = lastFoods.listIterator(); it.hasNext();) {
			int index = it.nextIndex();
			class_1799 streakFood = it.next();
			if (class_1799.method_31577(streakFood, food)) {
				it.remove();
				lastFoods.add(0, streakFood);
				return index;
			}
		}
		class_1799 newestFood = food.method_46651(1);
		lastFoods.add(0, newestFood);
		if (lastFoods.size() >= getMaxStreak()) {
			lastFoods.remove(lastFoods.size() - 1);
		}
		return getMaxStreak();
	}

	@Override
	public void tickFlower() {
		super.tickFlower();

		if (method_10997().field_9236) {
			return;
		}

		if (cooldown > -1) {
			cooldown--;
		}
		if (digestingMana != 0) {
			int munchInterval = 2 + (2 * lastFoodCount);

			if (cooldown == 0) {
				addMana(digestingMana);
				digestingMana = 0;

				float burpPitch = (float) Math.pow(2.0, (streakLength == 0 ? -lastFoodCount : streakLength) / 12.0);
				//Usage of vanilla sound event: Subtitle is just "Burp", at least in English, and not specific to players.
				method_10997().method_8396(null, getEffectivePos(), class_3417.field_19149, class_3419.field_15245, 1F, burpPitch);
				method_10997().method_33596(null, class_5712.field_28175, getEffectivePos());
				sync();
			} else if (cooldown % munchInterval == 0) {
				//Usage of vanilla sound event: Subtitle is "Eating", generic sounds are meant to be reused.
				method_10997().method_8396(null, getEffectivePos(), class_3417.field_20614, class_3419.field_15245, 0.5F, 1F);

				class_243 offset = method_10997().method_8320(getEffectivePos()).method_26226(method_10997(), getEffectivePos()).method_1031(0.4, 0.6, 0.4);

				((class_3218) method_10997()).method_14199(new class_2392(class_2398.field_11218, lastFoods.get(0)), getEffectivePos().method_10263() + offset.field_1352, getEffectivePos().method_10264() + offset.field_1351, getEffectivePos().method_10260() + offset.field_1350, 10, 0.1D, 0.1D, 0.1D, 0.03D);
			}
		}

		List<class_1542> items = method_10997().method_18467(class_1542.class, new class_238(getEffectivePos().method_10069(-RANGE, -RANGE, -RANGE), getEffectivePos().method_10069(RANGE + 1, RANGE + 1, RANGE + 1)));

		for (class_1542 item : items) {
			class_1799 stack = item.method_6983();

			if (DelayHelper.canInteractWithImmediate(this, item) && stack.method_7909().method_19263()) {
				if (cooldown <= 0) {
					streakLength = Math.min(streakLength + 1, processFood(stack));

					int val = getFoodValue(stack);
					digestingMana = getDigestingMana(val, getMultiplierForStreak(streakLength));
					cooldown = getCooldown(val);
					//Usage of vanilla sound event: Subtitle is "Eating", generic sounds are meant to be reused.
					item.method_5783(class_3417.field_20614, 0.2F, 0.6F);
					method_10997().method_43275(null, class_5712.field_28735, item.method_19538());
					method_10997().method_33596(null, class_5712.field_28174, getEffectivePos());
					sync();
					((class_3218) method_10997()).method_14199(new class_2392(class_2398.field_11218, stack), item.method_23317(), item.method_23318(), item.method_23321(), 20, 0.1D, 0.1D, 0.1D, 0.05D);
				}

				item.method_31472();
			}
		}
	}

	private static int getCooldown(int foodValue) {
		return Math.max(1, foodValue * FOOD_COOLDOWN_FACTOR);
	}

	private static int getDigestingMana(int foodValue, double streakFactor) {
		return Math.max(1, (int) (foodValue * foodValue * FOOD_MANA_FACTOR * streakFactor));
	}

	private static int getFoodValue(class_1799 stack) {
		// support for Forge's NBT-based food properties
		class_4174 foodProperties = XplatAbstractions.INSTANCE.getFoodProperties(stack);
		int nutrition = foodProperties != null ? foodProperties.method_19230() : 0;
		return Math.min(MAX_FOOD_VALUE, nutrition);
	}

	@Override
	public void writeToPacketNBT(class_2487 cmp) {
		super.writeToPacketNBT(cmp);
		cmp.method_10569(TAG_COOLDOWN, cooldown);
		cmp.method_10569(TAG_DIGESTING_MANA, digestingMana);
		class_2499 foodList = new class_2499();
		for (class_1799 food : lastFoods) {
			foodList.add(food.method_7953(new class_2487()));
		}
		cmp.method_10566(TAG_LAST_FOODS, foodList);
		cmp.method_10569(TAG_LAST_FOOD_COUNT, lastFoodCount);
		cmp.method_10569(TAG_STREAK_LENGTH, streakLength);
	}

	@Override
	public void readFromPacketNBT(class_2487 cmp) {
		super.readFromPacketNBT(cmp);
		cooldown = cmp.method_10550(TAG_COOLDOWN);
		digestingMana = cmp.method_10550(TAG_DIGESTING_MANA);
		lastFoods.clear();
		class_2499 foodList = cmp.method_10554(TAG_LAST_FOODS, class_2520.field_33260);
		for (int i = 0; i < foodList.size(); i++) {
			lastFoods.add(class_1799.method_7915(foodList.method_10602(i)));
		}
		lastFoodCount = cmp.method_10550(TAG_LAST_FOOD_COUNT);
		streakLength = cmp.method_10550(TAG_STREAK_LENGTH);
	}

	@Override
	public RadiusDescriptor getRadius() {
		return RadiusDescriptor.Rectangle.square(getEffectivePos(), RANGE);
	}

	@Override
	public int getMaxMana() {
		return MAX_MANA;
	}

	@Override
	public int getColor() {
		return 0xD3D604;
	}

}
