/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.impl.corporea;

import com.google.common.base.Predicates;

import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntMaps;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.corporea.*;
import vazkii.botania.common.block.block_entity.corporea.CorporeaRetainerBlockEntity;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.*;
import java.util.function.Function;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_238;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2960;
import net.minecraft.class_3532;

public class CorporeaHelperImpl implements CorporeaHelper {
	private final WeakHashMap<CorporeaSpark, Set<CorporeaNode>> cachedNetworks = new WeakHashMap<>();

	@Override
	public Set<CorporeaNode> getNodesOnNetwork(CorporeaSpark spark) {
		CorporeaSpark master = spark.getMaster();
		if (master == null) {
			return Collections.emptySet();
		}
		Set<CorporeaSpark> network = master.getConnections();

		var cache = cachedNetworks.get(master);
		if (cache != null) {
			return cache;
		}

		Set<CorporeaNode> nodes = new LinkedHashSet<>();
		if (network != null) {
			for (CorporeaSpark otherSpark : network) {
				if (otherSpark != null) {
					nodes.add(otherSpark.getSparkNode());
				}
			}
		}

		cachedNetworks.put(master, nodes);
		return nodes;
	}

	@Override
	public CorporeaRequestMatcher createMatcher(class_1799 stack, boolean checkNBT) {
		return new CorporeaItemStackMatcher(stack, checkNBT);
	}

	@Override
	public CorporeaRequestMatcher createMatcher(String name) {
		return new CorporeaStringMatcher(name);
	}

	@Override
	public CorporeaResult requestItem(CorporeaRequestMatcher matcher, int itemCount, CorporeaSpark spark, @Nullable class_1309 entity, boolean doit) {
		List<class_1799> stacks = new ArrayList<>();
		if (XplatAbstractions.INSTANCE.fireCorporeaRequestEvent(matcher, itemCount, spark, !doit)) {
			return new CorporeaResultImpl(stacks, 0, 0, Object2IntMaps.emptyMap());
		}

		Object2IntMap<CorporeaNode> matchCountByNode = new Object2IntOpenHashMap<>();
		Set<CorporeaNode> nodes = getNodesOnNetwork(spark);
		Map<CorporeaInterceptor, CorporeaSpark> interceptors = new HashMap<>();

		CorporeaRequest request = new CorporeaRequestImpl(matcher, itemCount, entity);
		for (CorporeaNode node : nodes) {
			CorporeaSpark invSpark = node.getSpark();

			class_2586 te = node.getWorld().method_8321(node.getPos());
			if (te instanceof CorporeaInterceptor interceptor) {
				interceptor.interceptRequest(matcher, itemCount, invSpark, spark, stacks, nodes, doit);
				interceptors.put(interceptor, invSpark);
			}

			var nodeStacks = doit ? node.extractItems(request) : node.countItems(request);
			int sum = 0;
			for (var stack : nodeStacks) {
				sum += stack.method_7947();
			}
			matchCountByNode.mergeInt(node, sum, Integer::sum);
			stacks.addAll(nodeStacks);
		}

		for (CorporeaInterceptor interceptor : interceptors.keySet()) {
			interceptor.interceptRequestLast(matcher, itemCount, interceptors.get(interceptor), spark, stacks, nodes, doit);
		}

		return new CorporeaResultImpl(stacks, request.getFound(), request.getExtracted(), matchCountByNode);
	}

	@Override
	public CorporeaSpark getSparkForBlock(class_1937 world, class_2338 pos) {
		List<class_1297> sparks = world.method_8390(class_1297.class, new class_238(pos.method_10084(), pos.method_10069(1, 2, 1)), Predicates.instanceOf(CorporeaSpark.class));
		return sparks.isEmpty() ? null : (CorporeaSpark) sparks.get(0);
	}

	@Override
	public int signalStrengthForRequestSize(int requestSize) {
		if (requestSize <= 0) {
			return 0;
		} else if (requestSize >= 16384) {
			return 15;
		} else {
			return Math.min(15, class_3532.method_15351(requestSize) + 1);
		}
	}

	@Override
	public <T extends CorporeaRequestMatcher> void registerRequestMatcher(class_2960 id, Class<T> clazz, Function<class_2487, T> deserializer) {
		CorporeaRetainerBlockEntity.addCorporeaRequestMatcher(id, clazz, deserializer);
	}

	public void clearCache() {
		cachedNetworks.clear();
	}
}
