/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.crafting.recipe;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mojang.authlib.GameProfile;
import com.mojang.authlib.minecraft.MinecraftProfileTexture;
import com.mojang.authlib.properties.Property;
import com.mojang.authlib.yggdrasil.response.MinecraftTexturesPayload;
import com.mojang.util.UUIDTypeAdapter;
import org.jetbrains.annotations.NotNull;

import vazkii.botania.common.crafting.BotaniaRecipeTypes;
import vazkii.botania.common.crafting.RunicAltarRecipe;
import vazkii.botania.common.helper.ItemNBTHelper;

import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.minecraft.class_1263;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1843;
import net.minecraft.class_1856;
import net.minecraft.class_1865;
import net.minecraft.class_1869;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2512;
import net.minecraft.class_2520;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_5348;
import net.minecraft.class_5455;

public class HeadRecipe extends RunicAltarRecipe {
	private static final Pattern PROFILE_PATTERN = Pattern.compile(
			"(?<base64>[A-Za-z0-9+/]{100,}={0,2})" +
					"|(?<url>(?=\\S{50,})https?://(?!bugs|education|feedback)\\w+\\.(?:minecraft\\.net|mojang\\.com)/\\S+)" +
					"|(?<hash>[0-9a-f]{64})");
	public static final String TEXTURE_URL_BASE = "https://textures.minecraft.net/texture/";
	private static final Supplier<Gson> gson = Suppliers.memoize(() -> new GsonBuilder()
			.registerTypeAdapter(UUID.class, new UUIDTypeAdapter()).create());
	private static final GameProfile PROFILE_VALID_RESULT = new GameProfile(null, "valid");

	public HeadRecipe(class_2960 id, class_1799 output, int mana, class_1856... inputs) {
		super(id, output, mana, inputs);
	}

	@Override
	public boolean method_8115(class_1263 inv, @NotNull class_1937 world) {
		boolean matches = super.method_8115(inv, world);
		boolean foundName = false;

		if (matches) {
			for (int i = 0; i < inv.method_5439(); i++) {
				class_1799 stack = inv.method_5438(i);
				if (stack.method_7960()) {
					break;
				}

				// either exactly one name tag or exactly one written book among ingredients
				if (stack.method_31574(class_1802.field_8448)) {
					if (foundName || !stack.method_7938() || stack.method_7964().getString().isBlank()) {
						return false;
					}
					foundName = true;
				} else if (stack.method_31574(class_1802.field_8360)) {
					if (foundName || !class_1843.method_8053(stack.method_7969())
							|| parseProfileFromBook(stack, true) == null) {
						return false;
					}
					foundName = true;
				}
			}
		}

		return matches;
	}

	@NotNull
	@Override
	public class_1799 method_8116(@NotNull class_1263 inv, @NotNull class_5455 registries) {
		class_1799 stack = method_8110(registries).method_7972();
		for (int i = 0; i < inv.method_5439(); i++) {
			class_1799 ingr = inv.method_5438(i);
			if (ingr.method_31574(class_1802.field_8448)) {
				ItemNBTHelper.setString(stack, "SkullOwner", ingr.method_7964().getString());
				break;
			}
			if (ingr.method_31574(class_1802.field_8360)) {
				GameProfile profile = parseProfileFromBook(ingr, false);
				ItemNBTHelper.setCompound(stack, "SkullOwner", class_2512.method_10684(new class_2487(), profile));
				break;
			}
		}
		return stack;
	}

	private GameProfile parseProfileFromBook(class_1799 stack, boolean validateOnly) {
		// tag has been validated, so we know all relevant elements exist
		class_2487 tag = stack.method_7969();
		String name = tag.method_10558(class_1843.field_30935);
		if (name.isBlank()) {
			return null;
		}

		class_2499 pages = tag.method_10554(class_1843.field_30938, class_2520.field_33258);

		// no-nonsense check; at most the first two pages are scanned, and the check fails at the first error
		int maxPages = Math.min(2, pages.size());
		for (int i = 0; i < maxPages; ++i) {
			String pageJson = pages.method_10608(i);
			String pageText = parsePage(pageJson);

			Matcher matcher = PROFILE_PATTERN.matcher(pageText);
			if (matcher.matches()) {
				// this appears to be the page we were looking for, figure out the skin texture it encodes
				String textureUrl;
				String hash, base64, url;
				if ((hash = matcher.group("hash")) != null) {
					// simplest case: just the texture hash; complete the URL
					textureUrl = TEXTURE_URL_BASE + hash;
				} else if ((url = matcher.group("url")) != null) {
					// an entire URL was specified; make sure it looks valid
					try {
						// just basic URL validation so we don't potentially spam error logs
						URL validUrl = new URL(url);
						textureUrl = validUrl.toString();
					} catch (Exception e) {
						return null;
					}
				} else if ((base64 = matcher.group("base64")) != null) {
					// complete profile properties; do rudimentary parsing
					try {
						final String json = new String(Base64.getDecoder().decode(base64), StandardCharsets.UTF_8);
						MinecraftTexturesPayload result = gson.get().fromJson(json, MinecraftTexturesPayload.class);
						MinecraftProfileTexture skinTexture = result.getTextures().get(MinecraftProfileTexture.Type.SKIN);
						String skinTextureUrl = skinTexture.getUrl();
						if (!PROFILE_PATTERN.matcher(skinTextureUrl).matches()) {
							return null;
						}
						URL validUrl = new URL(skinTextureUrl);
						textureUrl = validUrl.toString();
					} catch (Exception e) {
						return null;
					}
				} else {
					return null;
				}
				if (validateOnly) {
					return PROFILE_VALID_RESULT;
				}
				// we got something that looks like a valid skin texture URL, now build rudimentary profile data
				String profileTextureJson = "{textures:{SKIN:{url:\"%s\"}}}".formatted(textureUrl);
				var profile = new GameProfile(null, name);
				String propertyBase64 = Base64.getEncoder().encodeToString(profileTextureJson.getBytes(StandardCharsets.UTF_8));
				profile.getProperties().put("textures", new Property("Value", propertyBase64));
				return profile;
			}
		}
		return null;
	}

	private static String parsePage(String pageJson) {
		try {
			class_5348 formattedtext = class_2561.class_2562.method_10877(pageJson);
			return formattedtext != null ? formattedtext.getString() : pageJson;
		} catch (Exception exception) {
			return pageJson;
		}
	}

	public static class Serializer implements class_1865<HeadRecipe> {

		@NotNull
		@Override
		public HeadRecipe method_8121(@NotNull class_2960 id, @NotNull JsonObject json) {
			class_1799 output = class_1869.method_35228(class_3518.method_15296(json, "output"));
			int mana = class_3518.method_15260(json, "mana");
			JsonArray ingrs = class_3518.method_15261(json, "ingredients");
			List<class_1856> inputs = new ArrayList<>();
			for (JsonElement e : ingrs) {
				inputs.add(class_1856.method_52177(e));
			}
			return new HeadRecipe(id, output, mana, inputs.toArray(new class_1856[0]));
		}

		@Override
		public HeadRecipe method_8122(@NotNull class_2960 id, @NotNull class_2540 buf) {
			class_1856[] inputs = new class_1856[buf.method_10816()];
			for (int i = 0; i < inputs.length; i++) {
				inputs[i] = class_1856.method_8086(buf);
			}
			class_1799 output = buf.method_10819();
			int mana = buf.method_10816();
			return new HeadRecipe(id, output, mana, inputs);
		}

		@Override
		public void toNetwork(@NotNull class_2540 buf, @NotNull HeadRecipe recipe) {
			BotaniaRecipeTypes.RUNE_SERIALIZER.method_8124(buf, recipe);
		}
	}

}
