/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.flower.functional;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.block.Wandable;
import vazkii.botania.api.block_entity.FunctionalFlowerBlockEntity;
import vazkii.botania.api.block_entity.RadiusDescriptor;
import vazkii.botania.common.block.BotaniaFlowerBlocks;
import vazkii.botania.common.helper.DelayHelper;
import vazkii.botania.common.helper.EntityHelper;
import vazkii.botania.mixin.AnimalAccessor;
import vazkii.botania.mixin.MushroomCowAccessor;

import java.util.*;
import java.util.function.Predicate;
import net.minecraft.class_1074;
import net.minecraft.class_1296;
import net.minecraft.class_1429;
import net.minecraft.class_1438;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_2487;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3417;
import net.minecraft.class_3489;
import net.minecraft.class_3542;
import net.minecraft.class_6024;
import net.minecraft.class_7917;

public class PollidisiacBlockEntity extends FunctionalFlowerBlockEntity implements Wandable {
	private static final String TAG_FEEDING_MODE = "mode";
	private static final int RANGE = 6;
	private static final int MANA_COST = 12;

	@NotNull
	private Mode mode = Mode.FEED_ADULTS;

	public PollidisiacBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaFlowerBlocks.POLLIDISIAC, pos, state);
	}

	@Override
	public void tickFlower() {
		super.tickFlower();

		if (!method_10997().field_9236 && getMana() >= MANA_COST) {
			List<class_1542> items = getItems();
			if (!items.isEmpty()) {
				List<class_1429> animals = getAnimals();
				feedAnimal(animals, items);
			}
		}
	}

	/**
	 * Finds items around flower's actual position.
	 */
	private @NotNull List<class_1542> getItems() {
		var pickupBounds = new class_238(method_11016()).method_1014(RANGE);
		return method_10997().method_8390(class_1542.class, pickupBounds,
				itemEntity -> DelayHelper.canInteractWith(this, itemEntity));
	}

	/**
	 * Finds animals around flower's effective position. Depending on mode, adults, babies, or both will be selected.
	 */
	private @NotNull List<class_1429> getAnimals() {
		var bounds = new class_238(getEffectivePos()).method_1014(RANGE);
		return method_10997().method_8390(class_1429.class, bounds, mode);
	}

	/**
	 * Attempts to feed an animal with an available item. Only one animal will be fed per call. Feeding adults is
	 * prioritized, but if babies get their turn, they are prioritized by their age, youngest first. Among brown adult
	 * mooshrooms, breeding is prioritized over feeding flowers for suspicious stew, if both item types are available.
	 */
	private void feedAnimal(List<class_1429> animals, List<class_1542> items) {
		// randomize animals with same age
		Collections.shuffle(animals);
		// feed adults first, then babies, youngest to oldest
		animals.sort(Comparator.comparing(class_1429::method_6109).thenComparingInt(animal -> Math.min(animal.method_5618(), 0)));

		for (class_1429 animal : animals) {
			// Note: Empty item stacks are implicitly excluded in Animal::isFood and ItemStack::is(TagKey)
			if (animal.method_5618() == 0 && !animal.method_6479() || animal.method_5618() < -600 && -animal.method_5618() % 100 == 0) {
				for (class_1542 item : items) {
					if (!animal.method_6481(item.method_6983())) {
						continue;
					}
					consumeFoodItemAndMana(item);

					if (animal.method_6109()) {
						animal.method_5620(class_1296.method_41321(-animal.method_5618()), true);
					} else {
						animal.method_6476(1200);
						((AnimalAccessor) animal).botania_setLoveCause(null);
					}
					method_10997().method_8421(animal, class_6024.field_30043);
					break;
				}

				if (getMana() < MANA_COST) {
					break;
				}
			}

			if (!animal.method_6109() && isBrownMooshroomWithoutEffect(animal)) {
				for (class_1542 item : items) {
					class_1799 stack = item.method_6983();
					if (!stack.method_31573(class_3489.field_15543)) {
						continue;
					}
					var effect = class_7917.method_47380(stack.method_7909());
					if (effect == null) {
						continue;
					}
					consumeFoodItemAndMana(item);

					MushroomCowAccessor cowAccessor = (MushroomCowAccessor) animal;
					cowAccessor.setEffect(effect.method_10188());
					cowAccessor.setEffectDuration(effect.method_10187());
					animal.method_5783(class_3417.field_18267, 2.0F, 1.0F);
					break;
				}

				if (getMana() < MANA_COST) {
					break;
				}
			}
		}
	}

	private void consumeFoodItemAndMana(class_1542 itemEntity) {
		EntityHelper.shrinkItem(itemEntity);
		addMana(-MANA_COST);
	}

	private static boolean isBrownMooshroomWithoutEffect(class_1429 animal) {
		if (animal instanceof class_1438 mushroomCow && mushroomCow.method_47847() == class_1438.class_4053.field_18110) {
			MushroomCowAccessor cowAccessor = (MushroomCowAccessor) animal;
			return cowAccessor.getEffect() == null;
		}
		return false;
	}

	@Override
	public RadiusDescriptor getRadius() {
		return RadiusDescriptor.Rectangle.square(getEffectivePos(), RANGE);
	}

	@Override
	public RadiusDescriptor getSecondaryRadius() {
		return method_11016().equals(getEffectivePos()) ? null : RadiusDescriptor.Rectangle.square(method_11016(), RANGE);
	}

	@Override
	public int getMaxMana() {
		return 120;
	}

	@Override
	public int getColor() {
		return 0xCF4919;
	}

	@NotNull
	public Mode getMode() {
		return this.mode;
	}

	@Override
	public boolean onUsedByWand(@Nullable class_1657 player, class_1799 stack, class_2350 side) {
		if (player == null || player.method_5715()) {
			this.mode = this.mode.getNextMode();
			method_5431();
			sync();

			return true;
		}
		return false;
	}

	@Override
	public void readFromPacketNBT(class_2487 cmp) {
		super.readFromPacketNBT(cmp);
		this.mode = Mode.forName(cmp.method_10558(TAG_FEEDING_MODE));
	}

	@Override
	public void writeToPacketNBT(class_2487 cmp) {
		super.writeToPacketNBT(cmp);
		cmp.method_10582(TAG_FEEDING_MODE, this.mode.method_15434());
	}

	public enum Mode implements class_3542, Predicate<class_1429> {
		FEED_ADULTS("feed_adults", Predicate.not(class_1429::method_6109)),
		FEED_BABIES("feed_babies", class_1429::method_6109),
		FEED_ALL("feed_all", animal -> true);

		@SuppressWarnings("deprecation")
		private static final class_7292<Mode> CODEC = class_3542.method_28140(Mode::values);

		public static Mode forName(String name) {
			return CODEC.method_47920(name, FEED_ADULTS);
		}

		@NotNull
		private final String name;
		@NotNull
		private final Predicate<class_1429> predicate;

		Mode(@NotNull String name, @NotNull Predicate<class_1429> predicate) {
			this.name = name;
			this.predicate = predicate;
		}

		@Override
		public boolean test(class_1429 animal) {
			return predicate.test(animal);
		}

		@NotNull
		@Override
		public String method_15434() {
			return this.name;
		}

		@NotNull
		public Mode getNextMode() {
			Mode[] modes = values();
			int nextMode = ordinal() + 1;
			return modes[nextMode % modes.length];
		}
	}

	public static class WandHud extends BindableFlowerWandHud<PollidisiacBlockEntity> {
		public WandHud(PollidisiacBlockEntity flower) {
			super(flower);
		}

		@Override
		public void renderHUD(class_332 gui, class_310 mc) {
			String filter = class_1074.method_4662("botaniamisc.pollidisiac." + flower.getMode().method_15434());
			int filterWidth = mc.field_1772.method_1727(filter);
			int filterTextStart = (mc.method_22683().method_4486() - filterWidth) / 2;
			int halfMinWidth = (filterWidth + 4) / 2;
			int centerY = mc.method_22683().method_4502() / 2;

			super.renderHUD(gui, mc, halfMinWidth, halfMinWidth, 40);
			gui.method_25303(mc.field_1772, filter, filterTextStart, centerY + 30, flower.getColor());
		}
	}
}
