/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.item;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.item.BlockProvider;
import vazkii.botania.api.mana.ManaItemHandler;
import vazkii.botania.client.gui.ItemsRemainingRenderHandler;
import vazkii.botania.common.handler.BotaniaSounds;
import vazkii.botania.common.helper.BlockProviderHelper;
import vazkii.botania.common.helper.ItemNBTHelper;
import vazkii.botania.common.helper.PlayerHelper;
import vazkii.botania.common.item.equipment.tool.ToolCommons;
import vazkii.botania.common.item.relic.RingOfLokiItem;
import vazkii.botania.common.item.rod.ShiftingCrustRodItem;
import vazkii.botania.xplat.XplatAbstractions;
import D;
import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_124;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1271;
import net.minecraft.class_1320;
import net.minecraft.class_1657;
import net.minecraft.class_1750;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_1838;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_239;
import net.minecraft.class_2512;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_3965;
import net.minecraft.class_7871;
import net.minecraft.class_7924;

public class AstrolabeItem extends class_1792 {

	public static final String TAG_BLOCKSTATE = "blockstate";
	public static final String TAG_SIZE = "size";
	public static final int BASE_COST = 320;

	public AstrolabeItem(class_1793 props) {
		super(props);
	}

	@NotNull
	@Override
	public class_1269 method_7884(class_1838 ctx) {
		class_1799 stack = ctx.method_8041();
		class_2680 state = ctx.method_8045().method_8320(ctx.method_8037());
		class_1657 player = ctx.method_8036();

		if (player != null && player.method_21823()) {
			if (setBlock(stack, state)) {
				displayRemainderCounter(player, stack);
				return class_1269.method_29236(player.method_37908().method_8608());
			}
		} else if (player != null) {
			boolean did = placeAllBlocks(stack, player, ctx.method_20287());
			if (did) {
				displayRemainderCounter(player, stack);
			}

			return did ? class_1269.field_5812 : class_1269.field_5814;
		}

		return class_1269.field_5811;
	}

	@NotNull
	@Override
	public class_1271<class_1799> method_7836(class_1937 worldIn, class_1657 playerIn, @NotNull class_1268 hand) {
		class_1799 stack = playerIn.method_5998(hand);
		if (playerIn.method_21823()) {
			playerIn.method_5783(BotaniaSounds.astrolabeConfigure, 1F, 1F);
			if (!worldIn.field_9236) {
				int size = getSize(stack);
				int newSize = size == 11 ? 3 : size + 2;
				setSize(stack, newSize);
				ItemsRemainingRenderHandler.send(playerIn, stack, 0, class_2561.method_43470(newSize + "x" + newSize));
			}

			return class_1271.method_29237(stack, worldIn.method_8608());
		}

		return class_1271.method_22430(stack);
	}

	public boolean placeAllBlocks(class_1799 requester, class_1657 player, class_1268 hand) {
		class_2248 blockToPlace = getBlock(requester, player.method_37908().method_45448(class_7924.field_41254));
		int size = getSize(requester);
		class_1750 ctx = getBlockPlaceContext(player, hand, blockToPlace);
		List<class_2338> placePositions = getPlacePositions(ctx, size);
		List<BlockProvider> blockProviders = findBlockProviders(requester, player, placePositions.size(), blockToPlace);
		if (ctx == null || blockProviders.isEmpty()) {
			return false;
		}

		int cost = size * BASE_COST;
		if (!ManaItemHandler.instance().requestManaExact(requester, player, cost, true)) {
			return false;
		}

		for (class_2338 coords : placePositions) {
			if (!placeBlockAndConsume(requester, blockToPlace, ctx, coords, blockProviders)) {
				break;
			}
		}

		return true;
	}

	/**
	 * Attempts to place the specified block and consume it from the player's inventory.
	 *
	 * @return {@code true} if continuing to attempt placing blocks makes sense,
	 *         {@code false} if not, e.g. because there are fewer blocks available than expected.
	 */
	private boolean placeBlockAndConsume(class_1799 requester, class_2248 blockToPlace, class_1750 ctx,
			class_2338 pos, List<BlockProvider> providers) {
		final class_1657 player = ctx.method_8036();
		if (player == null) {
			return false;
		}

		class_2680 state = blockToPlace.method_9605(ctx);
		if (state == null) {
			return true;
		}

		if (providers.stream().noneMatch(prov -> prov.provideBlock(player, requester, blockToPlace, false))) {
			// don't place blocks we don't have (e.g. because mana calculations were inaccurate somehow)
			return false;
		}

		class_1838 useOnContext = RingOfLokiItem.getUseOnContext(player, ctx.method_20287(), pos, ctx.method_17698(), ctx.method_8038());
		if (!PlayerHelper.substituteUse(useOnContext, new class_1799(blockToPlace)).method_23665()) {
			return true;
		}
		for (BlockProvider prov : providers) {
			if (prov.provideBlock(player, requester, blockToPlace, true)) {
				break;
			}
		}
		return true;
	}

	public static boolean hasBlocks(class_1799 requester, class_1657 player, int required, class_2248 block) {
		if (player.method_31549().field_7477) {
			return true;
		}

		return !findBlockProviders(requester, player, required, block).isEmpty();
	}

	public static List<BlockProvider> findBlockProviders(class_1799 requester, class_1657 player, int required, class_2248 block) {
		if (block == class_2246.field_10124 || required == 0) {
			return List.of();
		}
		if (player.method_31549().field_7477) {
			return List.of(BlockProviderHelper.asInfiniteBlockProvider(new class_1799(block)));
		}

		int current = 0;
		List<BlockProvider> providersToCheck = new ArrayList<>();
		for (int i = 0; i < player.method_31548().method_5439(); i++) {
			class_1799 stackInSlot = player.method_31548().method_5438(i);
			if (stackInSlot.method_7960()) {
				continue;
			}
			if (stackInSlot.method_31574(block.method_8389())) {
				current += stackInSlot.method_7947();
				final var stackProvider = BlockProviderHelper.asBlockProvider(stackInSlot);
				providersToCheck.add(stackProvider);
			} else {
				var provider = XplatAbstractions.INSTANCE.findBlockProvider(stackInSlot);
				if (provider != null) {
					final int count = provider.getBlockCount(player, requester, block);
					if (count != 0) {
						current += count;
						providersToCheck.add(provider);
					}
				}
			}
		}

		return current >= required ? providersToCheck : List.of();
	}

	@Nullable
	public static class_1750 getBlockPlaceContext(class_1657 player, class_1268 hand, class_2248 blockToPlace) {
		if (blockToPlace == class_2246.field_10124) {
			return null;
		}
		var reachAttribute = XplatAbstractions.INSTANCE.getReachDistanceAttribute();
		var bonusReach = reachAttribute != null ? player.method_26825(reachAttribute) : 0;
		// Should probably not always have creative base reach, but it's grandfathered in now
		class_3965 rtr = ToolCommons.raytraceFromEntity(player, 5 + bonusReach, false);
		return rtr.method_17783() == class_239.class_240.field_1332
				? new class_1750(player, hand, new class_1799(blockToPlace.method_8389()), rtr)
				: null;
	}

	public static List<class_2338> getPlacePositions(class_1750 ctx, int size) {
		if (ctx == null || ctx.method_8036() == null) {
			return List.of();
		}
		List<class_2338> coords = new ArrayList<>();
		class_2338 pos = ctx.method_8037();
		class_2680 clickedState = ctx.method_8045().method_8320(pos);
		if (clickedState.method_45474() || clickedState.method_26166(ctx)) {
			pos = pos.method_10093(ctx.method_8038().method_10153());
		}

		int range = (size ^ 1) / 2;

		class_2350 dir = ctx.method_8038();
		class_2350 rotationDir = class_2350.method_10150(ctx.method_8036().method_36454());

		boolean pitchedVertically = Math.abs(ctx.method_8036().method_36455()) > 50;

		boolean axisX = rotationDir.method_10166() == class_2351.field_11048;
		boolean axisZ = rotationDir.method_10166() == class_2351.field_11051;

		int xOff = axisZ || pitchedVertically ? range : 0;
		int yOff = pitchedVertically ? 0 : range;
		int zOff = axisX || pitchedVertically ? range : 0;

		for (int x = -xOff; x < xOff + 1; x++) {
			for (int y = 0; y < yOff * 2 + 1; y++) {
				for (int z = -zOff; z < zOff + 1; z++) {
					int xp = pos.method_10263() + x + dir.method_10148();
					int yp = pos.method_10264() + y + dir.method_10164();
					int zp = pos.method_10260() + z + dir.method_10165();

					class_2338 newPos = new class_2338(xp, yp, zp);
					class_2680 state = ctx.method_8045().method_8320(newPos);
					if (ctx.method_8045().method_8621().method_11952(newPos)
							&& (state.method_26215() || state.method_45474() || state.method_26166(ctx))) {
						coords.add(newPos);
					}
				}
			}
		}

		return coords;
	}

	public void displayRemainderCounter(class_1657 player, class_1799 stack) {
		class_2248 block = getBlock(stack, player.method_37908().method_45448(class_7924.field_41254));
		int count = ShiftingCrustRodItem.getInventoryItemCount(player, stack, block.method_8389());
		if (!player.method_37908().field_9236) {
			ItemsRemainingRenderHandler.send(player, new class_1799(block), count);
		}
	}

	public static boolean setBlock(class_1799 stack, class_2680 state) {
		if (!state.method_26215()) {
			// This stores a block state (instead of just the block ID) for legacy reasons.
			ItemNBTHelper.setCompound(stack, TAG_BLOCKSTATE, class_2512.method_10686(state.method_26204().method_9564()));
			return true;
		}
		return false;
	}

	public static void setSize(class_1799 stack, int size) {
		ItemNBTHelper.setInt(stack, TAG_SIZE, size | 1);
	}

	public static int getSize(class_1799 stack) {
		return ItemNBTHelper.getInt(stack, TAG_SIZE, 3) | 1;
	}

	public static class_2248 getBlock(class_1799 stack, class_7871<class_2248> holderGetter) {
		return getBlockState(stack, holderGetter).method_26204();
	}

	public static class_2680 getBlockState(class_1799 stack, class_7871<class_2248> holderGetter) {
		return class_2512.method_10681(holderGetter, ItemNBTHelper.getCompound(stack, TAG_BLOCKSTATE, false));
	}

	@Override
	public void method_7851(class_1799 stack, @Nullable class_1937 level, List<class_2561> tip, class_1836 flags) {
		if (level == null) {
			return;
		}

		class_2248 block = getBlock(stack, level.method_45448(class_7924.field_41254));
		int size = getSize(stack);

		tip.add(class_2561.method_43470(size + " x " + size));
		if (block != class_2246.field_10124) {
			tip.add(new class_1799(block).method_7964().method_27662().method_27692(class_124.field_1080));
		}
	}

}
