package vazkii.botania.common.helper;

import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import vazkii.botania.common.lib.BotaniaTags;
import vazkii.botania.xplat.XplatAbstractions;
import java.util.Map.Entry;
import java.util.WeakHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import net.minecraft.class_1297;
import net.minecraft.class_1541;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2530;
import net.minecraft.class_2586;
import net.minecraft.class_2669;
import net.minecraft.class_2680;

/**
 * Helps track TNT entity spawning to check for unethical methods. Used with vanilla piston logic (via mixin) and
 * Force lens block moving (which is used by mana bursts with the lens itself, but also by the Force Relay block).
 * <br>
 * The concept of this check is to keep track of any TNT entity that was spawned as a result of a block update while a
 * block structure is converted into moving blocks. Moving blocks are placed in the destination position, so we look for
 * adjacent moving block entities that represent a TNT block and move away from the TNT entity. If we find such a moving
 * block, the TNT entity was unethically sourced.
 *
 * @see vazkii.botania.common.block.flower.generating.EntropinnyumBlockEntity
 * @see vazkii.botania.common.internal_caps.EthicalComponent
 * @see vazkii.botania.mixin.PistonBaseBlockMixin
 * @see vazkii.botania.common.item.lens.ForceLens#moveBlocks(class_1937, class_2338, class_2350)
 */
public class EthicalTntHelper {
	/**
	 * In case the various loaded dimensions' tick loops are run in their own threads, tracking data is kept
	 * thread-local.
	 */
	private static final ThreadLocal<EthicalTntHelper> tracker = ThreadLocal.withInitial(EthicalTntHelper::new);

	/**
	 * While the moveBlocks method should not be reentrant, we can't know what crazy ideas some mixin authors might
	 * have. To be safe, we account for reentrancy and hope that we don't miss any method exits. We also assume nobody
	 * is crazy enough to spread the blocks-start-to-move logic across multiple threads.
	 */
	private final AtomicInteger trackTntEntities = new AtomicInteger();

	/**
	 * To reduce the risk of memory leaks, TNT entities are stored using their ID and a weak reference to their Level.
	 */
	private final WeakHashMap<class_1937, IntOpenHashSet> trackedTntEntities = new WeakHashMap<>();

	/**
	 * A force relay, force lens mana burst or vanilla piston is about to start moving blocks.
	 */
	public static void startTrackingTntEntities() {
		tracker.get().startTracking();
	}

	/**
	 * A TNT entity just spawned. Check it for potentially unethical spawning methods.
	 *
	 * @param entity The TNT entity.
	 */
	public static void addTrackedTntEntity(class_1541 entity) {
		tracker.get().addTrackedEntity(entity);
	}

	/**
	 * A force relay, force lens mana burst,or vanilla piston finished converting all blocks into moving block
	 * entities.
	 */
	public static void endTrackingTntEntitiesAndCheck() {
		tracker.get().endTracking();
	}

	private void startTracking() {
		trackTntEntities.incrementAndGet();
	}

	private void addTrackedEntity(class_1541 entity) {
		if (trackTntEntities.get() > 0) {
			trackedTntEntities.computeIfAbsent(entity.method_37908(), lvl -> new IntOpenHashSet()).add(entity.method_5628());
		}
	}

	private void endTracking() {
		if (trackTntEntities.decrementAndGet() == 0) {
			for (final var entry : trackedTntEntities.entrySet()) {
				final var level = entry.getKey();
				final var trackedEntities = entry.getValue();
				if (trackedEntities != null) {
					for (final var tntId : trackedEntities) {
						final var entity = level.method_8469(tntId);
						if (entity instanceof class_1541 tnt) {
							checkUnethical(tnt);
						}
					}
					trackedEntities.clear();
				}
			}
		}
	}

	private static void checkUnethical(class_1541 entity) {
		class_2338 center = entity.method_24515();
		if (!entity.method_37908().method_8477(center)) {
			return;
		}

		class_2338.class_2339 blockPos = new class_2338.class_2339();
		for (final var dir : class_2350.values()) {
			blockPos.method_25505(center, dir);
			if (!entity.method_37908().method_8477(blockPos)) {
				continue;
			}

			final var blockState = entity.method_37908().method_8320(blockPos);
			if (blockState.method_27852(class_2246.field_10008)) {
				final var blockEntity = entity.method_37908().method_8321(blockPos);
				if (blockEntity instanceof class_2669 movingBlockEntity
						&& movingBlockEntity.method_11506() == dir
						&& (movingBlockEntity.method_11495().method_26204() instanceof class_2530
								|| movingBlockEntity.method_11495().method_26164(BotaniaTags.Blocks.UNETHICAL_TNT_CHECK))) {
					// found a moving block that marks the destination of a TNT block moving away from the TNT entity
					XplatAbstractions.INSTANCE.ethicalComponent(entity).markUnethical();
					break;
				}
			}
		}
	}
}
