/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.command;

import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.mojang.brigadier.tree.LiteralCommandNode;
import vazkii.botania.common.world.IslandPos;
import vazkii.botania.common.world.SkyblockChunkGenerator;
import vazkii.botania.common.world.SkyblockSavedData;
import vazkii.botania.common.world.SkyblockWorldEvents;

import java.util.UUID;
import net.minecraft.class_156;
import net.minecraft.class_1657;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2172;
import net.minecraft.class_2186;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_5242;

public class SkyblockCommand {
	private static final SimpleCommandExceptionType NOT_SKYBLOCK_WORLD = new SimpleCommandExceptionType(
			class_2561.method_43471("botaniamisc.command.skyblock.world"));
	private static final SimpleCommandExceptionType NO_ISLAND = new SimpleCommandExceptionType(
			class_2561.method_43471("botaniamisc.command.skyblock.noisland"));

	public static void register(CommandDispatcher<class_2168> dispatcher) {
		// This isn't what we consider the "primary" name. It's just here to be a reminder for old /botania-skyblock-spread users.
		// However some Mojang code seems to assume that aliases are made alphabetically...
		LiteralArgumentBuilder<class_2168> commandBuilder = class_2170.method_9247("botania-skyblock")
				.requires(s -> s.method_9259(2))
				.then(class_2170.method_9247("help")
						.executes(SkyblockCommand::printHelp))
				.then(class_2170.method_9247("island")
						.then(class_2170.method_9244("player", class_2186.method_9305())
								.executes(SkyblockCommand::createIsland)))
				.then(class_2170.method_9247("spawn")
						.executes(SkyblockCommand::teleportToSpawn))

				.then(class_2170.method_9247("visit")
						.then(class_2170.method_9244("player", class_2186.method_9305())
								.executes(ctx -> teleportToIsland(ctx, class_2186.method_9315(ctx, "player")))
						)
						.then(class_2170.method_9244("playerUuid", class_5242.method_27643())
								.suggests((ctx, builder) -> class_2172.method_9264(
										SkyblockSavedData.get(ctx.getSource().method_9225()).skyblocks
												.values().stream().map(UUID::toString),
										builder))
								.executes(ctx -> teleportToIsland(ctx, class_5242.method_27645(ctx, "playerUuid")))
						)
				)

				.then(class_2170.method_9247("regen-island")
						.then(class_2170.method_9244("player", class_2186.method_9305())
								.executes(ctx -> rebuildIsland(ctx, class_2186.method_9315(ctx, "player")))
						)
						.then(class_2170.method_9244("playerUuid", class_5242.method_27643())
								.suggests((ctx, builder) -> class_2172.method_9264(
										SkyblockSavedData.get(ctx.getSource().method_9225()).skyblocks
												.values().stream().map(UUID::toString),
										builder))
								.executes(ctx -> rebuildIsland(ctx, class_5242.method_27645(ctx, "playerUuid")))
						)
				);
		LiteralCommandNode<class_2168> command = dispatcher.register(commandBuilder);
		dispatcher.register(class_2170.method_9247("gardenofglass").redirect(command));
		dispatcher.register(class_2170.method_9247("gog").redirect(command));
	}

	private static int printHelp(CommandContext<class_2168> ctx) {
		for (int i = 0; i < 5; i++) {
			int finalI = i;
			ctx.getSource().method_9226(() -> class_2561.method_43471("botaniamisc.command.skyblock.help." + finalI), false);
		}
		return Command.SINGLE_SUCCESS;
	}

	private static int doTeleportToIsland(CommandContext<class_2168> ctx, UUID owner, class_2561 feedback) throws CommandSyntaxException {
		class_3222 player = ctx.getSource().method_9207();
		return doTeleportToIsland(ctx, player, owner, feedback);
	}

	private static int doTeleportToIsland(CommandContext<class_2168> ctx, class_3222 player, UUID owner, class_2561 feedback) throws CommandSyntaxException {
		class_3218 world = getSkyblockWorld(ctx);
		IslandPos pos = getIslandForUUID(owner, SkyblockSavedData.get(world));

		class_2338 blockPos = pos.getCenter();

		player.method_14251(world, blockPos.method_10263() + 0.5, blockPos.method_10264(),
				blockPos.method_10260() + 0.5, player.method_36454(), player.method_36455());
		ctx.getSource().method_9226(() -> feedback, true);
		return Command.SINGLE_SUCCESS;
	}

	private static int createIsland(CommandContext<class_2168> ctx) throws CommandSyntaxException {
		class_3222 player = class_2186.method_9315(ctx, "player");
		SkyblockSavedData data = SkyblockSavedData.get(getSkyblockWorld(ctx));
		UUID uuid = player.method_5667();

		if (data.skyblocks.containsValue(uuid)) {
			doTeleportToIsland(ctx, player, uuid, class_2561.method_43469("botaniamisc.command.skyblock.island.teleported",
					ctx.getSource().method_9223()));
			return Command.SINGLE_SUCCESS;
		}

		SkyblockWorldEvents.spawnPlayer(player, data.create(uuid));
		ctx.getSource().method_9226(() -> class_2561.method_43469("botaniamisc.command.skyblock.island.success", player.method_5476()), true);
		return Command.SINGLE_SUCCESS;
	}

	private static int doRebuildIsland(CommandContext<class_2168> ctx, UUID player, class_2561 feedback) throws CommandSyntaxException {
		class_3218 world = getSkyblockWorld(ctx);
		IslandPos pos = getIslandForUUID(player, SkyblockSavedData.get(world));

		SkyblockWorldEvents.createSkyblock(world, pos.getCenter());
		ctx.getSource().method_9226(() -> feedback, true);
		return Command.SINGLE_SUCCESS;
	}

	// Helper methods
	private static IslandPos getIslandForUUID(UUID player, SkyblockSavedData data) throws CommandSyntaxException {
		IslandPos pos = data.skyblocks.inverse().get(player);
		if (pos == null) {
			throw NO_ISLAND.create();
		}
		return pos;
	}

	private static class_3218 getSkyblockWorld(CommandContext<class_2168> ctx) throws CommandSyntaxException {
		class_3218 world = ctx.getSource().method_9225();
		if (!SkyblockChunkGenerator.isWorldSkyblock(world)) {
			throw NOT_SKYBLOCK_WORLD.create();
		}
		return world;
	}

	// Translation feedback helpers
	private static int teleportToIsland(CommandContext<class_2168> ctx, class_1657 owner) throws CommandSyntaxException {
		return doTeleportToIsland(ctx, owner.method_5667(), class_2561.method_43469("botaniamisc.command.skyblock.teleport.success",
				ctx.getSource().method_9223(), owner.method_5477()));
	}

	private static int teleportToIsland(CommandContext<class_2168> ctx, UUID owner) throws CommandSyntaxException {
		return doTeleportToIsland(ctx, owner, class_2561.method_43469("botaniamisc.command.skyblock.teleport.success",
				ctx.getSource().method_9223(), owner));
	}

	private static int teleportToSpawn(CommandContext<class_2168> ctx) throws CommandSyntaxException {
		return doTeleportToIsland(ctx, class_156.field_25140, class_2561.method_43469("botaniamisc.command.skyblock.spawn.success",
				ctx.getSource().method_9223()));
	}

	private static int rebuildIsland(CommandContext<class_2168> ctx, class_3222 owner) throws CommandSyntaxException {
		return doRebuildIsland(ctx, owner.method_5667(), class_2561.method_43469("botaniamisc.command.skyblock.regenisland.success", owner.method_5476()));
	}

	private static int rebuildIsland(CommandContext<class_2168> ctx, UUID owner) throws CommandSyntaxException {
		return doRebuildIsland(ctx, owner, class_2561.method_43469("botaniamisc.command.skyblock.regenisland.success", owner));
	}
}
