/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.flower.functional;

import com.google.common.base.Suppliers;

import it.unimi.dsi.fastutil.Pair;
import it.unimi.dsi.fastutil.longs.LongSet;
import it.unimi.dsi.fastutil.objects.*;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_1267;
import net.minecraft.class_1294;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_1304;
import net.minecraft.class_1308;
import net.minecraft.class_1309;
import net.minecraft.class_1324;
import net.minecraft.class_173;
import net.minecraft.class_1799;
import net.minecraft.class_181;
import net.minecraft.class_1831;
import net.minecraft.class_2338;
import net.minecraft.class_238;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_270;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3195;
import net.minecraft.class_3218;
import net.minecraft.class_332;
import net.minecraft.class_3449;
import net.minecraft.class_3730;
import net.minecraft.class_3732;
import net.minecraft.class_5134;
import net.minecraft.class_5138;
import net.minecraft.class_52;
import net.minecraft.class_5250;
import net.minecraft.class_5712;
import net.minecraft.class_5819;
import net.minecraft.class_60;
import net.minecraft.class_6088;
import net.minecraft.class_7061;
import net.minecraft.class_7924;
import net.minecraft.class_8567;
import net.minecraft.world.entity.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.block_entity.FunctionalFlowerBlockEntity;
import vazkii.botania.api.block_entity.RadiusDescriptor;
import vazkii.botania.api.configdata.ConfigDataManager;
import vazkii.botania.api.configdata.LooniumMobAttributeModifier;
import vazkii.botania.api.configdata.LooniumMobEffectToApply;
import vazkii.botania.api.configdata.LooniumMobSpawnData;
import vazkii.botania.api.configdata.LooniumStructureConfiguration;
import vazkii.botania.common.block.BotaniaFlowerBlocks;
import vazkii.botania.common.internal_caps.LooniumComponent;
import vazkii.botania.common.lib.BotaniaTags;
import vazkii.botania.common.loot.BotaniaLootTables;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static vazkii.botania.common.lib.ResourceLocationHelper.prefix;

public class LooniumBlockEntity extends FunctionalFlowerBlockEntity {
	private static final int RANGE = 5;
	private static final int CHECK_RANGE = 9;
	private static final String TAG_LOOT_TABLE = "lootTable";
	private static final String TAG_DETECTED_STRUCTURE = "detectedStructure";
	private static final String TAG_CONFIG_OVERRIDE = "configOverride";
	private static final String TAG_ATTUNE_DISPLAY_OVERRIDE = "attuneDisplayOverride";
	private static final Supplier<LooniumStructureConfiguration> FALLBACK_CONFIG =
			Suppliers.memoize(() -> LooniumStructureConfiguration.builder()
					.manaCost(LooniumStructureConfiguration.DEFAULT_COST)
					.maxNearbyMobs(LooniumStructureConfiguration.DEFAULT_MAX_NEARBY_MOBS)
					.boundingBoxType(class_7061.class_7062.field_37199)
					.spawnedMobs(LooniumMobSpawnData.entityWeight(class_1299.field_6051, 1).build())
					.attributeModifiers()
					.effectsToApply(
							LooniumMobEffectToApply.effect(class_1294.field_5924).build(),
							LooniumMobEffectToApply.effect(class_1294.field_5918).build(),
							LooniumMobEffectToApply.effect(class_1294.field_5907).build(),
							LooniumMobEffectToApply.effect(class_1294.field_5910).build()
					)
					.build());

	// this should never collide with the /team command, since space is not allowed in scoreboard team names
	public static final String LOONIUM_TEAM_NAME = "Loonium Monsters";
	public static final class_270 LOONIUM_TEAM = new class_270() {
		@NotNull
		@Override
		public String method_1197() {
			return LOONIUM_TEAM_NAME;
		}

		@NotNull
		@Override
		public class_5250 method_1198(class_2561 component) {
			return component.method_27661();
		}

		@Override
		public boolean method_1199() {
			return true;
		}

		@Override
		public boolean method_1205() {
			return true;
		}

		@NotNull
		@Override
		public class_272 method_1201() {
			return class_272.field_1442;
		}

		@NotNull
		@Override
		public class_124 method_1202() {
			return class_124.field_1070;
		}

		@NotNull
		@Override
		public Collection<String> method_1204() {
			return List.of();
		}

		@NotNull
		@Override
		public class_272 method_1200() {
			return class_272.field_1442;
		}

		@NotNull
		@Override
		public class_271 method_1203() {
			return class_271.field_1437;
		}
	};

	@Nullable
	private class_2960 lootTableOverride;
	@Nullable
	private Object2BooleanMap<class_2960> detectedStructures;
	@Nullable
	private class_2960 configOverride;
	@Nullable
	private String attuneDisplayOverride;

	public LooniumBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaFlowerBlocks.LOONIUM, pos, state);
	}

	@Override
	public void tickFlower() {
		super.tickFlower();

		if (!(method_10997() instanceof class_3218 world)) {
			return;
		}

		if (detectedStructures == null) {
			// Detection intentionally uses the flower position, not the effective position,
			// since the latter could change while this detection is only executed once.
			detectStructure(world);
		}

		if (redstoneSignal != 0 || ticksExisted % 100 != 0 || world.method_8407() == class_1267.field_5801
		// so mobs won't spawn in unloaded or border chunks
				|| !world.method_37118(getEffectivePos())) {
			return;
		}

		ConfigDataManager configData = BotaniaAPI.instance().getConfigData();
		Map<class_2960, LooniumStructureConfiguration> structureConfigs = determineStructureConfigs(configData, detectedStructures);
		List<Pair<class_2960, class_52>> lootTables = determineLootTables(world, structureConfigs.keySet());

		if (lootTables.isEmpty()) {
			return;
		}

		Pair<class_2960, class_52> randomPick = lootTables.get(world.field_9229.method_43048(lootTables.size()));
		LooniumStructureConfiguration pickedConfig = structureConfigs.getOrDefault(randomPick.key(),
				structureConfigs.get(LooniumStructureConfiguration.DEFAULT_CONFIG_ID));
		class_52 pickedLootTable = randomPick.value();

		if (getMana() < pickedConfig.manaCost) {
			return;
		}

		int numberOfMobsAround = countNearbyMobs(world, pickedConfig);
		if (numberOfMobsAround >= pickedConfig.maxNearbyMobs) {
			return;
		}

		LooniumMobSpawnData pickedMobType = pickedConfig.spawnedMobs.method_34992(world.field_9229).orElse(null);
		if (pickedMobType == null) {
			return;
		}

		spawnMob(world, pickedMobType, pickedConfig, pickedLootTable);
	}

	private void spawnMob(class_3218 world, LooniumMobSpawnData pickedMobType,
			LooniumStructureConfiguration pickedConfig, class_52 pickedLootTable) {

		class_1799 lootStack = pickRandomLootItem(world, pickedLootTable);
		if (lootStack.method_7960()) {
			return;
		}

		class_5819 random = world.field_9229;
		double x = getEffectivePos().method_10263() + 0.5 - RANGE + 2 * RANGE * random.method_43058();
		double y = getEffectivePos().method_10264();
		double z = getEffectivePos().method_10260() + 0.5 - RANGE + 2 * RANGE * random.method_43058();

		while (!world.method_18026(pickedMobType.type.method_17683(x, y, z))) {
			y += 1.0;
			if (y >= world.method_31600()) {
				return;
			}
		}

		class_1297 entity = pickedMobType.type.method_5883(world);
		if (!(entity instanceof class_1308 mob)) {
			return;
		}

		if (pickedMobType.nbt != null) {
			mob.method_5749(pickedMobType.nbt);
		}
		if (pickedMobType.spawnAsBaby != null) {
			mob.method_7217(pickedMobType.spawnAsBaby);
		}

		mob.method_5641(x, y, z, random.method_43057() * 360F, 0);
		mob.method_18799(class_243.field_1353);

		applyAttributesAndEffects(pickedMobType, pickedConfig, mob);

		LooniumComponent looniumComponent = XplatAbstractions.INSTANCE.looniumComponent(mob);
		if (looniumComponent != null) {
			looniumComponent.setSlowDespawn(true);
			looniumComponent.setOverrideDrop(true);
			looniumComponent.setDrop(lootStack);
		}

		mob.method_5943(world, world.method_8404(mob.method_24515()), class_3730.field_16469, null, null);
		if (Boolean.FALSE.equals(pickedMobType.spawnAsBaby) && mob.method_6109()) {
			// Note: might have already affected initial equipment/attribute selection, or even caused a special
			// mob configuration (such as chicken jockey) to spawn, which may look weird when reverting to adult.
			mob.method_7217(false);
		}

		if (pickedMobType.equipmentTable != null) {
			class_52 equipmentTable = world.method_8503().method_3857().getLootTable(pickedMobType.equipmentTable);
			if (equipmentTable != class_52.field_948) {
				class_8567 lootParams = new class_8567.class_8568(world)
						.method_51874(class_181.field_1226, mob)
						.method_51874(class_181.field_24424, mob.method_19538())
						// TODO 1.21: replace with LootContextParamSets.EQUIPMENT
						.method_51875(class_173.field_20762);
				var equippedSlots = new HashSet<class_1304>();
				equipmentTable.method_51882(lootParams, equipmentStack -> {
					class_1304 slot = equipmentStack.method_31573(BotaniaTags.Items.LOONIUM_OFFHAND_EQUIPMENT)
							? class_1304.field_6171
							: class_1309.method_32326(equipmentStack);
					if (equippedSlots.contains(slot)) {
						slot = equippedSlots.contains(class_1304.field_6173)
								&& !(equipmentStack.method_7909() instanceof class_1831)
										? class_1304.field_6171
										: class_1304.field_6173;
					}
					if (!equippedSlots.add(slot)) {
						return;
					}
					mob.method_5673(slot, slot.method_46643() ? equipmentStack.method_46651(1) : equipmentStack);
				});
			}
		}

		// in case the mob spawned with a vehicle or passenger(s), ensure those don't drop unexpected loot
		mob.method_5668().method_31748().forEach(e -> {
			if (e instanceof class_1308 otherMob) {
				// prevent armor/weapon drops on player kill, also no nautilus shells from drowned:
				Arrays.stream(class_1304.values()).forEach(slot -> otherMob.method_5946(slot, 0));

				if (mob instanceof class_3732 patroller && patroller.method_16219()) {
					//  Loonium may be presenting challenges, but not that type of challenge
					patroller.method_16217(false);
					patroller.method_5673(class_1304.field_6169, class_1799.field_8037);
				}

				if (e == mob) {
					return;
				}

				Optional<LooniumMobSpawnData> mobType = pickedConfig.spawnedMobs.method_34994().stream()
						.filter(mobSpawnData -> mobSpawnData.type.method_31488(otherMob) != null).findFirst();

				class_1799 bonusLoot;
				if (mobType.isPresent()) {
					applyAttributesAndEffects(mobType.get(), pickedConfig, mob);
					bonusLoot = pickRandomLootItem(world, pickedLootTable);
				} else {
					bonusLoot = class_1799.field_8037;
				}

				LooniumComponent otherLooniumComponent = XplatAbstractions.INSTANCE.looniumComponent(otherMob);
				if (otherLooniumComponent != null) {
					otherLooniumComponent.setSlowDespawn(true);
					otherLooniumComponent.setOverrideDrop(true);
					otherLooniumComponent.setDrop(bonusLoot);
				}
			}
		});

		if (!world.method_30736(mob)) {
			return;
		}

		mob.method_5990();
		world.method_20290(class_6088.field_31147, method_11016(), 0);
		world.method_43275(mob, class_5712.field_28738, mob.method_19538());

		addMana(-pickedConfig.manaCost);
		sync();
	}

	private static void applyAttributesAndEffects(LooniumMobSpawnData mobSpawnData,
			LooniumStructureConfiguration pickedConfig, class_1308 mob) {
		List<LooniumMobAttributeModifier> attributeModifiers = mobSpawnData.attributeModifiers != null
				? mobSpawnData.attributeModifiers
				: pickedConfig.attributeModifiers;
		for (LooniumMobAttributeModifier attributeModifier : attributeModifiers) {
			class_1324 attribute = mob.method_5996(attributeModifier.attribute);
			if (attribute != null) {
				attribute.method_26837(attributeModifier.createAttributeModifier());
				if (attribute.method_6198() == class_5134.field_23716) {
					mob.method_6033(mob.method_6063());
				}
			}
		}

		List<LooniumMobEffectToApply> effectsToApply = mobSpawnData.effectsToApply != null
				? mobSpawnData.effectsToApply
				: pickedConfig.effectsToApply;
		for (LooniumMobEffectToApply effectToApply : effectsToApply) {
			mob.method_6092(effectToApply.createMobEffectInstance());
		}
	}

	private int countNearbyMobs(class_3218 world, LooniumStructureConfiguration pickedConfig) {
		var setOfMobTypes = pickedConfig.spawnedMobs.method_34994().stream().map(msd -> msd.type).collect(Collectors.toSet());
		return world.method_8390(class_1308.class, new class_238(getEffectivePos()).method_1014(CHECK_RANGE),
				m -> setOfMobTypes.contains(m.method_5864())).size();
	}

	private static class_1799 pickRandomLootItem(class_3218 world, class_52 pickedLootTable) {
		class_8567 params = new class_8567.class_8568(world).method_51875(class_173.field_1175);
		List<class_1799> stacks = pickedLootTable.method_51879(params, world.field_9229.method_43055());
		stacks.removeIf(s -> s.method_7960() || s.method_31573(BotaniaTags.Items.LOONIUM_BLACKLIST));
		if (stacks.isEmpty()) {
			return class_1799.field_8037;
		} else {
			Collections.shuffle(stacks);
			return stacks.get(0);
		}
	}

	@NotNull
	private List<Pair<class_2960, class_52>> determineLootTables(class_3218 world,
			Set<class_2960> structureIds) {
		var lootTables = new ArrayList<Pair<class_2960, class_52>>();
		class_60 lootData = world.method_8503().method_3857();
		Supplier<class_52> defaultLootTableSupplier = Suppliers.memoize(() -> lootData.getLootTable(
				BotaniaLootTables.LOONIUM_DEFAULT_LOOT));
		if (lootTableOverride != null) {
			class_52 lootTable = lootData.getLootTable(lootTableOverride);
			if (lootTable != class_52.field_948) {
				lootTables.add(Pair.of(LooniumStructureConfiguration.DEFAULT_CONFIG_ID, lootTable));
			}
		} else {
			for (class_2960 structureId : structureIds) {
				if (structureId.equals(LooniumStructureConfiguration.DEFAULT_CONFIG_ID)) {
					continue;
				}
				class_2960 lootTableId = prefix("loonium/%s/%s".formatted(structureId.method_12836(), structureId.method_12832()));
				class_52 lootTable = lootData.getLootTable(lootTableId);
				if (lootTable != class_52.field_948) {
					lootTables.add(Pair.of(structureId, lootTable));
				} else {
					class_52 defaultLootTable = defaultLootTableSupplier.get();
					if (defaultLootTable != class_52.field_948) {
						lootTables.add(Pair.of(structureId, defaultLootTable));
					}
				}
			}
		}
		if (lootTables.isEmpty()) {
			class_52 defaultLootTable = defaultLootTableSupplier.get();
			if (defaultLootTable != class_52.field_948) {
				lootTables.add(Pair.of(LooniumStructureConfiguration.DEFAULT_CONFIG_ID, defaultLootTable));
			}
		}
		return lootTables;
	}

	/**
	 * Build a map of structure IDs to resolved Loonium configurations, i.e. no need to traverse any parents.
	 * 
	 * @param configData Configuration data to read from.
	 * @param structures Detected structures to work with.
	 * @return The map, which is guaranteed to not be empty.
	 */
	@NotNull
	private Map<class_2960, LooniumStructureConfiguration> determineStructureConfigs(
			@NotNull ConfigDataManager configData, @NotNull Object2BooleanMap<class_2960> structures) {
		if (configOverride != null) {
			LooniumStructureConfiguration overrideConfig =
					configData.getEffectiveLooniumStructureConfiguration(configOverride);
			return Map.of(LooniumStructureConfiguration.DEFAULT_CONFIG_ID,
					overrideConfig != null ? overrideConfig : getDefaultConfig(configData));
		}

		LooniumStructureConfiguration defaultConfig = getDefaultConfig(configData);
		var structureConfigs = new HashMap<class_2960, LooniumStructureConfiguration>();
		for (Object2BooleanMap.Entry<class_2960> structureEntry : structures.object2BooleanEntrySet()) {
			LooniumStructureConfiguration structureSpecificConfig =
					configData.getEffectiveLooniumStructureConfiguration(structureEntry.getKey());
			LooniumStructureConfiguration structureConfig = structureSpecificConfig != null ? structureSpecificConfig : defaultConfig;
			if (structureConfig != null && (structureEntry.getBooleanValue()
					|| structureConfig.boundingBoxType == class_7061.class_7062.field_37200)) {
				structureConfigs.put(structureEntry.getKey(), structureConfig);
			}
		}

		structureConfigs.put(LooniumStructureConfiguration.DEFAULT_CONFIG_ID, defaultConfig);
		return structureConfigs;
	}

	private static LooniumStructureConfiguration getDefaultConfig(ConfigDataManager configData) {
		LooniumStructureConfiguration defaultConfig = configData.getEffectiveLooniumStructureConfiguration(
				LooniumStructureConfiguration.DEFAULT_CONFIG_ID);
		return defaultConfig != null ? defaultConfig : FALLBACK_CONFIG.get();
	}

	private void detectStructure(class_3218 world) {
		// structure ID and whether the position is inside a structure piece (false = only overall bounding box)
		var structureMap = new Object2BooleanRBTreeMap<class_2960>();
		class_5138 structureManager = world.method_27056();
		class_2338 pos = method_11016();
		Map<class_3195, LongSet> structures = structureManager.method_41037(pos);
		for (Map.Entry<class_3195, LongSet> entry : structures.entrySet()) {
			class_3195 structure = entry.getKey();
			class_3449 start = structureManager.method_28388(pos, structure);
			if (start.method_16657()) {
				class_2960 structureId =
						world.method_30349().method_30530(class_7924.field_41246).method_10221(structure);
				boolean insidePiece = structureManager.method_41033(pos, start);
				if (insidePiece || !structureMap.getBoolean(structureId)) {
					structureMap.put(structureId, insidePiece);
				}
			}
		}

		detectedStructures = new Object2BooleanArrayMap<>(structureMap);

		method_5431();
		sync();
	}

	@Override
	public int getColor() {
		return 0xC29D62;
	}

	@Override
	public int getMaxMana() {
		return LooniumStructureConfiguration.DEFAULT_COST;
	}

	@Override
	public boolean acceptsRedstone() {
		return true;
	}

	@Override
	public RadiusDescriptor getRadius() {
		return RadiusDescriptor.Rectangle.square(getEffectivePos(), RANGE);
	}

	@Override
	public RadiusDescriptor getSecondaryRadius() {
		return RadiusDescriptor.Rectangle.square(getEffectivePos(), CHECK_RANGE);
	}

	@Override
	public void readFromPacketNBT(class_2487 cmp) {
		super.readFromPacketNBT(cmp);
		if (cmp.method_10545(TAG_LOOT_TABLE)) {
			lootTableOverride = new class_2960(cmp.method_10558(TAG_LOOT_TABLE));
		}
		if (cmp.method_10545(TAG_CONFIG_OVERRIDE)) {
			configOverride = new class_2960(cmp.method_10558(TAG_CONFIG_OVERRIDE));
		}
		if (cmp.method_10545(TAG_ATTUNE_DISPLAY_OVERRIDE)) {
			attuneDisplayOverride = cmp.method_10558(TAG_ATTUNE_DISPLAY_OVERRIDE);
		}
		if (cmp.method_10545(TAG_DETECTED_STRUCTURE)) {
			String rawString = cmp.method_10558(TAG_DETECTED_STRUCTURE);
			if (rawString.isEmpty()) {
				detectedStructures = Object2BooleanMaps.emptyMap();
			} else {
				List<ObjectBooleanPair<class_2960>> structureList = Arrays.stream(rawString.split(",")).map(part -> {
					if (part.contains("|")) {
						String[] components = part.split("\\|", 2);
						return ObjectBooleanPair.of(new class_2960(components[0]), Boolean.parseBoolean(components[1]));
					} else {
						return ObjectBooleanPair.of(new class_2960(part), false);
					}
				}).toList();
				// list should never contain more than a few entries, so array is fine and retains entry order
				var map = new Object2BooleanArrayMap<class_2960>(structureList.size());
				structureList.forEach(entry -> map.put(entry.key(), entry.valueBoolean()));
				detectedStructures = map;
			}
		}
	}

	@Override
	public void writeToPacketNBT(class_2487 cmp) {
		super.writeToPacketNBT(cmp);
		if (lootTableOverride != null) {
			cmp.method_10582(TAG_LOOT_TABLE, lootTableOverride.toString());
		}
		if (configOverride != null) {
			cmp.method_10582(TAG_CONFIG_OVERRIDE, configOverride.toString());
		}
		if (attuneDisplayOverride != null) {
			cmp.method_10582(TAG_ATTUNE_DISPLAY_OVERRIDE, attuneDisplayOverride);
		}
		if (detectedStructures != null) {
			var stringBuilder = new StringBuilder();
			boolean first = true;
			for (Object2BooleanMap.Entry<class_2960> entry : detectedStructures.object2BooleanEntrySet()) {
				if (first) {
					first = false;
				} else {
					stringBuilder.append(',');
				}
				stringBuilder.append(entry.getKey()).append('|').append(entry.getBooleanValue());
			}
			cmp.method_10582(TAG_DETECTED_STRUCTURE, stringBuilder.toString());
		}
	}

	public static void dropLooniumItems(class_1309 living, Consumer<class_1799> consumer) {
		LooniumComponent comp = XplatAbstractions.INSTANCE.looniumComponent(living);
		if (comp != null && comp.isOverrideDrop()) {
			consumer.accept(comp.getDrop());
		}
	}

	public static class WandHud extends BindableFlowerWandHud<LooniumBlockEntity> {
		public WandHud(LooniumBlockEntity flower) {
			super(flower);
		}

		@Override
		public void renderHUD(class_332 gui, class_310 mc) {
			String lootType;
			String structureName = "";
			if (flower.attuneDisplayOverride != null) {
				lootType = flower.attuneDisplayOverride;
			} else if (flower.lootTableOverride != null) {
				lootType = "attuned";
			} else if (flower.detectedStructures == null || flower.detectedStructures.isEmpty()) {
				lootType = "not_attuned";
			} else {
				if (flower.detectedStructures.size() == 1) {
					lootType = "attuned_one";
					structureName = flower.detectedStructures
							.keySet().stream().findFirst()
							.map(rl -> class_1074.method_4662("structure." + rl.method_12836() + "." + rl.method_12832().replace("/", ".")))
							.orElseGet(() -> "");
				} else {
					lootType = "attuned_many";
				}
			}

			String lootTypeMessage = class_1074.method_4662("botaniamisc.loonium." + lootType, structureName);
			int lootTypeWidth = mc.field_1772.method_1727(lootTypeMessage);
			int lootTypeTextStart = (mc.method_22683().method_4486() - lootTypeWidth) / 2;
			int halfMinWidth = (lootTypeWidth + 4) / 2;
			int centerY = mc.method_22683().method_4502() / 2;

			super.renderHUD(gui, mc, halfMinWidth, halfMinWidth, 40);
			gui.method_25303(mc.field_1772, lootTypeMessage, lootTypeTextStart, centerY + 30, flower.getColor());
		}
	}
}
