/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.item;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;

import vazkii.botania.api.block.FloatingFlower.IslandType;
import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.common.block.BotaniaBlocks;
import vazkii.botania.common.lib.BotaniaTags;

import java.util.*;
import net.minecraft.class_1269;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1838;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2488;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3486;
import net.minecraft.class_3558;
import net.minecraft.class_5321;

public class GrassSeedsItem extends class_1792 implements FloatingFlowerVariant {
	/**
	 * Represents a map of dimension IDs to a set of all block swappers
	 * active in that dimension.
	 */
	private static final Map<class_5321<class_1937>, Set<BlockSwapper>> blockSwappers = new HashMap<>();
	private static final Map<IslandType, Integer> COLORS = ImmutableMap.<IslandType, Integer>builder()
			.put(IslandType.GRASS, 0x006600)
			.put(IslandType.PODZOL, 0x805E00)
			.put(IslandType.MYCEL, 0x5E0054)
			.put(IslandType.DRY, 0x66800D)
			.put(IslandType.GOLDEN, 0xBFB300)
			.put(IslandType.VIVID, 0x00801A)
			.put(IslandType.SCORCHED, 0xBF0000)
			.put(IslandType.INFUSED, 0x008C8C)
			.put(IslandType.MUTATED, 0x661A66)
			.build();

	private final IslandType type;

	public GrassSeedsItem(IslandType type, class_1793 props) {
		super(props);
		this.type = type;
	}

	@NotNull
	@Override
	public class_1269 method_7884(class_1838 ctx) {
		class_1937 world = ctx.method_8045();
		class_2338 pos = ctx.method_8037();
		class_1799 stack = ctx.method_8041();

		return applySeeds(world, pos, stack);
	}

	public class_1269 applySeeds(class_1937 world, class_2338 pos, class_1799 stack) {
		class_2680 state = world.method_8320(pos);

		if (state.method_26164(BotaniaTags.Blocks.PASTURE_SEED_REPLACEABLE) && state != stateForType(type)) {
			if (!world.field_9236) {
				BlockSwapper swapper = addBlockSwapper(world, pos, type);
				world.method_8501(pos, swapper.stateToSet);
				stack.method_7934(1);
			} else {
				int color = getColor(type);
				spawnParticles(world, pos, extractR(color), extractG(color), extractB(color));
			}

			return class_1269.method_29236(world.method_8608());
		}

		return class_1269.field_5811;
	}

	public static void spawnParticles(class_1937 world, class_2338 pos, float r, float g, float b) {
		for (int i = 0; i < 50; i++) {
			double x = (Math.random() - 0.5) * 3;
			double y = Math.random() - 0.5 + 1;
			double z = (Math.random() - 0.5) * 3;
			float velMul = 0.025F;

			float motionx = (float) -x * velMul;
			float motiony = (float) -y * velMul;
			float motionz = (float) -z * velMul;
			WispParticleData data = WispParticleData.wisp((float) Math.random() * 0.15F + 0.15F, r, g, b);
			world.method_8406(data, pos.method_10263() + 0.5 + x, pos.method_10264() + 0.5 + y, pos.method_10260() + 0.5 + z, motionx, motiony, motionz);
		}
	}

	public static void onTickEnd(class_3218 world) {
		class_5321<class_1937> dim = world.method_27983();
		if (blockSwappers.containsKey(dim)) {
			blockSwappers.get(dim).removeIf(next -> next == null || !next.tick());
		}
	}

	/**
	 * Adds a grass seed block swapper to the world at the provided position
	 * and with the provided meta (which designates the type of the grass
	 * being spread).
	 * Block swappers are only actually created on the server, so a client
	 * calling this method will receive a marker block swapper which contains
	 * the provided information but is not ticked.
	 * 
	 * @param world The world the swapper will be in.
	 * @param pos   The position of the swapper.
	 * @param type  The IslandType of the grass seed
	 * @return The created block swapper.
	 */
	private static BlockSwapper addBlockSwapper(class_1937 world, class_2338 pos, IslandType type) {
		BlockSwapper swapper = new BlockSwapper(world, pos, stateForType(type));

		class_5321<class_1937> dim = world.method_27983();
		blockSwappers.computeIfAbsent(dim, d -> new HashSet<>()).add(swapper);

		return swapper;
	}

	private static class_2680 stateForType(IslandType type) {
		if (type == IslandType.PODZOL) {
			return class_2246.field_10520.method_9564();
		} else if (type == IslandType.MYCEL) {
			return class_2246.field_10402.method_9564();
		} else if (type == IslandType.DRY) {
			return BotaniaBlocks.dryGrass.method_9564();
		} else if (type == IslandType.GOLDEN) {
			return BotaniaBlocks.goldenGrass.method_9564();
		} else if (type == IslandType.VIVID) {
			return BotaniaBlocks.vividGrass.method_9564();
		} else if (type == IslandType.SCORCHED) {
			return BotaniaBlocks.scorchedGrass.method_9564();
		} else if (type == IslandType.INFUSED) {
			return BotaniaBlocks.infusedGrass.method_9564();
		} else if (type == IslandType.MUTATED) {
			return BotaniaBlocks.mutatedGrass.method_9564();
		} else {
			return class_2246.field_10219.method_9564();
		}
	}

	/**
	 * A block swapper for the Pasture Seeds, which swaps dirt and grass blocks
	 * centered around a provided point to a provided block/metadata.
	 */
	private static class BlockSwapper {

		/**
		 * The range of the block swapper, in blocks.
		 */
		public static final int RANGE = 3;

		/**
		 * The horizontal range around which a block can spread in a single tick.
		 */
		public static final int TICK_RANGE_HORIZONTAL = 1;

		/**
		 * The vertical range around which a block can spread in a single tick.
		 */
		public static final int TICK_RANGE_VERTICAL = 2;

		private final class_1937 world;
		private final Random rand;
		private final class_2680 stateToSet;

		private final class_2338 startCoords;
		private int ticksExisted = 0;

		/**
		 * Constructs a new block swapper with the provided world, starting
		 * coordinates, target block, and target metadata.
		 * 
		 * @param world  The world to swap blocks in.
		 * @param coords The central coordinates to swap blocks around.
		 * @param state  The target blockstate to swap dirt and grass to.
		 */
		public BlockSwapper(class_1937 world, class_2338 coords, class_2680 state) {
			this.world = world;
			stateToSet = state;
			rand = new Random(coords.hashCode());
			startCoords = coords;
		}

		/**
		 * Ticks this block swapper, allowing it to make an action during
		 * this game tick. This method should return "false" when the swapper
		 * has finished operation and should be removed from the world.
		 * 
		 * @return true if the swapper should continue to exist, false if it
		 *         should be removed.
		 */
		public boolean tick() {
			if (++ticksExisted % 20 == 0) {
				var tickPositions = new ArrayList<class_2338>();
				for (class_2338 pos : class_2338.method_10097(startCoords.method_10069(-RANGE, -RANGE, -RANGE),
						startCoords.method_10069(RANGE, RANGE, RANGE))) {
					if (world.method_8320(pos) == stateToSet && canPropagate(pos)) {
						tickPositions.add(pos.method_10062());
					}
				}
				tickPositions.forEach(this::tickBlock);
			}

			// This swapper should exist for 80 ticks
			return ticksExisted < 80;
		}

		/**
		 * Tick a specific block position, finding the valid blocks
		 * immediately adjacent to it and then replacing one at random.
		 * 
		 * @param pos The positions to use.
		 */
		public void tickBlock(class_2338 pos) {
			List<class_2338> validCoords = new ArrayList<>();

			// Go around this block and aggregate valid blocks.
			for (class_2338 targetPos : class_2338.method_10097(pos.method_10069(-TICK_RANGE_HORIZONTAL, -TICK_RANGE_VERTICAL, -TICK_RANGE_HORIZONTAL),
					pos.method_10069(TICK_RANGE_HORIZONTAL, TICK_RANGE_VERTICAL, TICK_RANGE_HORIZONTAL))) {
				// Skip the current block, and any blocks that are already converted
				if (targetPos.equals(pos) || world.method_8320(targetPos) == stateToSet) {
					continue;
				}

				if (isValidSwapPosition(targetPos)) {
					validCoords.add(targetPos.method_10062());
				}
			}

			// If we can make changes, and have at least 1 block to swap,
			// then swap a random block from the valid blocks we could swap.
			if (!validCoords.isEmpty()) {
				class_2338 toSwap = validCoords.get(rand.nextInt(validCoords.size()));

				world.method_8501(toSwap, stateToSet);
			}
		}

		/**
		 * Determines if a given position is a valid location to spread to, which
		 * means that the block must be either dirt or grass (with meta 0),
		 * and have a block above it which does not block grass growth.
		 * 
		 * @param pos The position to check.
		 * @return True if the position is valid to swap, false otherwise.
		 */
		public boolean isValidSwapPosition(class_2338 pos) {
			class_2680 state = world.method_8320(pos);
			return state.method_26164(BotaniaTags.Blocks.PASTURE_SEED_REPLACEABLE) && canBeGrass(pos, state);
		}

		// [VanillaCopy] net.minecraft.world.level.block.SpreadingSnowyDirtBlock#canBeGrass
		private boolean canBeGrass(class_2338 pos, class_2680 state) {
			class_2338 abovePos = pos.method_10084();
			class_2680 aboveState = world.method_8320(abovePos);
			if (aboveState.method_27852(class_2246.field_10477) && aboveState.method_11654(class_2488.field_11518) == 1) {
				// single snow layer, okay to spread below that
				return true;
			}
			if (aboveState.method_26227().method_15761() == 8) {
				// full-height liquid, don't spread
				return false;
			}
			int lightLevel = class_3558.method_20049(world, state, pos, aboveState, abovePos, class_2350.field_11036, aboveState.method_26193(world, abovePos));
			return lightLevel < world.method_8315();
		}

		// [VanillaCopy] net.minecraft.world.level.block.SpreadingSnowyDirtBlock#canPropagate
		private boolean canPropagate(class_2338 pos) {
			class_2338 abovePos = pos.method_10084();
			return canBeGrass(pos, stateToSet) && !world.method_8316(abovePos).method_15767(class_3486.field_15517);
		}
	}

	public static float extractR(int color) {
		return ((color >> 16) & 0xFF) / 255f;
	}

	public static float extractG(int color) {
		return ((color >> 8) & 0xFF) / 255f;
	}

	public static float extractB(int color) {
		return (color & 0xFF) / 255f;
	}

	public static int getColor(IslandType type) {
		return COLORS.get(type);
	}

	@Override
	public IslandType getIslandType(class_1799 stack) {
		return type;
	}

}
