/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.api.block_entity;

import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.BotaniaAPIClient;
import vazkii.botania.api.block.Bound;
import vazkii.botania.api.block.WandBindable;
import vazkii.botania.api.block.WandHUD;
import vazkii.botania.client.core.helper.RenderHelper;
import vazkii.botania.common.helper.MathHelper;
import vazkii.botania.common.item.BotaniaItems;

import java.util.Objects;
import net.minecraft.class_1074;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2512;
import net.minecraft.class_2586;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_332;

/**
 * Superclass of flowers that can be bound to some kind of target with the Wand of the Forest,
 * such as generating flowers to mana collectors, or functional flowers to pools.
 * Implements the bindability logic common to both types of flower.
 */
public abstract class BindableSpecialFlowerBlockEntity<T> extends SpecialFlowerBlockEntity implements WandBindable {
	/**
	 * Superclass (or interface) of all BlockEntities that this flower is able to bind to.
	 */
	private final Class<T> bindClass;

	protected @Nullable class_2338 bindingPos = null;
	private static final String TAG_BINDING = "binding";

	public BindableSpecialFlowerBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state, Class<T> bindClass) {
		super(type, pos, state);
		this.bindClass = bindClass;
	}

	public abstract int getBindingRadius();

	/**
	 * Returns the BlockPos of the nearest target within the binding radius, or `null` if there aren't any.
	 */
	public abstract @Nullable class_2338 findClosestTarget();

	@Override
	protected void tickFlower() {
		super.tickFlower();

		//First time the flower has been placed. This is the best time to check it; /setblock and friends don't call
		//the typical setPlacedBy method that player-placements do.
		if (Bound.UNBOUND_POS.equals(bindingPos)) {
			setBindingPos(null);
		} else if (ticksExisted == 1 && !field_11863.field_9236) {
			//Situations to consider:
			// the flower has been placed in the void, and there is nothing for it to bind to;
			// the flower has been placed next to a bind target, and I want to automatically bind to it;
			// the flower already has a valid binding due to ctrl-pick placement, and I should keep it;
			// the flower already has a binding from ctrl-pick placement, but it's invalid (out of range etc) and I should delete it.
			if (bindingPos == null || !isValidBinding()) {
				setBindingPos(findClosestTarget());
			}
		}
	}

	@Override
	public void setPlacedBy(class_1937 level, class_2338 pos, class_2680 state, @Nullable class_1309 placer, class_1799 stack) {
		if (placer != null && placer.method_24518(BotaniaItems.obedienceStick)) {
			setBindingPos(Bound.UNBOUND_POS);
		}
		super.setPlacedBy(level, pos, state, placer, stack);
	}

	public @Nullable class_2338 getBindingPos() {
		return bindingPos;
	}

	public void setBindingPos(@Nullable class_2338 bindingPos) {
		boolean changed = !Objects.equals(this.bindingPos, bindingPos);

		this.bindingPos = bindingPos;

		if (changed) {
			method_5431();
			sync();
		}
	}

	public @Nullable T findBindCandidateAt(class_2338 pos) {
		if (field_11863 == null || pos == null) {
			return null;
		}

		class_2586 be = field_11863.method_8321(pos);
		return bindClass.isInstance(be) ? bindClass.cast(be) : null;
	}

	public @Nullable T findBoundTile() {
		return findBindCandidateAt(bindingPos);
	}

	public boolean wouldBeValidBinding(@Nullable class_2338 pos) {
		if (field_11863 == null || pos == null || !field_11863.method_8477(pos) || MathHelper.distSqr(method_11016(), pos) > (long) getBindingRadius() * getBindingRadius()) {
			return false;
		} else {
			return findBindCandidateAt(pos) != null;
		}
	}

	public boolean isValidBinding() {
		return wouldBeValidBinding(bindingPos);
	}

	@Override
	public class_2338 getBinding() {
		//Used for Wand of the Forest overlays; only return the binding if it's valid.
		return isValidBinding() ? bindingPos : null;
	}

	@Override
	public boolean canSelect(class_1657 player, class_1799 wand, class_2338 pos, class_2350 side) {
		return true;
	}

	@Override
	public boolean bindTo(class_1657 player, class_1799 wand, class_2338 pos, class_2350 side) {
		if (wouldBeValidBinding(pos)) {
			setBindingPos(pos);
			return true;
		}

		return false;
	}

	@Override
	public void writeToPacketNBT(class_2487 cmp) {
		super.writeToPacketNBT(cmp);

		if (bindingPos != null) {
			cmp.method_10566(TAG_BINDING, class_2512.method_10692(bindingPos));
		}
	}

	@Override
	public void readFromPacketNBT(class_2487 cmp) {
		super.readFromPacketNBT(cmp);

		if (cmp.method_10545(TAG_BINDING)) {
			bindingPos = class_2512.method_10691(cmp.method_10562(TAG_BINDING));
		} else {
			//In older versions of the mod (1.16, early 1.17), GeneratingFlowerBlockEntity and SpecialFlowerBlockEntity
			//implemented their own copies of the binding logic. Read data from the old locations.
			if (cmp.method_10545("collectorX")) {
				bindingPos = new class_2338(cmp.method_10550("collectorX"), cmp.method_10550("collectorY"), cmp.method_10550("collectorZ"));
			} else if (cmp.method_10545("poolX")) {
				bindingPos = new class_2338(cmp.method_10550("poolX"), cmp.method_10550("poolY"), cmp.method_10550("poolZ"));
			}
			//These versions of the mod also sometimes used a binding with a Y of -1 to signify an unbound flower.
			//Currently, `null` is always used for unbound flowers. Coerce these positions to `null`.
			if (bindingPos != null && bindingPos.method_10264() == -1) {
				bindingPos = null;
			}
		}
	}

	public abstract int getMana();

	public abstract void addMana(int mana);

	public abstract int getMaxMana();

	public abstract int getColor();

	public abstract class_1799 getDefaultHudIcon();

	public class_1799 getHudIcon() {
		T boundTile = findBoundTile();
		if (boundTile != null) {
			return new class_1799(((class_2586) boundTile).method_11010().method_26204().method_8389());
		}
		return getDefaultHudIcon();
	}

	public static class BindableFlowerWandHud<F extends BindableSpecialFlowerBlockEntity<?>> implements WandHUD {
		protected final F flower;

		public BindableFlowerWandHud(F flower) {
			this.flower = flower;
		}

		public void renderHUD(class_332 gui, class_310 mc, int minLeft, int minRight, int minDown) {
			String name = class_1074.method_4662(flower.method_11010().method_26204().method_9539());
			int color = flower.getColor();

			int centerX = mc.method_22683().method_4486() / 2;
			int centerY = mc.method_22683().method_4502() / 2;
			int left = (Math.max(102, mc.field_1772.method_1727(name)) + 4) / 2;
			// padding + item
			int right = left + 20;

			left = Math.max(left, minLeft);
			right = Math.max(right, minRight);

			RenderHelper.renderHUDBox(gui, centerX - left, centerY + 8, centerX + right, centerY + Math.max(30, minDown));

			BotaniaAPIClient.instance().drawComplexManaHUD(gui, color, flower.getMana(), flower.getMaxMana(),
					name, flower.getHudIcon(), flower.isValidBinding());
		}

		@Override
		public void renderHUD(class_332 gui, class_310 mc) {
			renderHUD(gui, mc, 0, 0, 0);
		}
	}
}
