/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.api.brew;

import com.google.common.collect.ImmutableList;
import vazkii.botania.api.BotaniaAPI;

import java.util.List;
import java.util.function.Supplier;
import net.minecraft.class_1293;
import net.minecraft.class_1799;
import net.minecraft.class_1844;
import net.minecraft.class_2960;

/**
 * The class for a Brew definition, each one is a singleton.
 */
public class Brew {
	private final Supplier<Integer> color;
	private final int cost;
	private final Supplier<List<class_1293>> effects;
	private boolean canInfuseBloodPendant = true;
	private boolean canInfuseIncense = true;

	/**
	 * @param color   The color for the potion to be rendered in the bottle, note that it will get
	 *                changed a bit when it renders (for more or less brightness) to give a fancy effect.
	 *                See {@link net.minecraft.class_1844#method_8064} for a method
	 *                to calculate this automatically.
	 * @param cost    The cost, in Mana for this brew.
	 * @param effects A list of effects to apply to the player when they drink it.
	 */
	public Brew(int color, int cost, class_1293... effects) {
		this.color = () -> color;
		this.cost = cost;
		List<class_1293> savedEffects = ImmutableList.copyOf(effects);
		this.effects = () -> savedEffects;
	}

	/**
	 * @param cost    The cost, in Mana for this brew.
	 * @param effects A supplier that supplies a list of effects to apply to the player when they drink it.
	 */
	public Brew(int cost, Supplier<List<class_1293>> effects) {
		this.color = () -> class_1844.method_8055(effects.get());
		this.cost = cost;
		this.effects = effects;
	}

	/**
	 * Sets this brew to not be able to be infused onto the Tainted Blood Pendant.
	 */
	public Brew setNotBloodPendantInfusable() {
		canInfuseBloodPendant = false;
		return this;
	}

	/**
	 * Sets this brew to not be able to be infused onto Incense Sticks.
	 */
	public Brew setNotIncenseInfusable() {
		canInfuseIncense = false;
		return this;
	}

	public boolean canInfuseBloodPendant() {
		return canInfuseBloodPendant;
	}

	public boolean canInfuseIncense() {
		return canInfuseIncense;
	}

	/**
	 * Gets the insensitive unlocalized name. This is used for the lexicon.
	 */
	public String getTranslationKey() {
		class_2960 id = BotaniaAPI.instance().getBrewRegistry().method_10221(this);
		return String.format("%s.brew.%s", id.method_12836(), id.method_12832());
	}

	/**
	 * Gets the unlocalized name for the ItemStack passed in.
	 */
	public String getTranslationKey(class_1799 stack) {
		return getTranslationKey();
	}

	/**
	 * Gets the display color for the ItemStack passed in. Note that for
	 * the lexicon, this passes in a botania Managlass Vial or an
	 * Alfglass Flask at all times.
	 */
	public int getColor(class_1799 stack) {
		return color.get();
	}

	/**
	 * Gets the insensitive unlocalized mana cost. This is used for the lexicon.
	 */
	public int getManaCost() {
		return cost;
	}

	/**
	 * Gets the mana cost for the ItemStack passed in.
	 */
	public int getManaCost(class_1799 stack) {
		return getManaCost();
	}

	/**
	 * Gets the list of vanilla PotionEffects for the ItemStack passed in.
	 * Note that for the lexicon, this passes in a botania Managlass
	 * Vial or an Alfglass Flask at all times.
	 */
	public List<class_1293> getPotionEffects(class_1799 stack) {
		return effects.get();
	}

}
