/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.item;

import com.mojang.blaze3d.systems.RenderSystem;
import org.jetbrains.annotations.NotNull;

import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.common.handler.BotaniaSounds;
import vazkii.botania.common.helper.ItemNBTHelper;
import vazkii.botania.common.helper.MathHelper;
import vazkii.botania.common.helper.VecHelper;
import vazkii.botania.common.item.equipment.tool.ToolCommons;
import vazkii.botania.common.proxy.Proxy;
import vazkii.patchouli.api.IMultiblock;
import vazkii.patchouli.api.IStateMatcher;
import vazkii.patchouli.api.PatchouliAPI;

import java.math.RoundingMode;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;
import net.minecraft.class_124;
import net.minecraft.class_1268;
import net.minecraft.class_1271;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1839;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_239;
import net.minecraft.class_243;
import net.minecraft.class_2470;
import net.minecraft.class_2561;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_3965;
import net.minecraft.class_4587;
import net.minecraft.class_5328;

import static vazkii.botania.common.lib.ResourceLocationHelper.prefix;

public class WorldshaperssSextantItem extends class_1792 {
	public static final class_2960 MULTIBLOCK_ID = prefix("sextant");
	private static final int MAX_RADIUS = 256;
	private static final String TAG_SOURCE_X = "sourceX";
	private static final String TAG_SOURCE_Y = "sourceY";
	private static final String TAG_SOURCE_Z = "sourceZ";
	private static final String TAG_MODE = "mode";

	public WorldshaperssSextantItem(class_1793 builder) {
		super(builder);
	}

	@NotNull
	@Override
	public class_1839 method_7853(class_1799 stack) {
		return class_1839.field_8953;
	}

	@Override
	public int method_7881(class_1799 stack) {
		return 72000;
	}

	@Override
	public void method_7852(class_1937 world, class_1309 living, class_1799 stack, int count) {
		if (method_7881(stack) - count < 10
				|| !(living instanceof class_1657)
				|| !world.field_9236) {
			return;
		}

		int x = ItemNBTHelper.getInt(stack, TAG_SOURCE_X, 0);
		int y = ItemNBTHelper.getInt(stack, TAG_SOURCE_Y, Integer.MIN_VALUE);
		int z = ItemNBTHelper.getInt(stack, TAG_SOURCE_Z, 0);
		if (y != Integer.MIN_VALUE) {
			double radius = calculateRadius(stack, living);
			WispParticleData data = WispParticleData.wisp(0.3F, 0F, 1F, 1F, 1);
			world.method_8406(data, x + 0.5, y + 1, z + 0.5, 0, 0.1, 0);
			var visualizer = getMode(stack).getVisualizer();
			for (int i = count % 20; i < 360; i += 20) {
				float radian = (float) (i * Math.PI / 180);
				double cosR = Math.cos(radian) * radius;
				double sinR = Math.sin(radian) * radius;
				visualizer.visualize(world, x, y, z, data, cosR, sinR);
			}
		}
	}

	private static void visualizeSphere(class_1937 world, int x, int y, int z, WispParticleData data, double cosR, double sinR) {
		world.method_8406(data, x + cosR + 0.5, y + 1.3, z + sinR + 0.5, 0, 0.01, 0);
		world.method_8406(data, x + sinR + 0.5, y + cosR + 1.5, z + 0.3, 0, 0, 0.01);
		world.method_8406(data, x + 0.3, y + sinR + 1.5, z + cosR + 0.5, 0.01, 0, 0);
	}

	private static void visualizeCircle(class_1937 world, int x, int y, int z, WispParticleData data, double cosR, double sinR) {
		world.method_8406(data, x + cosR + 0.5, y + 1, z + sinR + 0.5, 0, 0.01, 0);
	}

	private static void makeSphere(IStateMatcher matcher, double radius, Map<class_2338, IStateMatcher> map) {
		// 3D version of Midpoint circle algorithm, based on https://stackoverflow.com/a/41666156/1331011
		// This algorithm generates all combinations of X, Y, and Z components, where:
		// - the X/Y/Z position is inside the sphere,
		// - Z has the greatest (or tied for greatest) value of the three components,
		// - making Z any larger would place the position outside the sphere, and
		// - X, Y, and Z are all positive or zero.
		final int maxR2 = (int) Math.floor(radius * radius);
		int zMax = (int) Math.floor(radius);
		for (int x = 0;; x++) {
			while (x * x + zMax * zMax > maxR2 && zMax >= x) {
				zMax--;
			}
			if (zMax < x) {
				break; // with this x, z can't be largest
			}
			int z = zMax;
			for (int y = 0;; y++) {
				while (x * x + y * y + z * z > maxR2 && z >= x && z >= y) {
					z--;
				}
				if (z < x || z < y) {
					break; // with this x and y, z can't be largest
				}
				// By rotating the components and mirroring the resulting positions to the other seven octants,
				// each set of values generates up to 24 blocks of the sphere.
				generateMirroredPositions(x, y, z, map, matcher);
				generateMirroredPositions(y, z, x, map, matcher);
				generateMirroredPositions(z, x, y, map, matcher);
			}
		}
	}

	private static void generateMirroredPositions(int x, int y, int z, Map<class_2338, IStateMatcher> map, IStateMatcher matcher) {
		Stream.of(
				new class_2338(x, y, z), new class_2338(-x, y, z),
				new class_2338(x, -y, z), new class_2338(-x, -y, z),
				new class_2338(x, y, -z), new class_2338(-x, y, -z),
				new class_2338(x, -y, -z), new class_2338(-x, -y, -z)
		).forEach(pos -> map.put(pos, matcher));
	}

	@Override
	public void method_7840(class_1799 stack, class_1937 world, class_1309 living, int time) {
		if (!(living instanceof class_1657)) {
			return;
		}

		double radius = calculateRadius(stack, living);
		if (1 < radius && radius <= MAX_RADIUS) {
			IStateMatcher matcher = PatchouliAPI.get().predicateMatcher(class_2246.field_10445, s -> !s.method_26215());
			int x = ItemNBTHelper.getInt(stack, TAG_SOURCE_X, 0);
			int y = ItemNBTHelper.getInt(stack, TAG_SOURCE_Y, Integer.MIN_VALUE);
			int z = ItemNBTHelper.getInt(stack, TAG_SOURCE_Z, 0);
			if (y != Integer.MIN_VALUE) {
				Map<class_2338, IStateMatcher> map = new HashMap<>();
				getMode(stack).getCreator().create(matcher, radius + 0.5, map);
				IMultiblock sparse = PatchouliAPI.get().makeSparseMultiblock(map).setId(MULTIBLOCK_ID);
				Proxy.INSTANCE.showMultiblock(sparse, class_2561.method_43470("r = " + getRadiusString(radius)),
						new class_2338(x, y, z), class_2470.field_11467);
			}
		}
	}

	private static void makeCircle(IStateMatcher matcher, double radius, Map<class_2338, IStateMatcher> map) {
		// 2D version of makeSphere, assuming y=0 at all times
		final int maxR2 = (int) Math.floor(radius * radius);
		int z = (int) Math.floor(radius);
		for (int x = 0;; x++) {
			while (x * x + z * z > maxR2 && z >= x) {
				z--;
			}
			if (z < x) {
				break;
			}
			generateMirroredPositions(x, z, map, matcher);
			generateMirroredPositions(z, x, map, matcher);
		}
	}

	private static void generateMirroredPositions(int x, int z, Map<class_2338, IStateMatcher> map, IStateMatcher matcher) {
		Stream.of(
				new class_2338(x, 0, z), new class_2338(-x, 0, z),
				new class_2338(x, 0, -z), new class_2338(-x, 0, -z)
		).forEach(pos -> map.put(pos, matcher));
	}

	private static Modes getMode(class_1799 stack) {
		String modeString = ItemNBTHelper.getString(stack, TAG_MODE, "circle");
		return Arrays.stream(Modes.values()).filter(m -> m.getKey().equals(modeString)).findFirst().orElse(Modes.CIRCLE);
	}

	private void reset(class_1937 world, class_1657 player, class_1799 stack) {
		if (ItemNBTHelper.getInt(stack, TAG_SOURCE_Y, Integer.MIN_VALUE) == Integer.MIN_VALUE) {
			if (!world.field_9236) {
				Modes currentMode = getMode(stack);
				int numModes = Modes.values().length;
				int nextMode = currentMode.ordinal() + 1;
				setMode(stack, Modes.values()[nextMode >= numModes ? 0 : nextMode]);
			} else {
				player.method_5783(BotaniaSounds.ding, 0.1F, 1F);
			}
		} else {
			ItemNBTHelper.setInt(stack, TAG_SOURCE_Y, Integer.MIN_VALUE);
		}
		if (world.field_9236) {
			Proxy.INSTANCE.clearSextantMultiblock();
		}
	}

	private static void setMode(class_1799 stack, Modes mode) {
		ItemNBTHelper.setString(stack, TAG_MODE, mode.getKey());
	}

	@NotNull
	@Override
	public class_1271<class_1799> method_7836(class_1937 world, class_1657 player, @NotNull class_1268 hand) {
		class_1799 stack = player.method_5998(hand);
		if (!player.method_21823()) {
			class_3965 rtr = ToolCommons.raytraceFromEntity(player, 128, false);
			if (rtr.method_17783() == class_239.class_240.field_1332) {
				if (!world.field_9236) {
					class_2338 pos = rtr.method_17777();
					ItemNBTHelper.setInt(stack, TAG_SOURCE_X, pos.method_10263());
					ItemNBTHelper.setInt(stack, TAG_SOURCE_Y, pos.method_10264());
					ItemNBTHelper.setInt(stack, TAG_SOURCE_Z, pos.method_10260());
				}
				return class_5328.method_29282(world, player, hand);
			}
			return class_1271.method_22430(stack);
		} else {
			reset(world, player, stack);
			return class_1271.method_22427(stack);
		}
	}

	private static double calculateRadius(class_1799 stack, class_1309 living) {
		int x = ItemNBTHelper.getInt(stack, TAG_SOURCE_X, 0);
		int y = ItemNBTHelper.getInt(stack, TAG_SOURCE_Y, Integer.MIN_VALUE);
		int z = ItemNBTHelper.getInt(stack, TAG_SOURCE_Z, 0);
		class_243 source = new class_243(x, y, z);

		class_243 centerVec = VecHelper.fromEntityCenter(living);
		class_243 diffVec = source.method_1020(centerVec);
		class_243 lookVec = living.method_5720();
		double mul = diffVec.field_1351 / lookVec.field_1351;
		lookVec = lookVec.method_1021(mul).method_1019(centerVec);

		lookVec = new class_243(net.minecraft.class_3532.method_15357(lookVec.field_1352),
				net.minecraft.class_3532.method_15357(lookVec.field_1351),
				net.minecraft.class_3532.method_15357(lookVec.field_1350));

		return MathHelper.pointDistancePlane(source.field_1352, source.field_1350, lookVec.field_1352, lookVec.field_1350);
	}

	@Override
	public class_2561 method_7864(@NotNull class_1799 stack) {
		class_2561 mode = class_2561.method_43470(" (")
				.method_10852(class_2561.method_43471(getModeString(stack)))
				.method_27693(")");
		return super.method_7864(stack).method_27662().method_10852(mode);
	}

	public static String getModeString(class_1799 stack) {
		return "botaniamisc.sextantMode." + getMode(stack).getKey();
	}

	private static String getRadiusString(double radius) {
		NumberFormat format = getNumberFormat();

		return format.format(radius);
	}

	private static NumberFormat getNumberFormat() {
		var format = NumberFormat.getInstance(Proxy.INSTANCE.getLocale());
		format.setRoundingMode(RoundingMode.HALF_UP);
		format.setMaximumFractionDigits(1);
		format.setMinimumFractionDigits(1);
		return format;
	}

	public static class Hud {
		public static void render(class_332 gui, class_1657 player, class_1799 stack) {
			class_4587 ms = gui.method_51448();
			class_1799 onUse = player.method_6030();
			int time = player.method_6014();

			if (onUse == stack && stack.method_7909().method_7881(stack) - time >= 10) {
				double radius = calculateRadius(stack, player);
				class_327 font = class_310.method_1551().field_1772;
				int x = class_310.method_1551().method_22683().method_4486() / 2 + 30;
				int y = class_310.method_1551().method_22683().method_4502() / 2;

				String s = getRadiusString(radius);
				boolean inRange = 0 < radius && radius <= MAX_RADIUS;
				if (!inRange) {
					s = class_124.field_1061 + s;
				}

				gui.method_25303(font, s, x - font.method_1727(s) / 2, y - 4, 0xFFFFFF);

				if (inRange) {
					radius += 4;
					RenderSystem.lineWidth(3F);
					class_289.method_1348().method_1349().method_1328(class_293.class_5596.field_27378, class_290.field_1592);
					RenderSystem.setShaderColor(0F, 1F, 1F, 1F);
					for (int i = 0; i < 361; i++) {
						float radian = (float) (i * Math.PI / 180);
						float xp = x + net.minecraft.class_3532.method_15362(radian) * (float) radius;
						float yp = y + net.minecraft.class_3532.method_15374(radian) * (float) radius;
						class_289.method_1348().method_1349().method_22918(ms.method_23760().method_23761(), xp, yp, 0).method_1344();
					}
					class_289.method_1348().method_1350();
				}
			}
		}
	}

	@FunctionalInterface
	private interface ShapeCreator {
		void create(IStateMatcher matcher, double radius, Map<class_2338, IStateMatcher> map);
	}

	@FunctionalInterface
	private interface ShapeVisualizer {
		void visualize(class_1937 world, int x, int y, int z, WispParticleData data, double cosR, double sinR);
	}

	public enum Modes {
		CIRCLE("circle", WorldshaperssSextantItem::makeCircle, WorldshaperssSextantItem::visualizeCircle),
		SPHERE("sphere", WorldshaperssSextantItem::makeSphere, WorldshaperssSextantItem::visualizeSphere);

		private final String key;
		private final ShapeCreator creator;
		private final ShapeVisualizer visualizer;

		Modes(String key, ShapeCreator creator, ShapeVisualizer visualizer) {
			this.key = key;
			this.creator = creator;
			this.visualizer = visualizer;
		}

		public String getKey() {
			return key;
		}

		public ShapeCreator getCreator() {
			return creator;
		}

		public ShapeVisualizer getVisualizer() {
			return visualizer;
		}
	}
}
