/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.handler;

import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.internal.ManaNetwork;
import vazkii.botania.api.mana.*;
import vazkii.botania.common.helper.MathHelper;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.*;
import net.minecraft.class_1937;
import net.minecraft.class_2338;

public final class ManaNetworkHandler implements ManaNetwork {

	public static final ManaNetworkHandler instance = new ManaNetworkHandler();

	private final Map<class_1937, Set<ManaPool>> manaPools = new WeakHashMap<>();
	private final Map<class_1937, Set<ManaCollector>> manaCollectors = new WeakHashMap<>();

	public void onNetworkEvent(ManaReceiver thing, ManaBlockType type, ManaNetworkAction action) {
		switch (type) {
			case COLLECTOR -> {
				if (action == ManaNetworkAction.ADD) {
					add(manaCollectors, thing.getManaReceiverLevel(), (ManaCollector) thing);
				} else {
					remove(manaCollectors, thing.getManaReceiverLevel(), (ManaCollector) thing);
				}
			}
			case POOL -> {
				if (action == ManaNetworkAction.ADD) {
					add(manaPools, thing.getManaReceiverLevel(), (ManaPool) thing);
				} else {
					remove(manaPools, thing.getManaReceiverLevel(), (ManaPool) thing);
				}
			}
		}
	}

	@Override
	public void clear() {
		manaPools.clear();
		manaCollectors.clear();
	}

	@Override
	public ManaPool getClosestPool(class_2338 pos, class_1937 world, int limit) {
		if (manaPools.containsKey(world)) {
			return getClosest(manaPools.get(world), pos, limit);
		}
		return null;
	}

	@Override
	public ManaCollector getClosestCollector(class_2338 pos, class_1937 world, int limit) {
		if (manaCollectors.containsKey(world)) {
			return getClosest(manaCollectors.get(world), pos, limit);
		}
		return null;
	}

	public boolean isCollectorIn(class_1937 level, ManaCollector collector) {
		return manaCollectors.getOrDefault(level, Collections.emptySet()).contains(collector);
	}

	public boolean isPoolIn(class_1937 level, ManaPool pool) {
		return manaPools.getOrDefault(level, Collections.emptySet()).contains(pool);
	}

	@Nullable
	private <T extends ManaReceiver> T getClosest(Set<T> receivers, class_2338 pos, int limit) {
		long minDist = Long.MAX_VALUE;
		long limitSquared = (long) limit * limit;
		T closest = null;

		for (var receiver : receivers) {
			long distance = MathHelper.distSqr(receiver.getManaReceiverPos(), pos);
			if (distance <= limitSquared && distance < minDist) {
				minDist = distance;
				closest = receiver;
			}
		}

		return closest;
	}

	private <T> void remove(Map<class_1937, Set<T>> map, class_1937 level, T thing) {
		if (!map.containsKey(level)) {
			return;
		}

		var set = map.get(level);
		set.remove(thing);
		if (set.isEmpty()) {
			map.remove(level);
		}
	}

	private <T> void add(Map<class_1937, Set<T>> map, class_1937 level, T thing) {
		map.computeIfAbsent(level, k -> new HashSet<>()).add(thing);
	}

	@Override
	public Set<ManaCollector> getAllCollectorsInWorld(class_1937 world) {
		return getAllInWorld(manaCollectors, world);
	}

	@Override
	public Set<ManaPool> getAllPoolsInWorld(class_1937 world) {
		return getAllInWorld(manaPools, world);
	}

	@Override
	public void fireManaNetworkEvent(ManaReceiver thing, ManaBlockType type, ManaNetworkAction action) {
		XplatAbstractions.INSTANCE.fireManaNetworkEvent(thing, type, action);
	}

	private <T> Set<T> getAllInWorld(Map<class_1937, Set<T>> map, class_1937 world) {
		Set<T> ret = map.get(world);
		if (ret == null) {
			return Collections.emptySet();
		} else {
			return Collections.unmodifiableSet(ret);
		}
	}

}
