/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.entity;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.internal.ManaBurst;
import vazkii.botania.api.internal.VanillaPacketDispatcher;
import vazkii.botania.api.mana.*;
import vazkii.botania.api.mana.ManaCollisionGhost.Behaviour;
import vazkii.botania.client.fx.SparkleParticleData;
import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.common.block.block_entity.mana.ThrottledPacket;
import vazkii.botania.common.item.equipment.bauble.ManaseerMonocleItem;
import vazkii.botania.common.proxy.Proxy;
import vazkii.botania.xplat.BotaniaConfig;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.*;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_1657;
import net.minecraft.class_1682;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2261;
import net.minecraft.class_2338;
import net.minecraft.class_239;
import net.minecraft.class_2397;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2940;
import net.minecraft.class_2943;
import net.minecraft.class_2945;
import net.minecraft.class_3532;
import net.minecraft.class_3611;
import net.minecraft.class_3965;
import net.minecraft.class_3966;
import net.minecraft.class_6024;
import net.minecraft.class_6862;

public class ManaBurstEntity extends class_1682 implements ManaBurst {
	private static final String TAG_TICKS_EXISTED = "ticksExisted";
	private static final String TAG_COLOR = "color";
	private static final String TAG_MANA = "mana";
	private static final String TAG_STARTING_MANA = "startingMana";
	private static final String TAG_MIN_MANA_LOSS = "minManaLoss";
	private static final String TAG_TICK_MANA_LOSS = "manaLossTick";
	private static final String TAG_SPREADER_X = "spreaderX";
	private static final String TAG_SPREADER_Y = "spreaderY";
	private static final String TAG_SPREADER_Z = "spreaderZ";
	private static final String TAG_GRAVITY = "gravity";
	private static final String TAG_LENS_STACK = "lensStack";
	private static final String TAG_HAS_SHOOTER = "hasShooter";
	private static final String TAG_SHOOTER = "shooterUUID";
	private static final String TAG_LAST_COLLISION_X = "lastCollisionX";
	private static final String TAG_LAST_COLLISION_Y = "lastCollisionY";
	private static final String TAG_LAST_COLLISION_Z = "lastCollisionZ";
	private static final String TAG_WARPED = "warped";
	private static final String TAG_ORBIT_TIME = "orbitTime";
	private static final String TAG_TRIPPED = "tripped";
	private static final String TAG_MAGNETIZE_POS = "magnetizePos";
	private static final String TAG_LEFT_SOURCE = "leftSource";
	private static final String TAG_ALREADY_COLLIDED_AT = "alreadyCollidedAt";

	private static final class_2940<Integer> COLOR = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13327);
	private static final class_2940<Integer> MANA = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13327);
	private static final class_2940<Integer> START_MANA = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13327);
	private static final class_2940<Integer> MIN_MANA_LOSS = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13327);
	private static final class_2940<Float> MANA_LOSS_PER_TICK = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13320);
	private static final class_2940<Float> GRAVITY = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13320);
	private static final class_2940<class_2338> SOURCE_COORDS = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13324);
	private static final class_2940<class_1799> SOURCE_LENS = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13322);
	private static final class_2940<Boolean> LEFT_SOURCE_POS = class_2945.method_12791(ManaBurstEntity.class, class_2943.field_13323);

	private float accumulatedManaLoss = 0;
	private boolean fake = false;
	private final Set<class_2338> alreadyCollidedAt = new HashSet<>();
	private boolean fullManaLastTick = true;
	private UUID shooterIdentity = null;
	private int _ticksExisted = 0;
	private boolean scanBeam = false;
	private class_2338 lastCollision;
	private boolean warped = false;
	private int orbitTime = 0;
	private boolean tripped = false;
	private class_2338 magnetizePos = null;

	public final List<PositionProperties> propsList = new ArrayList<>();

	public ManaBurstEntity(class_1299<ManaBurstEntity> type, class_1937 world) {
		super(type, world);
	}

	@Override
	protected void method_5693() {
		field_6011.method_12784(COLOR, 0);
		field_6011.method_12784(MANA, 0);
		field_6011.method_12784(START_MANA, 0);
		field_6011.method_12784(MIN_MANA_LOSS, 0);
		field_6011.method_12784(MANA_LOSS_PER_TICK, 0F);
		field_6011.method_12784(GRAVITY, 0F);
		field_6011.method_12784(SOURCE_COORDS, ManaBurst.NO_SOURCE);
		field_6011.method_12784(SOURCE_LENS, class_1799.field_8037);
		field_6011.method_12784(LEFT_SOURCE_POS, false);
	}

	public static class_243 calculateBurstVelocity(float xRot, float yRot) {
		float f = 0.4F;
		double mx = class_3532.method_15374(yRot / 180.0F * (float) Math.PI) * class_3532.method_15362(xRot / 180.0F * (float) Math.PI) * f / 2D;
		double mz = -(class_3532.method_15362(yRot / 180.0F * (float) Math.PI) * class_3532.method_15362(xRot / 180.0F * (float) Math.PI) * f) / 2D;
		double my = class_3532.method_15374(xRot / 180.0F * (float) Math.PI) * f / 2D;
		return new class_243(mx, my, mz);
	}

	public ManaBurstEntity(class_1937 level, class_2338 pos, float rotX, float rotY, boolean fake) {
		this(BotaniaEntities.MANA_BURST, level);

		this.fake = fake;

		setBurstSourceCoords(pos);
		method_5808(pos.method_10263() + 0.5, pos.method_10264() + 0.5, pos.method_10260() + 0.5, 0, 0);
		/* NB: this looks backwards but it's right. spreaders take rotX/rotY to respectively mean
		* "rotation *parallel* to the X and Y axes", while vanilla's methods take XRot/YRot
		* to respectively mean "rotation *around* the X and Y axes".
		* TODO consider renaming our versions to match vanilla
		*/
		method_36456(-(rotX + 90F));
		method_36457(rotY);
		method_18799(calculateBurstVelocity(method_36455(), method_36454()));
	}

	public ManaBurstEntity(class_1657 player) {
		super(BotaniaEntities.MANA_BURST, player, player.field_6002);

		setBurstSourceCoords(NO_SOURCE);
		method_5710(player.method_36454() + 180, -player.method_36455());
		method_18799(calculateBurstVelocity(method_36455(), method_36454()));
	}

	@Override
	public void method_5773() {
		setTicksExisted(getTicksExisted() + 1);
		if ((!field_6002.field_9236 || fake)
				&& !hasLeftSource()
				&& !method_24515().equals(getBurstSourceBlockPos())) {
			// XXX: Should this check by bounding box instead of simply blockPosition()?
			// The burst's origin could be in another coord but part of its box still intersecting the source block
			// Not sure if that will trigger a collision then
			field_6011.method_12778(LEFT_SOURCE_POS, true);
		}

		super.method_5773();

		if (!fake && method_5805() && !scanBeam) {
			ping();
		}

		LensEffectItem lens = getLensInstance();
		if (lens != null) {
			lens.updateBurst(this, getSourceLens());
		}

		int mana = getMana();
		if (getTicksExisted() >= getMinManaLoss()) {
			accumulatedManaLoss += getManaLossPerTick();
			int loss = (int) accumulatedManaLoss;
			setMana(mana - loss);
			accumulatedManaLoss -= loss;

			if (getMana() <= 0) {
				method_31472();
			}
		}

		particles();

		fullManaLastTick = getMana() == getStartingMana();

		if (scanBeam) {
			PositionProperties props = PositionProperties.fromEntity(this);
			if (propsList.isEmpty()) {
				propsList.add(props);
			} else {
				PositionProperties lastProps = propsList.get(propsList.size() - 1);
				if (!props.coordsEqual(lastProps)) {
					propsList.add(props);
				}
			}
		}
	}

	@Override
	public boolean method_5692(class_6862<class_3611> fluid, double mag) {
		return false;
	}

	@Override
	public boolean method_5771() {
		return false;
	}

	private ManaReceiver collidedTile = null;
	private boolean noParticles = false;

	public ManaReceiver getCollidedTile(boolean noParticles) {
		this.noParticles = noParticles;

		int iterations = 0;
		while (method_5805() && iterations < BotaniaConfig.common().spreaderTraceTime()) {
			method_5773();
			iterations++;
		}

		if (fake) {
			incrementFakeParticleTick();
		}

		return collidedTile;
	}

	@Override
	public boolean method_5822() {
		return !fake;
	}

	@Override
	public void method_5652(class_2487 tag) {
		super.method_5652(tag);
		if (fake) {
			var msg = String.format("Fake bursts should never be saved at any time! Source pos %s, owner %s",
					getBurstSourceBlockPos(), method_24921());
			throw new IllegalStateException(msg);
		}
		tag.method_10569(TAG_TICKS_EXISTED, getTicksExisted());
		tag.method_10569(TAG_COLOR, getColor());
		tag.method_10569(TAG_MANA, getMana());
		tag.method_10569(TAG_STARTING_MANA, getStartingMana());
		tag.method_10569(TAG_MIN_MANA_LOSS, getMinManaLoss());
		tag.method_10548(TAG_TICK_MANA_LOSS, getManaLossPerTick());
		tag.method_10548(TAG_GRAVITY, getBurstGravity());

		class_1799 stack = getSourceLens();
		class_2487 lensCmp = new class_2487();
		if (!stack.method_7960()) {
			lensCmp = stack.method_7953(lensCmp);
		}
		tag.method_10566(TAG_LENS_STACK, lensCmp);

		class_2338 coords = getBurstSourceBlockPos();
		tag.method_10569(TAG_SPREADER_X, coords.method_10263());
		tag.method_10569(TAG_SPREADER_Y, coords.method_10264());
		tag.method_10569(TAG_SPREADER_Z, coords.method_10260());

		if (lastCollision != null) {
			tag.method_10569(TAG_LAST_COLLISION_X, lastCollision.method_10263());
			tag.method_10569(TAG_LAST_COLLISION_Y, lastCollision.method_10264());
			tag.method_10569(TAG_LAST_COLLISION_Z, lastCollision.method_10260());
		}

		UUID identity = getShooterUUID();
		boolean hasShooter = identity != null;
		tag.method_10556(TAG_HAS_SHOOTER, hasShooter);
		if (hasShooter) {
			tag.method_25927(TAG_SHOOTER, identity);
		}
		tag.method_10556(TAG_WARPED, warped);
		tag.method_10569(TAG_ORBIT_TIME, orbitTime);
		tag.method_10556(TAG_TRIPPED, tripped);
		if (magnetizePos != null) {
			tag.method_10566(TAG_MAGNETIZE_POS, class_2338.field_25064.encodeStart(class_2509.field_11560, magnetizePos).get().orThrow());
		}
		tag.method_10556(TAG_LEFT_SOURCE, hasLeftSource());

		var alreadyCollidedAt = new class_2499();
		for (class_2338 pos : this.alreadyCollidedAt) {
			alreadyCollidedAt.add(class_2338.field_25064.encodeStart(class_2509.field_11560, pos).get().orThrow());
		}
		tag.method_10566(TAG_ALREADY_COLLIDED_AT, alreadyCollidedAt);
	}

	@Override
	public void method_5749(class_2487 cmp) {
		super.method_5749(cmp);
		setTicksExisted(cmp.method_10550(TAG_TICKS_EXISTED));
		setColor(cmp.method_10550(TAG_COLOR));
		setMana(cmp.method_10550(TAG_MANA));
		setStartingMana(cmp.method_10550(TAG_STARTING_MANA));
		setMinManaLoss(cmp.method_10550(TAG_MIN_MANA_LOSS));
		setManaLossPerTick(cmp.method_10583(TAG_TICK_MANA_LOSS));
		setGravity(cmp.method_10583(TAG_GRAVITY));

		class_2487 lensCmp = cmp.method_10562(TAG_LENS_STACK);
		class_1799 stack = class_1799.method_7915(lensCmp);
		if (!stack.method_7960()) {
			setSourceLens(stack);
		} else {
			setSourceLens(class_1799.field_8037);
		}

		int x = cmp.method_10550(TAG_SPREADER_X);
		int y = cmp.method_10550(TAG_SPREADER_Y);
		int z = cmp.method_10550(TAG_SPREADER_Z);

		setBurstSourceCoords(new class_2338(x, y, z));

		if (cmp.method_10545(TAG_LAST_COLLISION_X)) {
			x = cmp.method_10550(TAG_LAST_COLLISION_X);
			y = cmp.method_10550(TAG_LAST_COLLISION_Y);
			z = cmp.method_10550(TAG_LAST_COLLISION_Z);
			lastCollision = new class_2338(x, y, z);
		}

		// Reread Motion because Entity.load clamps it to +/-10
		class_2499 motion = cmp.method_10554("Motion", class_2520.field_33256);
		method_18800(motion.method_10611(0), motion.method_10611(1), motion.method_10611(2));

		boolean hasShooter = cmp.method_10577(TAG_HAS_SHOOTER);
		if (hasShooter) {
			UUID serializedUuid = cmp.method_25926(TAG_SHOOTER);
			UUID identity = getShooterUUID();
			if (!serializedUuid.equals(identity)) {
				setShooterUUID(serializedUuid);
			}
		}
		warped = cmp.method_10577(TAG_WARPED);
		orbitTime = cmp.method_10550(TAG_ORBIT_TIME);
		tripped = cmp.method_10577(TAG_TRIPPED);
		if (cmp.method_10545(TAG_MAGNETIZE_POS)) {
			magnetizePos = class_2338.field_25064.parse(class_2509.field_11560, cmp.method_10580(TAG_MAGNETIZE_POS)).get().orThrow();
		} else {
			magnetizePos = null;
		}
		field_6011.method_12778(LEFT_SOURCE_POS, cmp.method_10577(TAG_LEFT_SOURCE));

		this.alreadyCollidedAt.clear();
		for (var tag : cmp.method_10554(TAG_ALREADY_COLLIDED_AT, class_2520.field_33261)) {
			var pos = class_2338.field_25064.parse(class_2509.field_11560, tag).result();
			pos.ifPresent(this.alreadyCollidedAt::add);
		}
	}

	public void particles() {
		if (!method_5805() || !field_6002.field_9236) {
			return;
		}

		LensEffectItem lens = getLensInstance();
		if (lens != null && !lens.doParticles(this, getSourceLens())) {
			return;
		}

		int color = getColor();
		float r = (color >> 16 & 0xFF) / 255F;
		float g = (color >> 8 & 0xFF) / 255F;
		float b = (color & 0xFF) / 255F;
		float osize = getParticleSize();
		float size = osize;

		if (fake) {
			if (getMana() == getStartingMana()) {
				size = 2F;
			} else if (fullManaLastTick) {
				size = 4F;
			}

			if (!noParticles && shouldDoFakeParticles()) {
				SparkleParticleData data = SparkleParticleData.fake(0.4F * size, r, g, b, 1);
				field_6002.method_8466(data, true, method_23317(), method_23318(), method_23321(), 0, 0, 0);
			}
		} else {
			class_1657 player = Proxy.INSTANCE.getClientPlayer();
			boolean depth = player == null || !ManaseerMonocleItem.hasMonocle(player);

			if (BotaniaConfig.client().subtlePowerSystem()) {
				WispParticleData data = WispParticleData.wisp(0.1F * size, r, g, b, depth);
				Proxy.INSTANCE.addParticleForceNear(field_6002, data, method_23317(), method_23318(), method_23321(), (float) (Math.random() - 0.5F) * 0.02F, (float) (Math.random() - 0.5F) * 0.02F, (float) (Math.random() - 0.5F) * 0.01F);
			} else {
				float or = r;
				float og = g;
				float ob = b;

				double luminance = 0.2126 * r + 0.7152 * g + 0.0722 * b; // Standard relative luminance calculation

				double iterX = method_23317();
				double iterY = method_23318();
				double iterZ = method_23321();

				class_243 currentPos = method_19538();
				class_243 oldPos = new class_243(field_6014, field_6036, field_5969);
				class_243 diffVec = oldPos.method_1020(currentPos);
				class_243 diffVecNorm = diffVec.method_1029();

				double distance = 0.095;

				do {
					if (luminance < 0.1) {
						r = or + (float) Math.random() * 0.125F;
						g = og + (float) Math.random() * 0.125F;
						b = ob + (float) Math.random() * 0.125F;
					}
					size = osize + ((float) Math.random() - 0.5F) * 0.065F + (float) Math.sin(new Random(field_6021.getMostSignificantBits()).nextInt(9001)) * 0.4F;
					WispParticleData data = WispParticleData.wisp(0.2F * size, r, g, b, depth);
					Proxy.INSTANCE.addParticleForceNear(field_6002, data, iterX, iterY, iterZ,
							(float) -method_18798().method_10216() * 0.01F,
							(float) -method_18798().method_10214() * 0.01F,
							(float) -method_18798().method_10215() * 0.01F);

					iterX += diffVecNorm.field_1352 * distance;
					iterY += diffVecNorm.field_1351 * distance;
					iterZ += diffVecNorm.field_1350 * distance;

					currentPos = new class_243(iterX, iterY, iterZ);
					diffVec = oldPos.method_1020(currentPos);
					if (getOrbitTime() > 0) {
						break;
					}
				} while (Math.abs(diffVec.method_1033()) > distance);

				WispParticleData data = WispParticleData.wisp(0.1F * size, or, og, ob, depth);
				field_6002.method_8406(data, iterX, iterY, iterZ, (float) (Math.random() - 0.5F) * 0.06F, (float) (Math.random() - 0.5F) * 0.06F, (float) (Math.random() - 0.5F) * 0.06F);
			}
		}
	}

	@Override
	public void method_5711(byte event) {
		if (event == class_6024.field_30028) {
			int color = getColor();
			float r = (color >> 16 & 0xFF) / 255F;
			float g = (color >> 8 & 0xFF) / 255F;
			float b = (color & 0xFF) / 255F;

			int mana = getMana();
			int maxMana = getStartingMana();
			float size = (float) mana / (float) maxMana;

			if (!BotaniaConfig.client().subtlePowerSystem()) {
				for (int i = 0; i < 4; i++) {
					WispParticleData data = WispParticleData.wisp(0.15F * size, r, g, b);
					field_6002.method_8406(data, method_23317(), method_23318(), method_23321(), (float) (Math.random() - 0.5F) * 0.04F, (float) (Math.random() - 0.5F) * 0.04F, (float) (Math.random() - 0.5F) * 0.04F);
				}
			}
			SparkleParticleData data = SparkleParticleData.sparkle((float) 4, r, g, b, 2);
			field_6002.method_8406(data, method_23317(), method_23318(), method_23321(), 0, 0, 0);
		} else {
			super.method_5711(event);
		}
	}

	public float getParticleSize() {
		return (float) getMana() / (float) getStartingMana();
	}

	@Override
	protected void method_24920(@NotNull class_3965 hit) {
		if (!isFake()) {
			super.method_24920(hit);
		}
		class_2338 collidePos = hit.method_17777();
		if (collidePos.equals(lastCollision)) {
			return;
		}
		lastCollision = collidePos.method_10062();
		class_2586 tile = field_6002.method_8321(collidePos);
		class_2680 state = field_6002.method_8320(collidePos);
		class_2248 block = state.method_26204();

		var ghost = XplatAbstractions.INSTANCE.findManaGhost(field_6002, collidePos, state, tile);
		var ghostBehaviour = ghost != null ? ghost.getGhostBehaviour() : ManaCollisionGhost.Behaviour.RUN_ALL;

		if (ghostBehaviour == ManaCollisionGhost.Behaviour.SKIP_ALL
				|| block instanceof class_2261
				|| block instanceof class_2397) {
			return;
		}

		class_2338 sourcePos = getBurstSourceBlockPos();
		if (!hasLeftSource() && collidePos.equals(sourcePos)) {
			return;
		}

		var receiver = XplatAbstractions.INSTANCE.findManaReceiver(field_6002, collidePos, state, tile, hit.method_17780());
		collidedTile = receiver;

		if (!fake && !noParticles && !field_6002.field_9236) {
			if (receiver != null && receiver.canReceiveManaFromBursts() && onReceiverImpact(receiver)) {
				if (tile instanceof ThrottledPacket throttledPacket) {
					throttledPacket.markDispatchable();
				} else if (tile != null) {
					VanillaPacketDispatcher.dispatchTEToNearbyPlayers(tile);
				}
			}
		}

		var trigger = XplatAbstractions.INSTANCE.findManaTrigger(field_6002, collidePos, state, tile);
		if (trigger != null) {
			trigger.onBurstCollision(this);
		}

		if (ghostBehaviour == ManaCollisionGhost.Behaviour.RUN_RECEIVER_TRIGGER) {
			return;
		}

		onHitCommon(hit, true);

		setCollidedAt(collidePos);
	}

	@Override
	protected void method_7454(@NotNull class_3966 hit) {
		super.method_7454(hit);
		onHitCommon(hit, false);
	}

	private void onHitCommon(class_239 hit, boolean shouldKill) {
		LensEffectItem lens = getLensInstance();
		if (lens != null) {
			shouldKill = lens.collideBurst(this, hit, collidedTile != null
					&& collidedTile.canReceiveManaFromBursts(), shouldKill, getSourceLens());
		}

		if (shouldKill && method_5805()) {
			if (fake) {
				method_31472();
			} else if (!this.field_6002.field_9236) {
				this.field_6002.method_8421(this, class_6024.field_30028);
				method_31472();
			}
		}
	}

	private boolean onReceiverImpact(ManaReceiver receiver) {
		if (hasWarped()) {
			return false;
		}

		LensEffectItem lens = getLensInstance();
		int mana = getMana();

		if (lens != null) {
			class_1799 stack = getSourceLens();
			mana = lens.getManaToTransfer(this, stack, receiver);
		}

		if (receiver instanceof ManaCollector collector) {
			mana *= collector.getManaYieldMultiplier(this);
		}

		if (mana > 0) {
			receiver.receiveMana(mana);
			return true;
		}

		return false;
	}

	@Override
	public void method_5650(class_5529 reason) {
		super.method_5650(reason);

		if (!fake) {
			var spreader = getShooter();
			if (spreader != null) {
				spreader.setCanShoot(true);
			}
		} else {
			setDeathTicksForFakeParticle();
		}
	}

	@Nullable
	private ManaSpreader getShooter() {
		var receiver = XplatAbstractions.INSTANCE.findManaReceiver(field_6002, getBurstSourceBlockPos(), null);
		return receiver instanceof ManaSpreader spreader ? spreader : null;
	}

	@Override
	public float method_7490() {
		return getBurstGravity();
	}

	@Override
	public boolean isFake() {
		return fake;
	}

	@Override
	public void setFake(boolean fake) {
		this.fake = fake;
	}

	public void setScanBeam() {
		scanBeam = true;
	}

	@Override
	public int getColor() {
		return field_6011.method_12789(COLOR);
	}

	@Override
	public void setColor(int color) {
		field_6011.method_12778(COLOR, color);
	}

	@Override
	public int getMana() {
		return field_6011.method_12789(MANA);
	}

	@Override
	public void setMana(int mana) {
		field_6011.method_12778(MANA, mana);
	}

	@Override
	public int getStartingMana() {
		return field_6011.method_12789(START_MANA);
	}

	@Override
	public void setStartingMana(int mana) {
		field_6011.method_12778(START_MANA, mana);
	}

	@Override
	public int getMinManaLoss() {
		return field_6011.method_12789(MIN_MANA_LOSS);
	}

	@Override
	public void setMinManaLoss(int minManaLoss) {
		field_6011.method_12778(MIN_MANA_LOSS, minManaLoss);
	}

	@Override
	public float getManaLossPerTick() {
		return field_6011.method_12789(MANA_LOSS_PER_TICK);
	}

	@Override
	public void setManaLossPerTick(float mana) {
		field_6011.method_12778(MANA_LOSS_PER_TICK, mana);
	}

	@Override
	public float getBurstGravity() {
		return field_6011.method_12789(GRAVITY);
	}

	@Override
	public void setGravity(float gravity) {
		field_6011.method_12778(GRAVITY, gravity);
	}

	@Override
	public class_2338 getBurstSourceBlockPos() {
		return field_6011.method_12789(SOURCE_COORDS);
	}

	@Override
	public void setBurstSourceCoords(class_2338 pos) {
		field_6011.method_12778(SOURCE_COORDS, pos);
	}

	@Override
	public class_1799 getSourceLens() {
		return field_6011.method_12789(SOURCE_LENS);
	}

	@Override
	public void setSourceLens(class_1799 lens) {
		field_6011.method_12778(SOURCE_LENS, lens);
	}

	@Override
	public int getTicksExisted() {
		return _ticksExisted;
	}

	@Override
	public boolean hasLeftSource() {
		return field_6011.method_12789(LEFT_SOURCE_POS);
	}

	public void setTicksExisted(int ticks) {
		_ticksExisted = ticks;
	}

	private LensEffectItem getLensInstance() {
		class_1799 lens = getSourceLens();
		if (!lens.method_7960() && lens.method_7909() instanceof LensEffectItem effect) {
			return effect;
		}

		return null;
	}

	@Override
	public boolean hasAlreadyCollidedAt(class_2338 pos) {
		return alreadyCollidedAt.contains(pos);
	}

	@Override
	public void setCollidedAt(class_2338 pos) {
		if (!hasAlreadyCollidedAt(pos)) {
			alreadyCollidedAt.add(pos.method_10062());
		}
	}

	@Override
	public void setShooterUUID(UUID uuid) {
		shooterIdentity = uuid;
	}

	@Override
	public UUID getShooterUUID() {
		return shooterIdentity;
	}

	@Override
	public void ping() {
		var spreader = getShooter();
		if (spreader != null) {
			spreader.pingback(this, getShooterUUID());
		}
	}

	@Override
	public boolean hasWarped() {
		return warped;
	}

	@Override
	public void setWarped(boolean warped) {
		this.warped = warped;
	}

	@Override
	public int getOrbitTime() {
		return orbitTime;
	}

	@Override
	public void setOrbitTime(int time) {
		this.orbitTime = time;
	}

	@Override
	public boolean hasTripped() {
		return tripped;
	}

	@Override
	public void setTripped(boolean tripped) {
		this.tripped = tripped;
	}

	@Nullable
	@Override
	public class_2338 getMagnetizedPos() {
		return magnetizePos;
	}

	@Override
	public void setMagnetizePos(@Nullable class_2338 pos) {
		this.magnetizePos = pos;
	}

	protected boolean shouldDoFakeParticles() {
		if (BotaniaConfig.client().staticWandBeam()) {
			return true;
		}

		var spreader = getShooter();
		return spreader != null
				&& (getMana() != getStartingMana() && fullManaLastTick
						|| Math.abs(spreader.getBurstParticleTick() - getTicksExisted()) < 4);
	}

	private void incrementFakeParticleTick() {
		var spreader = getShooter();
		if (spreader != null) {
			spreader.setBurstParticleTick(spreader.getBurstParticleTick() + 2);
			if (spreader.getLastBurstDeathTick() != -1 && spreader.getBurstParticleTick() > spreader.getLastBurstDeathTick()) {
				spreader.setBurstParticleTick(0);
			}
		}
	}

	private void setDeathTicksForFakeParticle() {
		var spreader = getShooter();
		if (spreader != null) {
			spreader.setLastBurstDeathTick(getTicksExisted());
		}
	}

	public record PositionProperties(class_2338 coords, class_2680 state) {
		public static PositionProperties fromEntity(class_1297 entity) {
			return new PositionProperties(entity.method_24515(), entity.method_36601());
		}

		public boolean coordsEqual(PositionProperties props) {
			return coords.equals(props.coords);
		}

		public boolean isInvalidIn(class_1937 level) {
			return !level.method_22340(coords);
		}

		public boolean contentsEqual(class_1937 world) {
			if (isInvalidIn(world)) {
				return false;
			}

			return world.method_8320(coords) == state;
		}
	}

}
