/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.flower.functional;

import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.block.Wandable;
import vazkii.botania.api.block_entity.FunctionalFlowerBlockEntity;
import vazkii.botania.api.block_entity.RadiusDescriptor;
import vazkii.botania.common.block.BotaniaFlowerBlocks;
import vazkii.botania.common.helper.DelayHelper;
import vazkii.botania.common.helper.EntityHelper;
import vazkii.botania.common.helper.InventoryHelper;
import vazkii.botania.common.helper.ItemNBTHelper;
import vazkii.botania.common.internal_caps.ItemFlagsComponent;
import vazkii.botania.xplat.XplatAbstractions;
import I;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Predicate;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_1533;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2281;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_2487;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_2745;
import net.minecraft.class_310;
import net.minecraft.class_4587;

public class HopperhockBlockEntity extends FunctionalFlowerBlockEntity implements Wandable {
	private static final String TAG_FILTER_TYPE = "filterType";
	private static final int RANGE_MANA = 10;
	private static final int RANGE = 6;

	private static final int RANGE_MANA_MINI = 2;
	private static final int RANGE_MINI = 1;

	private int filterType = 0;

	protected HopperhockBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state) {
		super(type, pos, state);
	}

	public HopperhockBlockEntity(class_2338 pos, class_2680 state) {
		this(BotaniaFlowerBlocks.HOPPERHOCK, pos, state);
	}

	@Override
	public void tickFlower() {
		super.tickFlower();

		if (method_10997().field_9236 || redstoneSignal > 0) {
			return;
		}

		boolean pulledAny = false;
		int range = getRange();

		class_2338 inPos = method_11016();
		class_2338 outPos = getEffectivePos();

		Predicate<class_1542> shouldPickup = item -> {
			if (XplatAbstractions.INSTANCE.preventsRemoteMovement(item)) {
				return false;
			}

			// Flat 5 tick delay for newly infused items
			var manaInfusionCooldown = XplatAbstractions.INSTANCE.itemFlagsComponent(item).getManaInfusionCooldown();
			if (manaInfusionCooldown > 0) {
				return manaInfusionCooldown <= ItemFlagsComponent.INITIAL_MANA_INFUSION_COOLDOWN - 5;
			}

			return DelayHelper.canInteractWith(this, item);
		};
		List<class_1542> items = method_10997().method_8390(class_1542.class, new class_238(inPos.method_10069(-range, -range, -range), inPos.method_10069(range + 1, range + 1, range + 1)), shouldPickup);

		for (class_1542 item : items) {
			class_1799 stack = item.method_6983();
			boolean priorityInv = false;
			int amountToPutIn = 0;
			class_2350 direction = null;

			for (class_2350 dir : class_2350.values()) {
				class_2338 inventoryPos = outPos.method_10093(dir);
				class_2350 sideOfInventory = dir.method_10153();

				if (XplatAbstractions.INSTANCE.hasInventory(field_11863, inventoryPos, sideOfInventory)) {
					List<class_1799> filter = getFilterForInventory(method_10997(), inventoryPos, true);
					boolean canAccept = canAcceptItem(stack, filter, filterType);

					class_1799 simulate = XplatAbstractions.INSTANCE.insertToInventory(field_11863, inventoryPos, sideOfInventory, stack, true);
					int inserted = stack.method_7947() - simulate.method_7947();

					canAccept = canAccept && inserted > 0;

					if (canAccept) {
						boolean priority = !filter.isEmpty();

						if (!priorityInv || priority) {
							priorityInv = priority;
							amountToPutIn = inserted;
							direction = dir;
						}
					}
				}
			}

			if (direction != null && item.method_5805()) {
				SpectranthemumBlockEntity.spawnExplosionParticles(item, 3);
				InventoryHelper.checkEmpty(
						XplatAbstractions.INSTANCE.insertToInventory(field_11863, outPos.method_10093(direction),
								direction.method_10153(), stack.method_7971(amountToPutIn), false)
				);

				EntityHelper.syncItem(item);
				pulledAny = true;
			}
		}

		if (pulledAny && getMana() > 0) {
			addMana(-1);
		}
	}

	public static boolean canAcceptItem(class_1799 stack, List<class_1799> filter, int filterType) {
		if (stack.method_7960()) {
			return false;
		}

		if (filter.isEmpty()) {
			return true;
		}

		switch (filterType) {
			case 0: { // Accept items in frames only
				boolean anyFilter = false;
				for (class_1799 filterEntry : filter) {
					if (filterEntry == null || filterEntry.method_7960()) {
						continue;
					}
					anyFilter = true;

					if (ItemNBTHelper.matchTagAndManaFullness(stack, filterEntry)) {
						return true;
					}
				}

				return !anyFilter;
			}
			case 1:
				return !canAcceptItem(stack, filter, 0); // Accept items not in frames only
			default:
				return true; // Accept all items
		}
	}

	public static List<class_1799> getFilterForInventory(class_1937 level, class_2338 pos, boolean recursiveForDoubleChests) {
		List<class_1799> filter = new ArrayList<>();

		if (recursiveForDoubleChests) {
			class_2680 chest = level.method_8320(pos);

			if (chest.method_28498(class_2281.field_10770)) {
				class_2745 type = chest.method_11654(class_2281.field_10770);
				if (type != class_2745.field_12569) {
					class_2338 other = pos.method_10093(class_2281.method_9758(chest));
					if (level.method_8320(other).method_27852(chest.method_26204())) {
						filter.addAll(getFilterForInventory(level, other, false));
					}
				}
			}
		}

		for (class_2350 dir : class_2350.values()) {
			class_238 aabb = new class_238(pos.method_10093(dir));
			List<class_1533> frames = level.method_18467(class_1533.class, aabb);
			for (class_1533 frame : frames) {
				if (frame.method_5735() == dir) {
					filter.add(frame.method_6940());
				}
			}
		}

		return filter;
	}

	@Override
	public boolean acceptsRedstone() {
		return true;
	}

	@Override
	public boolean onUsedByWand(@Nullable class_1657 player, class_1799 wand, class_2350 side) {
		if (player == null || player.method_5715()) {
			filterType = filterType == 2 ? 0 : filterType + 1;
			sync();

			return true;
		}
		return false;
	}

	@Override
	public RadiusDescriptor getRadius() {
		return RadiusDescriptor.Rectangle.square(getEffectivePos(), getRange());
	}

	@Override
	public RadiusDescriptor getSecondaryRadius() {
		return RadiusDescriptor.Rectangle.square(method_11016(), 1);
	}

	public int getRange() {
		return getMana() > 0 ? RANGE_MANA : RANGE;
	}

	@Override
	public void writeToPacketNBT(class_2487 cmp) {
		super.writeToPacketNBT(cmp);

		cmp.method_10569(TAG_FILTER_TYPE, filterType);
	}

	@Override
	public void readFromPacketNBT(class_2487 cmp) {
		super.readFromPacketNBT(cmp);

		filterType = cmp.method_10550(TAG_FILTER_TYPE);
	}

	public static class WandHud extends FunctionalWandHud<HopperhockBlockEntity> {
		public WandHud(HopperhockBlockEntity flower) {
			super(flower);
		}

		@Override
		public void renderHUD(class_4587 ms, class_310 mc) {
			super.renderHUD(ms, mc);

			String filter = class_1074.method_4662("botaniamisc.filter" + flower.filterType);
			int x = mc.method_22683().method_4486() / 2 - mc.field_1772.method_1727(filter) / 2;
			int y = mc.method_22683().method_4502() / 2 + 30;

			mc.field_1772.method_1720(ms, filter, x, y, class_124.field_1080.method_532());
		}
	}

	@Override
	public int getMaxMana() {
		return 20;
	}

	@Override
	public int getColor() {
		return 0x3F3F3F;
	}

	public static class Mini extends HopperhockBlockEntity {
		public Mini(class_2338 pos, class_2680 state) {
			super(BotaniaFlowerBlocks.HOPPERHOCK_CHIBI, pos, state);
		}

		@Override
		public int getRange() {
			return getMana() > 0 ? RANGE_MANA_MINI : RANGE_MINI;
		}
	}
}
