/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.block_entity.mana;

import com.google.common.base.Predicates;
import com.mojang.blaze3d.systems.RenderSystem;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.BotaniaAPIClient;
import vazkii.botania.api.block.WandBindable;
import vazkii.botania.api.block.WandHUD;
import vazkii.botania.api.block.Wandable;
import vazkii.botania.api.internal.ManaBurst;
import vazkii.botania.api.internal.VanillaPacketDispatcher;
import vazkii.botania.api.mana.*;
import vazkii.botania.common.block.block_entity.BotaniaBlockEntities;
import vazkii.botania.common.block.block_entity.ExposedSimpleInventoryBlockEntity;
import vazkii.botania.common.block.mana.ManaSpreaderBlock;
import vazkii.botania.common.entity.ManaBurstEntity;
import vazkii.botania.common.entity.ManaBurstEntity.PositionProperties;
import vazkii.botania.common.handler.BotaniaSounds;
import vazkii.botania.common.handler.ManaNetworkHandler;
import vazkii.botania.common.helper.MathHelper;
import vazkii.botania.common.item.LexicaBotaniaItem;
import vazkii.botania.xplat.BotaniaConfig;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.List;
import java.util.UUID;
import net.minecraft.class_1277;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1682;
import net.minecraft.class_1767;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_2415;
import net.minecraft.class_243;
import net.minecraft.class_2470;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_265;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_3419;
import net.minecraft.class_3959;
import net.minecraft.class_3965;
import net.minecraft.class_4587;

public class ManaSpreaderBlockEntity extends ExposedSimpleInventoryBlockEntity implements WandBindable, KeyLocked, ThrottledPacket, ManaSpreader, Wandable {
	private static final int TICKS_ALLOWED_WITHOUT_PINGBACK = 20;
	private static final double PINGBACK_EXPIRED_SEARCH_DISTANCE = 0.5;

	private static final String TAG_UUID = "uuid";
	private static final String TAG_MANA = "mana";
	private static final String TAG_REQUEST_UPDATE = "requestUpdate";
	private static final String TAG_ROTATION_X = "rotationX";
	private static final String TAG_ROTATION_Y = "rotationY";
	private static final String TAG_PADDING_COLOR = "paddingColor";
	private static final String TAG_CAN_SHOOT_BURST = "canShootBurst";
	private static final String TAG_PINGBACK_TICKS = "pingbackTicks";
	private static final String TAG_LAST_PINGBACK_X = "lastPingbackX";
	private static final String TAG_LAST_PINGBACK_Y = "lastPingbackY";
	private static final String TAG_LAST_PINGBACK_Z = "lastPingbackZ";

	private static final String TAG_FORCE_CLIENT_BINDING_X = "forceClientBindingX";
	private static final String TAG_FORCE_CLIENT_BINDING_Y = "forceClientBindingY";
	private static final String TAG_FORCE_CLIENT_BINDING_Z = "forceClientBindingZ";

	// Map Maker Tags

	private static final String TAG_INPUT_KEY = "inputKey";
	private static final String TAG_OUTPUT_KEY = "outputKey";

	private static final String TAG_MAPMAKER_OVERRIDE = "mapmakerOverrideEnabled";
	private static final String TAG_FORCED_COLOR = "mmForcedColor";
	private static final String TAG_FORCED_MANA_PAYLOAD = "mmForcedManaPayload";
	private static final String TAG_FORCED_TICKS_BEFORE_MANA_LOSS = "mmForcedTicksBeforeManaLoss";
	private static final String TAG_FORCED_MANA_LOSS_PER_TICK = "mmForcedManaLossPerTick";
	private static final String TAG_FORCED_GRAVITY = "mmForcedGravity";
	private static final String TAG_FORCED_VELOCITY_MULTIPLIER = "mmForcedVelocityMultiplier";

	private boolean mapmakerOverride = false;
	private int mmForcedColor = 0x20FF20;
	private int mmForcedManaPayload = 160;
	private int mmForcedTicksBeforeManaLoss = 60;
	private float mmForcedManaLossPerTick = 4F;
	private float mmForcedGravity = 0F;
	private float mmForcedVelocityMultiplier = 1F;

	private String inputKey = "";
	private final String outputKey = "";

	// End Map Maker Tags

	private UUID identity = UUID.randomUUID();

	private int mana;
	public float rotationX, rotationY;

	@Nullable
	public class_1767 paddingColor = null;

	private boolean requestsClientUpdate = false;
	private boolean hasReceivedInitialPacket = false;

	private ManaReceiver receiver = null;
	private ManaReceiver receiverLastTick = null;

	private boolean poweredLastTick = true;
	public boolean canShootBurst = true;
	public int lastBurstDeathTick = -1;
	public int burstParticleTick = 0;

	public int pingbackTicks = 0;
	public double lastPingbackX = 0;
	public double lastPingbackY = Integer.MIN_VALUE;
	public double lastPingbackZ = 0;

	private List<PositionProperties> lastTentativeBurst;
	private boolean invalidTentativeBurst = false;

	public ManaSpreaderBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaBlockEntities.SPREADER, pos, state);
	}

	@Override
	public boolean isFull() {
		return mana >= getMaxMana();
	}

	@Override
	public void receiveMana(int mana) {
		this.mana = Math.min(this.mana + mana, getMaxMana());
		this.method_5431();
	}

	@Override
	public void method_11012() {
		super.method_11012();
		BotaniaAPI.instance().getManaNetworkInstance().fireManaNetworkEvent(this, ManaBlockType.COLLECTOR, ManaNetworkAction.REMOVE);
	}

	public static void commonTick(class_1937 level, class_2338 worldPosition, class_2680 state, ManaSpreaderBlockEntity self) {
		boolean inNetwork = ManaNetworkHandler.instance.isCollectorIn(level, self);
		boolean wasInNetwork = inNetwork;
		if (!inNetwork && !self.method_11015()) {
			BotaniaAPI.instance().getManaNetworkInstance().fireManaNetworkEvent(self, ManaBlockType.COLLECTOR, ManaNetworkAction.ADD);
		}

		boolean powered = false;

		for (class_2350 dir : class_2350.values()) {
			var relPos = worldPosition.method_10093(dir);
			if (level.method_22340(relPos)) {
				var receiverAt = XplatAbstractions.INSTANCE.findManaReceiver(level, relPos,
						level.method_8320(relPos), level.method_8321(relPos), dir.method_10153());
				if (receiverAt instanceof ManaPool pool) {
					if (wasInNetwork && (pool != self.receiver || self.getVariant() == ManaSpreaderBlock.Variant.REDSTONE)) {
						if (pool instanceof KeyLocked locked && !locked.getOutputKey().equals(self.getInputKey())) {
							continue;
						}

						int manaInPool = pool.getCurrentMana();
						if (manaInPool > 0 && !self.isFull()) {
							int manaMissing = self.getMaxMana() - self.mana;
							int manaToRemove = Math.min(manaInPool, manaMissing);
							pool.receiveMana(-manaToRemove);
							self.receiveMana(manaToRemove);
						}
					}
				}
				powered = powered || level.method_8459(relPos, dir);
			}
		}

		if (self.needsNewBurstSimulation()) {
			self.checkForReceiver();
		}

		if (!self.canShootBurst) {
			if (self.pingbackTicks <= 0) {
				double x = self.lastPingbackX;
				double y = self.lastPingbackY;
				double z = self.lastPingbackZ;
				class_238 aabb = new class_238(x, y, z, x, y, z).method_1009(PINGBACK_EXPIRED_SEARCH_DISTANCE, PINGBACK_EXPIRED_SEARCH_DISTANCE, PINGBACK_EXPIRED_SEARCH_DISTANCE);
				@SuppressWarnings("unchecked")
				List<ManaBurst> bursts = (List<ManaBurst>) (List<?>) level.method_8390(class_1682.class, aabb, Predicates.instanceOf(ManaBurst.class));
				ManaBurst found = null;
				UUID identity = self.getIdentifier();
				for (ManaBurst burst : bursts) {
					if (burst != null && identity.equals(burst.getShooterUUID())) {
						found = burst;
						break;
					}
				}

				if (found != null) {
					found.ping();
				} else {
					self.setCanShoot(true);
				}
			} else {
				self.pingbackTicks--;
			}
		}

		boolean shouldShoot = !powered;

		boolean redstoneSpreader = self.getVariant() == ManaSpreaderBlock.Variant.REDSTONE;
		if (redstoneSpreader) {
			shouldShoot = powered && !self.poweredLastTick;
		}

		if (shouldShoot && self.receiver instanceof KeyLocked locked) {
			shouldShoot = locked.getInputKey().equals(self.getOutputKey());
		}

		class_1799 lens = self.getItemHandler().method_5438(0);
		ControlLensItem control = self.getLensController(lens);
		if (control != null) {
			if (redstoneSpreader) {
				if (shouldShoot) {
					control.onControlledSpreaderPulse(lens, self);
				}
			} else {
				control.onControlledSpreaderTick(lens, self, powered);
			}

			shouldShoot = shouldShoot && control.allowBurstShooting(lens, self, powered);
		}

		if (shouldShoot) {
			self.tryShootBurst();
		}

		if (self.receiverLastTick != self.receiver && !level.field_9236) {
			self.requestsClientUpdate = true;
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(self);
		}

		self.poweredLastTick = powered;
		self.receiverLastTick = self.receiver;
	}

	@Override
	public void writePacketNBT(class_2487 cmp) {
		super.writePacketNBT(cmp);

		cmp.method_25927(TAG_UUID, getIdentifier());

		cmp.method_10569(TAG_MANA, mana);
		cmp.method_10548(TAG_ROTATION_X, rotationX);
		cmp.method_10548(TAG_ROTATION_Y, rotationY);
		cmp.method_10556(TAG_REQUEST_UPDATE, requestsClientUpdate);
		cmp.method_10569(TAG_PADDING_COLOR, paddingColor == null ? -1 : paddingColor.method_7789());
		cmp.method_10556(TAG_CAN_SHOOT_BURST, canShootBurst);

		cmp.method_10569(TAG_PINGBACK_TICKS, pingbackTicks);
		cmp.method_10549(TAG_LAST_PINGBACK_X, lastPingbackX);
		cmp.method_10549(TAG_LAST_PINGBACK_Y, lastPingbackY);
		cmp.method_10549(TAG_LAST_PINGBACK_Z, lastPingbackZ);

		cmp.method_10582(TAG_INPUT_KEY, inputKey);
		cmp.method_10582(TAG_OUTPUT_KEY, outputKey);

		cmp.method_10569(TAG_FORCE_CLIENT_BINDING_X, receiver == null ? 0 : receiver.getManaReceiverPos().method_10263());
		cmp.method_10569(TAG_FORCE_CLIENT_BINDING_Y, receiver == null ? Integer.MIN_VALUE : receiver.getManaReceiverPos().method_10264());
		cmp.method_10569(TAG_FORCE_CLIENT_BINDING_Z, receiver == null ? 0 : receiver.getManaReceiverPos().method_10260());

		cmp.method_10556(TAG_MAPMAKER_OVERRIDE, mapmakerOverride);
		cmp.method_10569(TAG_FORCED_COLOR, mmForcedColor);
		cmp.method_10569(TAG_FORCED_MANA_PAYLOAD, mmForcedManaPayload);
		cmp.method_10569(TAG_FORCED_TICKS_BEFORE_MANA_LOSS, mmForcedTicksBeforeManaLoss);
		cmp.method_10548(TAG_FORCED_MANA_LOSS_PER_TICK, mmForcedManaLossPerTick);
		cmp.method_10548(TAG_FORCED_GRAVITY, mmForcedGravity);
		cmp.method_10548(TAG_FORCED_VELOCITY_MULTIPLIER, mmForcedVelocityMultiplier);

		requestsClientUpdate = false;
	}

	@Override
	public void readPacketNBT(class_2487 cmp) {
		super.readPacketNBT(cmp);

		String tagUuidMostDeprecated = "uuidMost";
		String tagUuidLeastDeprecated = "uuidLeast";

		if (cmp.method_25928(TAG_UUID)) {
			identity = cmp.method_25926(TAG_UUID);
		} else if (cmp.method_10545(tagUuidLeastDeprecated) && cmp.method_10545(tagUuidMostDeprecated)) { // legacy world compat
			long most = cmp.method_10537(tagUuidMostDeprecated);
			long least = cmp.method_10537(tagUuidLeastDeprecated);
			if (identity == null || most != identity.getMostSignificantBits() || least != identity.getLeastSignificantBits()) {
				this.identity = new UUID(most, least);
			}
		}

		mana = cmp.method_10550(TAG_MANA);
		rotationX = cmp.method_10583(TAG_ROTATION_X);
		rotationY = cmp.method_10583(TAG_ROTATION_Y);
		requestsClientUpdate = cmp.method_10577(TAG_REQUEST_UPDATE);

		if (cmp.method_10545(TAG_INPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_INPUT_KEY);
		}
		if (cmp.method_10545(TAG_OUTPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_OUTPUT_KEY);
		}

		mapmakerOverride = cmp.method_10577(TAG_MAPMAKER_OVERRIDE);
		mmForcedColor = cmp.method_10550(TAG_FORCED_COLOR);
		mmForcedManaPayload = cmp.method_10550(TAG_FORCED_MANA_PAYLOAD);
		mmForcedTicksBeforeManaLoss = cmp.method_10550(TAG_FORCED_TICKS_BEFORE_MANA_LOSS);
		mmForcedManaLossPerTick = cmp.method_10583(TAG_FORCED_MANA_LOSS_PER_TICK);
		mmForcedGravity = cmp.method_10583(TAG_FORCED_GRAVITY);
		mmForcedVelocityMultiplier = cmp.method_10583(TAG_FORCED_VELOCITY_MULTIPLIER);

		if (cmp.method_10545(TAG_PADDING_COLOR)) {
			paddingColor = cmp.method_10550(TAG_PADDING_COLOR) == -1 ? null : class_1767.method_7791(cmp.method_10550(TAG_PADDING_COLOR));
		}
		if (cmp.method_10545(TAG_CAN_SHOOT_BURST)) {
			canShootBurst = cmp.method_10577(TAG_CAN_SHOOT_BURST);
		}

		pingbackTicks = cmp.method_10550(TAG_PINGBACK_TICKS);
		lastPingbackX = cmp.method_10574(TAG_LAST_PINGBACK_X);
		lastPingbackY = cmp.method_10574(TAG_LAST_PINGBACK_Y);
		lastPingbackZ = cmp.method_10574(TAG_LAST_PINGBACK_Z);

		if (requestsClientUpdate && field_11863 != null) {
			int x = cmp.method_10550(TAG_FORCE_CLIENT_BINDING_X);
			int y = cmp.method_10550(TAG_FORCE_CLIENT_BINDING_Y);
			int z = cmp.method_10550(TAG_FORCE_CLIENT_BINDING_Z);
			if (y != Integer.MIN_VALUE) {
				var pos = new class_2338(x, y, z);
				var state = field_11863.method_8320(pos);
				var be = field_11863.method_8321(pos);
				receiver = XplatAbstractions.INSTANCE.findManaReceiver(field_11863, pos, state, be, null);
			} else {
				receiver = null;
			}
		}

		if (field_11863 != null && field_11863.field_9236) {
			hasReceivedInitialPacket = true;
		}
	}

	@Override
	public boolean canReceiveManaFromBursts() {
		return true;
	}

	@Override
	public class_1937 getManaReceiverLevel() {
		return method_10997();
	}

	@Override
	public class_2338 getManaReceiverPos() {
		return method_11016();
	}

	@Override
	public int getCurrentMana() {
		return mana;
	}

	@Override
	public boolean onUsedByWand(@Nullable class_1657 player, class_1799 wand, class_2350 side) {
		if (player == null) {
			return false;
		}

		if (!player.method_5715()) {
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
		} else {
			class_3965 bpos = LexicaBotaniaItem.doRayTrace(field_11863, player, class_3959.class_242.field_1348);
			if (!field_11863.field_9236) {
				double x = bpos.method_17784().field_1352 - method_11016().method_10263() - 0.5;
				double y = bpos.method_17784().field_1351 - method_11016().method_10264() - 0.5;
				double z = bpos.method_17784().field_1350 - method_11016().method_10260() - 0.5;

				if (bpos.method_17780() != class_2350.field_11033 && bpos.method_17780() != class_2350.field_11036) {
					class_243 clickVector = new class_243(x, 0, z);
					class_243 relative = new class_243(-0.5, 0, 0);
					double angle = Math.acos(clickVector.method_1026(relative) / (relative.method_1033() * clickVector.method_1033())) * 180D / Math.PI;

					rotationX = (float) angle + 180F;
					if (clickVector.field_1350 < 0) {
						rotationX = 360 - rotationX;
					}
				}

				double angle = y * 180;
				rotationY = -(float) angle;

				method_5431();
				requestsClientUpdate = true;
			}
		}
		return true;
	}

	private boolean needsNewBurstSimulation() {
		if (field_11863.field_9236 && !hasReceivedInitialPacket) {
			return false;
		}

		if (lastTentativeBurst == null) {
			return true;
		}

		for (PositionProperties props : lastTentativeBurst) {
			if (!props.contentsEqual(field_11863)) {
				invalidTentativeBurst = props.isInvalidIn(field_11863);
				return !invalidTentativeBurst;
			}
		}

		return false;
	}

	private void tryShootBurst() {
		boolean redstone = getVariant() == ManaSpreaderBlock.Variant.REDSTONE;
		if ((receiver != null || redstone) && !invalidTentativeBurst) {
			if (canShootBurst && (redstone || receiver.canReceiveManaFromBursts() && !receiver.isFull())) {
				ManaBurstEntity burst = getBurst(false);
				if (burst != null) {
					if (!field_11863.field_9236) {
						mana -= burst.getStartingMana();
						burst.setShooterUUID(getIdentifier());
						field_11863.method_8649(burst);
						burst.ping();
						if (!BotaniaConfig.common().silentSpreaders()) {
							field_11863.method_8396(null, field_11867, BotaniaSounds.spreaderFire, class_3419.field_15245, 0.05F * (paddingColor != null ? 0.2F : 1F), 0.7F + 0.3F * (float) Math.random());
						}
					}
				}
			}
		}
	}

	public ManaSpreaderBlock.Variant getVariant() {
		class_2248 b = method_11010().method_26204();
		if (b instanceof ManaSpreaderBlock spreader) {
			return spreader.variant;
		} else {
			return ManaSpreaderBlock.Variant.MANA;
		}
	}

	// Should only be called on server
	public void checkForReceiver() {
		class_1799 stack = getItemHandler().method_5438(0);
		ControlLensItem control = getLensController(stack);
		if (control != null && !control.allowBurstShooting(stack, this, false)) {
			return;
		}

		ManaBurstEntity fakeBurst = getBurst(true);
		fakeBurst.setScanBeam();
		ManaReceiver receiver = fakeBurst.getCollidedTile(true);

		if (receiver != null && receiver.getManaReceiverLevel().method_22340(receiver.getManaReceiverPos())) {
			this.receiver = receiver;
		} else {
			this.receiver = null;
		}
		lastTentativeBurst = fakeBurst.propsList;
	}

	@Override
	public ManaBurst runBurstSimulation() {
		ManaBurstEntity fakeBurst = getBurst(true);
		fakeBurst.setScanBeam();
		fakeBurst.getCollidedTile(true);
		return fakeBurst;
	}

	private ManaBurstEntity getBurst(boolean fake) {
		ManaSpreaderBlock.Variant variant = getVariant();
		float gravity = 0F;
		BurstProperties props = new BurstProperties(variant.burstMana, variant.preLossTicks, variant.lossPerTick, gravity, variant.motionModifier, variant.color);

		class_1799 lens = getItemHandler().method_5438(0);
		if (!lens.method_7960() && lens.method_7909() instanceof LensEffectItem lensEffectItem) {
			lensEffectItem.apply(lens, props, field_11863);
		}

		if (getCurrentMana() >= props.maxMana || fake) {
			ManaBurstEntity burst = new ManaBurstEntity(method_10997(), method_11016(), getRotationX(), getRotationY(), fake);
			burst.setSourceLens(lens);

			if (mapmakerOverride) {
				burst.setColor(mmForcedColor);
				burst.setMana(mmForcedManaPayload);
				burst.setStartingMana(mmForcedManaPayload);
				burst.setMinManaLoss(mmForcedTicksBeforeManaLoss);
				burst.setManaLossPerTick(mmForcedManaLossPerTick);
				burst.setGravity(mmForcedGravity);
				burst.method_18799(burst.method_18798().method_1021(mmForcedVelocityMultiplier));
			} else {
				burst.setColor(props.color);
				burst.setMana(props.maxMana);
				burst.setStartingMana(props.maxMana);
				burst.setMinManaLoss(props.ticksBeforeManaLoss);
				burst.setManaLossPerTick(props.manaLossPerTick);
				burst.setGravity(props.gravity);
				burst.method_18799(burst.method_18798().method_1021(props.motionModifier));
			}

			return burst;
		}
		return null;
	}

	public ControlLensItem getLensController(class_1799 stack) {
		if (!stack.method_7960() && stack.method_7909() instanceof ControlLensItem control) {
			if (control.isControlLens(stack)) {
				return control;
			}
		}

		return null;
	}

	public static class WandHud implements WandHUD {
		private final ManaSpreaderBlockEntity spreader;

		public WandHud(ManaSpreaderBlockEntity spreader) {
			this.spreader = spreader;
		}

		@Override
		public void renderHUD(class_4587 ms, class_310 mc) {
			String name = new class_1799(spreader.method_11010().method_26204()).method_7964().getString();
			int color = spreader.getVariant().hudColor;
			BotaniaAPIClient.instance().drawSimpleManaHUD(ms, color, spreader.getCurrentMana(),
					spreader.getMaxMana(), name);

			class_1799 lens = spreader.getItemHandler().method_5438(0);
			if (!lens.method_7960()) {
				class_2561 lensName = lens.method_7964();
				int width = 16 + mc.field_1772.method_27525(lensName) / 2;
				int x = mc.method_22683().method_4486() / 2 - width;
				int y = mc.method_22683().method_4502() / 2 + 50;

				mc.field_1772.method_30881(ms, lensName, x + 20, y + 5, color);
				mc.method_1480().method_4023(lens, x, y);
			}

			if (spreader.receiver != null) {
				var receiverPos = spreader.receiver.getManaReceiverPos();
				class_1799 recieverStack = new class_1799(spreader.field_11863.method_8320(receiverPos).method_26204());
				if (!recieverStack.method_7960()) {
					String stackName = recieverStack.method_7964().getString();
					int width = 16 + mc.field_1772.method_1727(stackName) / 2;
					int x = mc.method_22683().method_4486() / 2 - width;
					int y = mc.method_22683().method_4502() / 2 + 30;

					mc.field_1772.method_1720(ms, stackName, x + 20, y + 5, color);
					mc.method_1480().method_4023(recieverStack, x, y);
				}
			}

			RenderSystem.setShaderColor(1F, 1F, 1F, 1F);
		}
	}

	@Override
	public void onClientDisplayTick() {
		if (field_11863 != null) {
			ManaBurstEntity burst = getBurst(true);
			burst.getCollidedTile(false);
		}
	}

	@Override
	public float getManaYieldMultiplier(ManaBurst burst) {
		return 1F;
	}

	@Override
	protected class_1277 createItemHandler() {
		return new class_1277(1) {
			@Override
			public int method_5444() {
				return 1;
			}

			@Override
			public boolean method_5437(int index, class_1799 stack) {
				return !stack.method_7960() && stack.method_7909() instanceof BasicLensItem;
			}
		};
	}

	@Override
	public void method_5431() {
		super.method_5431();
		if (field_11863 != null) {
			if (!field_11863.field_9236) {
				checkForReceiver();
				VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
			}
		}
	}

	@Override
	public class_2338 getBinding() {
		if (receiver == null) {
			return null;
		}

		return receiver.getManaReceiverPos();
	}

	@Override
	public int getMaxMana() {
		return getVariant().manaCapacity;
	}

	@Override
	public String getInputKey() {
		return inputKey;
	}

	@Override
	public String getOutputKey() {
		return outputKey;
	}

	@Override
	public boolean canSelect(class_1657 player, class_1799 wand, class_2338 pos, class_2350 side) {
		return true;
	}

	@Override
	public boolean bindTo(class_1657 player, class_1799 wand, class_2338 pos, class_2350 side) {
		class_265 shape = player.field_6002.method_8320(pos).method_26218(player.field_6002, pos);
		class_238 axis = shape.method_1110() ? new class_238(pos) : shape.method_1107().method_996(pos);

		class_243 thisVec = class_243.method_24953(method_11016());
		class_243 blockVec = new class_243(axis.field_1323 + (axis.field_1320 - axis.field_1323) / 2, axis.field_1322 + (axis.field_1325 - axis.field_1322) / 2, axis.field_1321 + (axis.field_1324 - axis.field_1321) / 2);

		class_243 diffVec = blockVec.method_1020(thisVec);
		class_243 diffVec2D = new class_243(diffVec.field_1352, diffVec.field_1350, 0);
		class_243 rotVec = new class_243(0, 1, 0);
		double angle = MathHelper.angleBetween(rotVec, diffVec2D) / Math.PI * 180.0;

		if (blockVec.field_1352 < thisVec.field_1352) {
			angle = -angle;
		}

		rotationX = (float) angle + 90;

		rotVec = new class_243(diffVec.field_1352, 0, diffVec.field_1350);
		angle = MathHelper.angleBetween(diffVec, rotVec) * 180F / Math.PI;
		if (blockVec.field_1351 < thisVec.field_1351) {
			angle = -angle;
		}
		rotationY = (float) angle;

		method_5431();
		return true;
	}

	@Override
	public void markDispatchable() {}

	@Override
	public float getRotationX() {
		return rotationX;
	}

	@Override
	public float getRotationY() {
		return rotationY;
	}

	@Override
	public void setRotationX(float rot) {
		rotationX = rot;
	}

	@Override
	public void setRotationY(float rot) {
		rotationY = rot;
	}

	public void rotate(class_2470 rotation) {
		switch (rotation) {
			case field_11463 -> rotationX += 270F;
			case field_11464 -> rotationX += 180F;
			case field_11465 -> rotationX += 90F;
			case field_11467 -> {}
		}

		if (rotationX >= 360F) {
			rotationX -= 360F;
		}
	}

	public void mirror(class_2415 mirror) {
		switch (mirror) {
			case field_11300 -> rotationX = 360F - rotationX;
			case field_11301 -> rotationX = 180F - rotationX;
			case field_11302 -> {}
		}

		if (rotationX < 0F) {
			rotationX += 360F;
		}
	}

	@Override
	public void commitRedirection() {
		method_5431();
	}

	@Override
	public void setCanShoot(boolean canShoot) {
		canShootBurst = canShoot;
	}

	@Override
	public int getBurstParticleTick() {
		return burstParticleTick;
	}

	@Override
	public void setBurstParticleTick(int i) {
		burstParticleTick = i;
	}

	@Override
	public int getLastBurstDeathTick() {
		return lastBurstDeathTick;
	}

	@Override
	public void setLastBurstDeathTick(int i) {
		lastBurstDeathTick = i;
	}

	@Override
	public void pingback(ManaBurst burst, UUID expectedIdentity) {
		if (getIdentifier().equals(expectedIdentity)) {
			pingbackTicks = TICKS_ALLOWED_WITHOUT_PINGBACK;
			class_1297 e = burst.entity();
			lastPingbackX = e.method_23317();
			lastPingbackY = e.method_23318();
			lastPingbackZ = e.method_23321();
			setCanShoot(false);
		}
	}

	@Override
	public UUID getIdentifier() {
		return identity;
	}

}
