/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.block_entity.mana;

import com.google.common.base.Predicates;
import com.mojang.blaze3d.systems.RenderSystem;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.opengl.GL11;

import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.BotaniaAPIClient;
import vazkii.botania.api.block.WandHUD;
import vazkii.botania.api.block.Wandable;
import vazkii.botania.api.internal.VanillaPacketDispatcher;
import vazkii.botania.api.item.ManaDissolvable;
import vazkii.botania.api.mana.*;
import vazkii.botania.api.mana.spark.ManaSpark;
import vazkii.botania.api.mana.spark.SparkAttachable;
import vazkii.botania.api.recipe.ManaInfusionRecipe;
import vazkii.botania.api.state.BotaniaStateProperties;
import vazkii.botania.client.core.helper.RenderHelper;
import vazkii.botania.client.fx.SparkleParticleData;
import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.client.gui.HUDHandler;
import vazkii.botania.common.block.BotaniaBlocks;
import vazkii.botania.common.block.block_entity.BotaniaBlockEntities;
import vazkii.botania.common.block.block_entity.BotaniaBlockEntity;
import vazkii.botania.common.block.mana.ManaPoolBlock;
import vazkii.botania.common.crafting.BotaniaRecipeTypes;
import vazkii.botania.common.handler.BotaniaSounds;
import vazkii.botania.common.handler.ManaNetworkHandler;
import vazkii.botania.common.helper.EntityHelper;
import vazkii.botania.common.item.BotaniaItems;
import vazkii.botania.common.item.ManaTabletItem;
import vazkii.botania.common.proxy.Proxy;
import vazkii.botania.xplat.BotaniaConfig;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_1297;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1767;
import net.minecraft.class_1799;
import net.minecraft.class_1860;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_3419;
import net.minecraft.class_4587;

import static vazkii.botania.api.state.BotaniaStateProperties.OPTIONAL_DYE_COLOR;

public class ManaPoolBlockEntity extends BotaniaBlockEntity implements ManaPool, KeyLocked, SparkAttachable,
		ThrottledPacket, Wandable {
	public static final int PARTICLE_COLOR = 0x00C6FF;
	public static final int MAX_MANA = 1000000;
	private static final int MAX_MANA_DILLUTED = 10000;

	private static final String TAG_MANA = "mana";
	private static final String TAG_OUTPUTTING = "outputting";
	private static final String TAG_MANA_CAP = "manaCap";
	private static final String TAG_CAN_ACCEPT = "canAccept";
	private static final String TAG_CAN_SPARE = "canSpare";
	private static final String TAG_INPUT_KEY = "inputKey";
	private static final String TAG_OUTPUT_KEY = "outputKey";
	private static final int CRAFT_EFFECT_EVENT = 0;
	private static final int CHARGE_EFFECT_EVENT = 1;

	private boolean outputting = false;

	private Optional<class_1767> legacyColor = Optional.empty();
	private int mana;

	private int manaCap = -1;
	private int soundTicks = 0;
	private boolean canAccept = true;
	private boolean canSpare = true;
	boolean isDoingTransfer = false;
	int ticksDoingTransfer = 0;

	private String inputKey = "";
	private final String outputKey = "";

	private int ticks = 0;
	private boolean sendPacket = false;

	public ManaPoolBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaBlockEntities.POOL, pos, state);
	}

	@Override
	public boolean isFull() {
		class_2680 stateBelow = field_11863.method_8320(field_11867.method_10074());
		return !stateBelow.method_27852(BotaniaBlocks.manaVoid) && getCurrentMana() >= getMaxMana();
	}

	@Override
	public void receiveMana(int mana) {
		int old = this.mana;
		this.mana = Math.max(0, Math.min(getCurrentMana() + mana, getMaxMana()));
		if (old != this.mana) {
			method_5431();
			markDispatchable();
		}
	}

	@Override
	public void method_11012() {
		super.method_11012();
		BotaniaAPI.instance().getManaNetworkInstance().fireManaNetworkEvent(this, ManaBlockType.POOL, ManaNetworkAction.REMOVE);
	}

	public static int calculateComparatorLevel(int mana, int max) {
		int val = (int) ((double) mana / (double) max * 15.0);
		if (mana > 0) {
			val = Math.max(val, 1);
		}
		return val;
	}

	public ManaInfusionRecipe getMatchingRecipe(@NotNull class_1799 stack, @NotNull class_2680 state) {
		List<ManaInfusionRecipe> matchingNonCatRecipes = new ArrayList<>();
		List<ManaInfusionRecipe> matchingCatRecipes = new ArrayList<>();

		for (var r : BotaniaRecipeTypes.getRecipes(field_11863, BotaniaRecipeTypes.MANA_INFUSION_TYPE).values()) {
			if (r instanceof ManaInfusionRecipe recipe && recipe.matches(stack)) {
				if (recipe.getRecipeCatalyst() == null) {
					matchingNonCatRecipes.add(recipe);
				} else if (recipe.getRecipeCatalyst().test(state)) {
					matchingCatRecipes.add(recipe);
				}
			}
		}

		// Recipes with matching catalyst take priority above recipes with no catalyst specified
		return !matchingCatRecipes.isEmpty() ? matchingCatRecipes.get(0) : !matchingNonCatRecipes.isEmpty() ? matchingNonCatRecipes.get(0) : null;
	}

	public boolean collideEntityItem(class_1542 item) {
		if (field_11863.field_9236 || !item.method_5805() || item.method_6983().method_7960()) {
			return false;
		}

		class_1799 stack = item.method_6983();

		if (stack.method_7909() instanceof ManaDissolvable dissolvable) {
			dissolvable.onDissolveTick(this, item);
		}

		if (XplatAbstractions.INSTANCE.itemFlagsComponent(item).getManaInfusionCooldown() > 0) {
			return false;
		}

		ManaInfusionRecipe recipe = getMatchingRecipe(stack, field_11863.method_8320(field_11867.method_10074()));

		if (recipe != null) {
			int mana = recipe.getManaToConsume();
			if (getCurrentMana() >= mana) {
				receiveMana(-mana);

				class_1799 output = recipe.getRecipeOutput(stack);
				EntityHelper.shrinkItem(item);
				item.method_24830(false); //Force entity collision update to run every tick if crafting is in progress

				class_1542 outputItem = new class_1542(field_11863, field_11867.method_10263() + 0.5, field_11867.method_10264() + 1.5, field_11867.method_10260() + 0.5, output);
				XplatAbstractions.INSTANCE.itemFlagsComponent(outputItem).markNewlyInfused();
				field_11863.method_8649(outputItem);

				craftingFanciness();
				return true;
			}
		}

		return false;
	}

	private void craftingFanciness() {
		if (soundTicks == 0) {
			field_11863.method_8396(null, field_11867, BotaniaSounds.manaPoolCraft, class_3419.field_15245, 1F, 1F);
			soundTicks = 6;
		}

		field_11863.method_8427(method_11016(), method_11010().method_26204(), CRAFT_EFFECT_EVENT, 0);
	}

	@Override
	public boolean method_11004(int event, int param) {
		switch (event) {
			case CRAFT_EFFECT_EVENT: {
				if (field_11863.field_9236) {
					for (int i = 0; i < 25; i++) {
						float red = (float) Math.random();
						float green = (float) Math.random();
						float blue = (float) Math.random();
						SparkleParticleData data = SparkleParticleData.sparkle((float) Math.random(), red, green, blue, 10);
						field_11863.method_8406(data, field_11867.method_10263() + 0.5 + Math.random() * 0.4 - 0.2, field_11867.method_10264() + 0.75, field_11867.method_10260() + 0.5 + Math.random() * 0.4 - 0.2, 0, 0, 0);
					}
				}

				return true;
			}
			case CHARGE_EFFECT_EVENT: {
				if (field_11863.field_9236) {
					if (BotaniaConfig.common().chargingAnimationEnabled()) {
						boolean outputting = param == 1;
						class_243 itemVec = class_243.method_24954(field_11867).method_1031(0.5, 0.5 + Math.random() * 0.3, 0.5);
						class_243 tileVec = class_243.method_24954(field_11867).method_1031(0.2 + Math.random() * 0.6, 0, 0.2 + Math.random() * 0.6);
						Proxy.INSTANCE.lightningFX(field_11863, outputting ? tileVec : itemVec,
								outputting ? itemVec : tileVec, 80, field_11863.field_9229.method_43055(), 0x4400799c, 0x4400C6FF);
					}
				}
				return true;
			}
			default:
				return super.method_11004(event, param);
		}
	}

	private void initManaCapAndNetwork() {
		if (getMaxMana() == -1) {
			manaCap = ((ManaPoolBlock) method_11010().method_26204()).variant == ManaPoolBlock.Variant.DILUTED ? MAX_MANA_DILLUTED : MAX_MANA;
		}
		if (!ManaNetworkHandler.instance.isPoolIn(field_11863, this) && !method_11015()) {
			BotaniaAPI.instance().getManaNetworkInstance().fireManaNetworkEvent(this, ManaBlockType.POOL, ManaNetworkAction.ADD);
		}
	}

	public static void clientTick(class_1937 level, class_2338 worldPosition, class_2680 state, ManaPoolBlockEntity self) {
		self.initManaCapAndNetwork();
		double particleChance = 1F - (double) self.getCurrentMana() / (double) self.getMaxMana() * 0.1;
		if (Math.random() > particleChance) {
			float red = (PARTICLE_COLOR >> 16 & 0xFF) / 255F;
			float green = (PARTICLE_COLOR >> 8 & 0xFF) / 255F;
			float blue = (PARTICLE_COLOR & 0xFF) / 255F;
			WispParticleData data = WispParticleData.wisp((float) Math.random() / 3F, red, green, blue, 2F);
			level.method_8406(data, worldPosition.method_10263() + 0.3 + Math.random() * 0.5, worldPosition.method_10264() + 0.6 + Math.random() * 0.25, worldPosition.method_10260() + Math.random(), 0, (float) Math.random() / 25F, 0);
		}
	}

	public static void serverTick(class_1937 level, class_2338 worldPosition, class_2680 state, ManaPoolBlockEntity self) {

		// Legacy color format
		if (self.legacyColor.isPresent()) {
			self.setColor(self.legacyColor);
			self.legacyColor = Optional.empty();
		}

		self.initManaCapAndNetwork();
		boolean wasDoingTransfer = self.isDoingTransfer;
		self.isDoingTransfer = false;

		if (self.soundTicks > 0) {
			self.soundTicks--;
		}

		if (self.sendPacket && self.ticks % 10 == 0) {
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(self);
			self.sendPacket = false;
		}

		List<class_1542> items = level.method_18467(class_1542.class, new class_238(worldPosition, worldPosition.method_10069(1, 1, 1)));
		for (class_1542 item : items) {
			if (!item.method_5805()) {
				continue;
			}

			class_1799 stack = item.method_6983();
			var mana = XplatAbstractions.INSTANCE.findManaItem(stack);
			if (!stack.method_7960() && mana != null) {
				if (self.outputting && mana.canReceiveManaFromPool(self) || !self.outputting && mana.canExportManaToPool(self)) {
					boolean didSomething = false;

					int bellowCount = 0;
					if (self.outputting) {
						for (class_2350 dir : class_2350.class_2353.field_11062) {
							class_2586 tile = level.method_8321(worldPosition.method_10093(dir));
							if (tile instanceof BellowsBlockEntity bellows && bellows.getLinkedTile() == self) {
								bellowCount++;
							}
						}
					}
					int transfRate = 1000 * (bellowCount + 1);

					if (self.outputting) {
						if (self.canSpare) {
							if (self.getCurrentMana() > 0 && mana.getMana() < mana.getMaxMana()) {
								didSomething = true;
							}

							int manaVal = Math.min(transfRate, Math.min(self.getCurrentMana(), mana.getMaxMana() - mana.getMana()));
							mana.addMana(manaVal);
							self.receiveMana(-manaVal);
						}
					} else {
						if (self.canAccept) {
							if (mana.getMana() > 0 && !self.isFull()) {
								didSomething = true;
							}

							int manaVal = Math.min(transfRate, Math.min(self.getMaxMana() - self.getCurrentMana(), mana.getMana()));
							if (manaVal == 0 && self.field_11863.method_8320(worldPosition.method_10074()).method_27852(BotaniaBlocks.manaVoid)) {
								manaVal = Math.min(transfRate, mana.getMana());
							}
							mana.addMana(-manaVal);
							self.receiveMana(manaVal);
						}
					}

					if (didSomething) {
						if (BotaniaConfig.common().chargingAnimationEnabled() && level.field_9229.method_43048(20) == 0) {
							level.method_8427(worldPosition, state.method_26204(), CHARGE_EFFECT_EVENT, self.outputting ? 1 : 0);
						}
						EntityHelper.syncItem(item);
						self.isDoingTransfer = self.outputting;
					}
				}
			}
		}

		if (self.isDoingTransfer) {
			self.ticksDoingTransfer++;
		} else {
			self.ticksDoingTransfer = 0;
			if (wasDoingTransfer) {
				VanillaPacketDispatcher.dispatchTEToNearbyPlayers(self);
			}
		}

		self.ticks++;
	}

	@Override
	public void writePacketNBT(class_2487 cmp) {
		cmp.method_10569(TAG_MANA, getCurrentMana());
		cmp.method_10556(TAG_OUTPUTTING, outputting);

		cmp.method_10569(TAG_MANA_CAP, getMaxMana());
		cmp.method_10556(TAG_CAN_ACCEPT, canAccept);
		cmp.method_10556(TAG_CAN_SPARE, canSpare);

		cmp.method_10582(TAG_INPUT_KEY, inputKey);
		cmp.method_10582(TAG_OUTPUT_KEY, outputKey);
	}

	@Override
	public void readPacketNBT(class_2487 cmp) {
		mana = cmp.method_10550(TAG_MANA);
		outputting = cmp.method_10577(TAG_OUTPUTTING);

		// Legacy color format
		if (cmp.method_10545("color")) {
			class_1767 color = class_1767.method_7791(cmp.method_10550("color"));
			// White was previously used as "no color"
			if (color != class_1767.field_7952) {
				legacyColor = Optional.of(color);
			} else {
				legacyColor = Optional.empty();
			}
		}
		if (cmp.method_10545(TAG_MANA_CAP)) {
			manaCap = cmp.method_10550(TAG_MANA_CAP);
		}
		if (cmp.method_10545(TAG_CAN_ACCEPT)) {
			canAccept = cmp.method_10577(TAG_CAN_ACCEPT);
		}
		if (cmp.method_10545(TAG_CAN_SPARE)) {
			canSpare = cmp.method_10577(TAG_CAN_SPARE);
		}

		if (cmp.method_10545(TAG_INPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_INPUT_KEY);
		}
		if (cmp.method_10545(TAG_OUTPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_OUTPUT_KEY);
		}

	}

	@Override
	public boolean onUsedByWand(@Nullable class_1657 player, class_1799 stack, class_2350 side) {
		if (player == null || player.method_5715()) {
			outputting = !outputting;
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
		}
		return true;
	}

	public static class WandHud implements WandHUD {
		private final ManaPoolBlockEntity pool;

		public WandHud(ManaPoolBlockEntity pool) {
			this.pool = pool;
		}

		@Override
		public void renderHUD(class_4587 ms, class_310 mc) {
			class_1799 poolStack = new class_1799(pool.method_11010().method_26204());
			String name = poolStack.method_7964().getString();
			int color = 0x4444FF;
			BotaniaAPIClient.instance().drawSimpleManaHUD(ms, color, pool.getCurrentMana(), pool.getMaxMana(), name);

			int x = class_310.method_1551().method_22683().method_4486() / 2 - 11;
			int y = class_310.method_1551().method_22683().method_4502() / 2 + 30;

			int u = pool.outputting ? 22 : 0;
			int v = 38;

			RenderSystem.enableBlend();
			RenderSystem.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);

			RenderSystem.setShaderTexture(0, HUDHandler.manaBar);
			RenderHelper.drawTexturedModalRect(ms, x, y, u, v, 22, 15);
			RenderSystem.setShaderColor(1F, 1F, 1F, 1F);

			class_1799 tablet = new class_1799(BotaniaItems.manaTablet);
			ManaTabletItem.setStackCreative(tablet);

			mc.method_1480().method_4023(tablet, x - 20, y);
			mc.method_1480().method_4023(poolStack, x + 26, y);

			RenderSystem.disableBlend();
		}
	}

	@Override
	public boolean canReceiveManaFromBursts() {
		return true;
	}

	@Override
	public boolean isOutputtingPower() {
		return outputting;
	}

	@Override
	public class_1937 getManaReceiverLevel() {
		return method_10997();
	}

	@Override
	public class_2338 getManaReceiverPos() {
		return method_11016();
	}

	@Override
	public int getCurrentMana() {
		if (method_11010().method_26204() instanceof ManaPoolBlock pool) {
			return pool.variant == ManaPoolBlock.Variant.CREATIVE ? MAX_MANA : mana;
		}
		return 0;
	}

	@Override
	public int getMaxMana() {
		return manaCap;
	}

	@Override
	public String getInputKey() {
		return inputKey;
	}

	@Override
	public String getOutputKey() {
		return outputKey;
	}

	@Override
	public boolean canAttachSpark(class_1799 stack) {
		return true;
	}

	@Override
	public ManaSpark getAttachedSpark() {
		List<class_1297> sparks = field_11863.method_8390(class_1297.class, new class_238(field_11867.method_10084(), field_11867.method_10084().method_10069(1, 1, 1)), Predicates.instanceOf(ManaSpark.class));
		if (sparks.size() == 1) {
			class_1297 e = sparks.get(0);
			return (ManaSpark) e;
		}

		return null;
	}

	@Override
	public boolean areIncomingTranfersDone() {
		return false;
	}

	@Override
	public int getAvailableSpaceForMana() {
		int space = Math.max(0, getMaxMana() - getCurrentMana());
		if (space > 0) {
			return space;
		} else if (field_11863.method_8320(field_11867.method_10074()).method_27852(BotaniaBlocks.manaVoid)) {
			return getMaxMana();
		} else {
			return 0;
		}
	}

	@Override
	public Optional<class_1767> getColor() {
		return method_11010().method_11654(OPTIONAL_DYE_COLOR).toDyeColor();
	}

	@Override
	public void setColor(Optional<class_1767> color) {
		field_11863.method_8501(field_11867, method_11010().method_11657(OPTIONAL_DYE_COLOR, BotaniaStateProperties.OptionalDyeColor.fromOptionalDyeColor(color)));
	}

	@Override
	public void markDispatchable() {
		sendPacket = true;
	}
}
