/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.block_entity.corporea;

import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.corporea.*;
import vazkii.botania.common.BotaniaStats;
import vazkii.botania.common.advancements.CorporeaRequestTrigger;
import vazkii.botania.common.block.block_entity.BotaniaBlockEntities;
import vazkii.botania.common.helper.MathHelper;
import vazkii.botania.network.serverbound.IndexStringRequestPacket;
import vazkii.botania.xplat.ClientXplatAbstractions;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.minecraft.class_124;
import net.minecraft.class_1309;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_238;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_3222;

public class CorporeaIndexBlockEntity extends BaseCorporeaBlockEntity implements CorporeaRequestor {
	public static final double RADIUS = 2.5;
	public static final int MAX_REQUEST = 1 << 16;

	private static final Set<CorporeaIndexBlockEntity> serverIndexes = Collections.newSetFromMap(new WeakHashMap<>());
	private static final Set<CorporeaIndexBlockEntity> clientIndexes = Collections.newSetFromMap(new WeakHashMap<>());

	private static final Map<Pattern, IRegexStacker> patterns = new LinkedHashMap<>();

	/**
	 * (name) = Item name, or "this" for the name of the item in your hand
	 * (n), (n1), (n2), etc = Numbers
	 * [text] = Optional
	 * <a/b> = Either a or b
	 */
	static {
		// (name) = 1
		addPattern("(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 1;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a][n] (name) = 1
		addPattern("a??n?? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 1;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		//(n)[x][ of] (name) = n
		addPattern("(\\d+)x?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// [a ]stack[ of] (name) = 64
		addPattern("(?:a )?stack(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// (n)[x] stack[s][ of] (name) = n * 64
		addPattern("(\\d+)x?? stacks?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64 * i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// [a ]stack <and/+> (n)[x][ of] (name) = 64 + n
		addPattern("(?:a )?stack (?:(?:and)|(?:\\+)) (\\d+)(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64 + i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// (n1)[x] stack[s] <and/+> (n2)[x][ of] (name) = n1 * 64 + n2
		addPattern("(\\d+)x?? stacks? (?:(?:and)|(?:\\+)) (\\d+)x?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64 * i(m, 1) + i(m, 2);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(3);
			}
		});

		// [a ]half [of ][a ]stack[ of] (name) = 32
		addPattern("(?:a )?half (?:of )?(?:a )?stack(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 32;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a ]quarter [of ][a ]stack[ of] (name) = 16
		addPattern("(?:a )?quarter (?:of )?(?:a )?stack(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 16;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a ]dozen[ of] (name) = 12
		addPattern("(?:a )?dozen(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 12;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// (n)[x] dozen[s][ of] (name) = n * 12
		addPattern("(\\d+)x?? dozens?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 12 * i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// <all/every> [<of/the> ](name) = 2147483647
		addPattern("(?:all|every) (?:(?:of|the) )?(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return MAX_REQUEST;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [the ]answer to life[,] the universe and everything [of ](name) = 42
		addPattern("(?:the )?answer to life,? the universe and everything (?:of )?(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 42;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a ]nice [of ](name) = 69 
		addPattern("(?:a )?nice (?:of )?(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 69;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// (n)[x] nice[s][ of] (name) = n * 69
		addPattern("(\\d+)x?? nices?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 69 * i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// <count/show/display/tell> (name) = 0 (display only)
		addPattern("(?:count|show|display|tell) (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 0;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});
	}

	public int ticksWithCloseby = 0;
	public float closeby = 0F;
	public boolean hasCloseby;

	public CorporeaIndexBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaBlockEntities.CORPOREA_INDEX, pos, state);
	}

	public static void commonTick(class_1937 level, class_2338 worldPosition, class_2680 state, CorporeaIndexBlockEntity self) {
		double x = worldPosition.method_10263() + 0.5;
		double y = worldPosition.method_10264() + 0.5;
		double z = worldPosition.method_10260() + 0.5;

		List<class_1657> players = level.method_18467(class_1657.class, new class_238(x - RADIUS, y - RADIUS, z - RADIUS, x + RADIUS, y + RADIUS, z + RADIUS));
		self.hasCloseby = false;
		if (self.getSpark() != null) {
			for (class_1657 player : players) {
				if (self.isInRange(player)) {
					self.hasCloseby = true;
					break;
				}
			}
		}

		float step = 0.2F;
		if (self.hasCloseby) {
			self.ticksWithCloseby++;
			if (self.closeby < 1F) {
				self.closeby += step;
			}
		} else if (self.closeby > 0F) {
			self.closeby -= step;
		}

		if (!self.method_11015()) {
			addIndex(self);
		}
	}

	public static List<CorporeaIndexBlockEntity> getNearbyValidIndexes(class_1657 player) {
		List<CorporeaIndexBlockEntity> result = new ArrayList<>();
		for (var index : (player.field_6002.field_9236 ? clientIndexes : serverIndexes)) {
			if (index.getSpark() != null && index.isInRange(player)) {
				result.add(index);
			}
		}
		return result;
	}

	@Override
	public void method_11012() {
		super.method_11012();
		removeIndex(this);
	}

	@Override
	public void doCorporeaRequest(CorporeaRequestMatcher request, int count, CorporeaSpark spark, @Nullable class_1309 entity) {
		doRequest(request, count, spark, entity);
	}

	private CorporeaResult doRequest(CorporeaRequestMatcher matcher, int count, CorporeaSpark spark, @Nullable class_1309 entity) {
		CorporeaResult result = CorporeaHelper.instance().requestItem(matcher, count, spark, entity, true);
		List<class_1799> stacks = result.stacks();
		spark.onItemsRequested(stacks);
		for (class_1799 stack : stacks) {
			if (!stack.method_7960()) {
				class_1542 item = new class_1542(field_11863, field_11867.method_10263() + 0.5, field_11867.method_10264() + 1.5, field_11867.method_10260() + 0.5, stack);
				field_11863.method_8649(item);
			}
		}
		return result;
	}

	private boolean isInRange(class_1657 player) {
		return player.field_6002.method_27983() == field_11863.method_27983()
				&& MathHelper.pointDistancePlane(method_11016().method_10263() + 0.5, method_11016().method_10260() + 0.5, player.method_23317(), player.method_23321()) < RADIUS
				&& Math.abs(method_11016().method_10264() + 0.5 - player.method_23318()) < 5;
	}

	public static void addPattern(String pattern, IRegexStacker stacker) {
		patterns.put(Pattern.compile(pattern), stacker);
	}

	public static int i(Matcher m, int g) {
		try {
			int i = Math.abs(Integer.parseInt(m.group(g)));
			return i;
		} catch (NumberFormatException e) {
			return 0;
		}
	}

	private static void addIndex(CorporeaIndexBlockEntity index) {
		Set<CorporeaIndexBlockEntity> set = index.field_11863.field_9236 ? clientIndexes : serverIndexes;
		set.add(index);
	}

	private static void removeIndex(CorporeaIndexBlockEntity index) {
		Set<CorporeaIndexBlockEntity> set = index.field_11863.field_9236 ? clientIndexes : serverIndexes;
		set.remove(index);
	}

	public static void clearIndexCache() {
		clientIndexes.clear();
		serverIndexes.clear();
	}

	public void performPlayerRequest(class_3222 player, CorporeaRequestMatcher request, int count) {
		if (!XplatAbstractions.INSTANCE.fireCorporeaIndexRequestEvent(player, request, count, this.getSpark())) {
			CorporeaResult res = this.doRequest(request, count, this.getSpark(), player);

			player.method_43496(class_2561.method_43469("botaniamisc.requestMsg", count, request.getRequestName(), res.matchedCount(), res.extractedCount()).method_27692(class_124.field_1076));
			player.method_7339(BotaniaStats.CORPOREA_ITEMS_REQUESTED, res.extractedCount());
			CorporeaRequestTrigger.INSTANCE.trigger(player, player.method_14220(), this.method_11016(), res.extractedCount());
		}
	}

	public static class ClientHandler {
		public static boolean onChat(class_1657 player, String message) {
			if (!getNearbyValidIndexes(player).isEmpty()) {
				ClientXplatAbstractions.INSTANCE.sendToServer(new IndexStringRequestPacket(message));
				return true;
			}
			return false;
		}
	}

	public static void onChatMessage(class_3222 player, String message) {
		if (player.method_7325()) {
			return;
		}

		List<CorporeaIndexBlockEntity> nearbyIndexes = getNearbyValidIndexes(player);
		if (!nearbyIndexes.isEmpty()) {
			String msg = message.toLowerCase(Locale.ROOT).trim();
			for (CorporeaIndexBlockEntity index : nearbyIndexes) {
				String name = "";
				int count = 0;
				for (Pattern pattern : patterns.keySet()) {
					Matcher matcher = pattern.matcher(msg);
					if (matcher.matches()) {
						IRegexStacker stacker = patterns.get(pattern);
						count = Math.min(MAX_REQUEST, stacker.getCount(matcher));
						name = stacker.getName(matcher).toLowerCase(Locale.ROOT).trim();
					}
				}

				if (name.equals("this")) {
					class_1799 stack = player.method_6047();
					if (!stack.method_7960()) {
						name = stack.method_7964().getString().toLowerCase(Locale.ROOT).trim();
					}
				}

				index.performPlayerRequest(player, CorporeaHelper.instance().createMatcher(name), count);
			}
		}
	}

	public interface IRegexStacker {
		int getCount(Matcher m);

		String getName(Matcher m);

	}

}
