/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.block_entity;

import com.google.common.base.Predicates;
import com.google.common.base.Suppliers;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.internal.VanillaPacketDispatcher;
import vazkii.botania.api.mana.ManaPool;
import vazkii.botania.api.mana.ManaReceiver;
import vazkii.botania.api.mana.spark.ManaSpark;
import vazkii.botania.api.mana.spark.SparkAttachable;
import vazkii.botania.api.mana.spark.SparkHelper;
import vazkii.botania.api.recipe.TerrestrialAgglomerationRecipe;
import vazkii.botania.common.block.BotaniaBlocks;
import vazkii.botania.common.crafting.BotaniaRecipeTypes;
import vazkii.botania.common.handler.BotaniaSounds;
import vazkii.botania.common.lib.BotaniaTags;
import vazkii.botania.network.EffectType;
import vazkii.botania.network.clientbound.BotaniaEffectPacket;
import vazkii.botania.xplat.XplatAbstractions;
import vazkii.patchouli.api.IMultiblock;
import vazkii.patchouli.api.PatchouliAPI;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;
import net.minecraft.class_1277;
import net.minecraft.class_1297;
import net.minecraft.class_1301;
import net.minecraft.class_1542;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2378;
import net.minecraft.class_238;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_3419;
import net.minecraft.class_6862;

public class TerrestrialAgglomerationPlateBlockEntity extends BotaniaBlockEntity implements SparkAttachable, ManaReceiver {
	public static final Supplier<IMultiblock> MULTIBLOCK = Suppliers.memoize(() -> PatchouliAPI.get().makeMultiblock(
			new String[][] {
					{
							"___",
							"_P_",
							"___"
					},
					{
							"RLR",
							"L0L",
							"RLR"
					}
			},
			'P', BotaniaBlocks.terraPlate,
			'R', PatchouliAPI.get().tagMatcher(BotaniaTags.Blocks.TERRA_PLATE_BASE),
			'0', PatchouliAPI.get().tagMatcher(BotaniaTags.Blocks.TERRA_PLATE_BASE),
			'L', PatchouliAPI.get().tagMatcher(
					XplatAbstractions.INSTANCE.isFabric()
							? class_6862.method_40092(class_2378.field_25105, new class_2960("c", "lapis_blocks"))
							: class_6862.method_40092(class_2378.field_25105, new class_2960("forge", "storage_blocks/lapis")))
	));

	private static final String TAG_MANA = "mana";

	private int mana;

	public TerrestrialAgglomerationPlateBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaBlockEntities.TERRA_PLATE, pos, state);
	}

	public static void serverTick(class_1937 level, class_2338 worldPosition, class_2680 state, TerrestrialAgglomerationPlateBlockEntity self) {
		boolean removeMana = true;

		if (self.hasValidPlatform()) {
			List<class_1799> items = self.getItems();
			class_1277 inv = self.getInventory();

			TerrestrialAgglomerationRecipe recipe = self.getCurrentRecipe(inv);
			if (recipe != null) {
				removeMana = false;
				ManaSpark spark = self.getAttachedSpark();
				if (spark != null) {
					var otherSparks = SparkHelper.getSparksAround(level, worldPosition.method_10263() + 0.5, worldPosition.method_10264() + 0.5, worldPosition.method_10260() + 0.5, spark.getNetwork());
					for (var otherSpark : otherSparks) {
						if (spark != otherSpark && otherSpark.getAttachedManaReceiver() instanceof ManaPool) {
							otherSpark.registerTransfer(spark);
						}
					}
				}
				if (self.mana > 0) {
					VanillaPacketDispatcher.dispatchTEToNearbyPlayers(self);
					int proportion = Float.floatToIntBits(self.getCompletion());
					XplatAbstractions.INSTANCE.sendToNear(level, worldPosition,
							new BotaniaEffectPacket(EffectType.TERRA_PLATE, worldPosition.method_10263(), worldPosition.method_10264(), worldPosition.method_10260(), proportion));
				}

				if (self.mana >= recipe.getMana()) {
					class_1799 result = recipe.method_8116(inv);
					for (class_1799 item : items) {
						item.method_7939(0);
					}
					class_1542 item = new class_1542(level, worldPosition.method_10263() + 0.5, worldPosition.method_10264() + 0.2, worldPosition.method_10260() + 0.5, result);
					item.method_18799(class_243.field_1353);
					level.method_8649(item);
					level.method_43128(null, item.method_23317(), item.method_23318(), item.method_23321(), BotaniaSounds.terrasteelCraft, class_3419.field_15245, 1F, 1F);
					self.mana = 0;
					level.method_8455(worldPosition, state.method_26204());
					VanillaPacketDispatcher.dispatchTEToNearbyPlayers(self);
				}
			}
		}

		if (removeMana) {
			self.receiveMana(-1000);
		}
	}

	private List<class_1799> getItems() {
		List<class_1542> itemEntities = field_11863.method_8390(class_1542.class, new class_238(field_11867, field_11867.method_10069(1, 1, 1)), class_1301.field_6154);
		List<class_1799> stacks = new ArrayList<>();
		for (class_1542 entity : itemEntities) {
			if (!entity.method_6983().method_7960()) {
				stacks.add(entity.method_6983());
			}
		}
		return stacks;
	}

	private class_1277 getInventory() {
		List<class_1799> items = getItems();
		return new class_1277(flattenStacks(items));
	}

	/**
	 * Flattens the list of stacks into an array of stacks with size 1,
	 * for recipe matching purposes only.
	 * If the total count of items exceeds 64, returns no items.
	 */
	private static class_1799[] flattenStacks(List<class_1799> items) {
		class_1799[] stacks;
		int i = 0;
		for (class_1799 item : items) {
			i += item.method_7947();
		}
		if (i > 64) {
			return new class_1799[0];
		}

		stacks = new class_1799[i];
		int j = 0;
		for (class_1799 item : items) {
			if (item.method_7947() > 1) {
				class_1799 temp = item.method_7972();
				temp.method_7939(1);
				for (int count = 0; count < item.method_7947(); count++) {
					stacks[j] = temp.method_7972();
					j++;
				}
			} else {
				stacks[j] = item;
				j++;
			}
		}
		return stacks;
	}

	@Nullable
	private TerrestrialAgglomerationRecipe getCurrentRecipe(class_1277 items) {
		if (items.method_5442()) {
			return null;
		}
		return field_11863.method_8433().method_8132(BotaniaRecipeTypes.TERRA_PLATE_TYPE, items, field_11863).orElse(null);
	}

	private boolean isActive() {
		return getCurrentRecipe(getInventory()) != null;
	}

	private boolean hasValidPlatform() {
		return MULTIBLOCK.get().validate(field_11863, method_11016().method_10074()) != null;
	}

	@Override
	public void writePacketNBT(class_2487 cmp) {
		cmp.method_10569(TAG_MANA, mana);
	}

	@Override
	public void readPacketNBT(class_2487 cmp) {
		mana = cmp.method_10550(TAG_MANA);
	}

	@Override
	public class_1937 getManaReceiverLevel() {
		return method_10997();
	}

	@Override
	public class_2338 getManaReceiverPos() {
		return method_11016();
	}

	@Override
	public int getCurrentMana() {
		return mana;
	}

	@Override
	public boolean isFull() {
		TerrestrialAgglomerationRecipe recipe = getCurrentRecipe(getInventory());
		return recipe == null || getCurrentMana() >= recipe.getMana();
	}

	@Override
	public void receiveMana(int mana) {
		this.mana = Math.max(0, this.mana + mana);
		field_11863.method_8455(field_11867, method_11010().method_26204());
	}

	@Override
	public boolean canReceiveManaFromBursts() {
		return isActive();
	}

	@Override
	public boolean canAttachSpark(class_1799 stack) {
		return true;
	}

	@Override
	public ManaSpark getAttachedSpark() {
		List<class_1297> sparks = field_11863.method_8390(class_1297.class, new class_238(field_11867.method_10084(), field_11867.method_10084().method_10069(1, 1, 1)), Predicates.instanceOf(ManaSpark.class));
		if (sparks.size() == 1) {
			class_1297 e = sparks.get(0);
			return (ManaSpark) e;
		}

		return null;
	}

	@Override
	public boolean areIncomingTranfersDone() {
		return !isActive();
	}

	@Override
	public int getAvailableSpaceForMana() {
		TerrestrialAgglomerationRecipe recipe = getCurrentRecipe(getInventory());
		return recipe == null ? 0 : Math.max(0, recipe.getMana() - getCurrentMana());
	}

	public float getCompletion() {
		TerrestrialAgglomerationRecipe recipe = getCurrentRecipe(getInventory());
		if (recipe == null) {
			return 0;
		}
		return ((float) getCurrentMana()) / recipe.getMana();
	}

	public int getComparatorLevel() {
		int val = (int) (getCompletion() * 15.0);
		if (getCurrentMana() > 0) {
			val = Math.max(val, 1);
		}
		return val;
	}

}
