/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.block_entity;

import com.google.common.base.Predicates;
import com.google.common.base.Suppliers;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.block.WandHUD;
import vazkii.botania.api.block.Wandable;
import vazkii.botania.api.internal.VanillaPacketDispatcher;
import vazkii.botania.api.mana.ManaPool;
import vazkii.botania.api.mana.ManaReceiver;
import vazkii.botania.api.mana.spark.ManaSpark;
import vazkii.botania.api.mana.spark.SparkAttachable;
import vazkii.botania.api.mana.spark.SparkHelper;
import vazkii.botania.api.state.BotaniaStateProperties;
import vazkii.botania.client.core.helper.RenderHelper;
import vazkii.botania.client.fx.SparkleParticleData;
import vazkii.botania.common.block.BotaniaBlocks;
import vazkii.botania.common.handler.BotaniaSounds;
import vazkii.botania.common.lib.BotaniaTags;
import vazkii.botania.network.EffectType;
import vazkii.botania.network.clientbound.BotaniaEffectPacket;
import vazkii.botania.xplat.XplatAbstractions;
import vazkii.patchouli.api.IMultiblock;
import vazkii.patchouli.api.IStateMatcher;
import vazkii.patchouli.api.PatchouliAPI;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import net.minecraft.class_1297;
import net.minecraft.class_151;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1887;
import net.minecraft.class_1889;
import net.minecraft.class_1890;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2378;
import net.minecraft.class_238;
import net.minecraft.class_2470;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3419;
import net.minecraft.class_3829;
import net.minecraft.class_4587;

public class ManaEnchanterBlockEntity extends BotaniaBlockEntity implements ManaReceiver, SparkAttachable, Wandable, class_3829 {
	private static final String TAG_STAGE = "stage";
	private static final String TAG_STAGE_TICKS = "stageTicks";
	private static final String TAG_STAGE_3_END_TICKS = "stage3EndTicks";
	private static final String TAG_MANA_REQUIRED = "manaRequired";
	private static final String TAG_MANA = "mana";
	private static final String TAG_ITEM = "item";
	private static final String TAG_ENCHANTS = "enchantsToApply";
	private static final int CRAFT_EFFECT_EVENT = 0;

	private static final String[][] PATTERN = new String[][] {
			{
					"_P_______P_",
					"___________",
					"___________",
					"P_________P",
					"___________",
					"___________",
					"_P_______P_",
			},
			{
					"_F_______F_",
					"___________",
					"____F_F____",
					"F____L____F",
					"____F_F____",
					"___________",
					"_F_______F_",
			},
			{
					"___________",
					"____BBB____",
					"___B_B_B___",
					"___BB0BB___",
					"___B_B_B___",
					"____BBB____",
					"___________",
			}
	};

	private static final Supplier<IStateMatcher> OBSIDIAN_MATCHER = Suppliers.memoize(() -> PatchouliAPI.get().predicateMatcher(
			class_2246.field_10540,
			state -> state.method_27852(class_2246.field_10540) || state.method_27852(class_2246.field_22423)
	));

	public static final Supplier<IMultiblock> MULTIBLOCK = Suppliers.memoize(() -> PatchouliAPI.get().makeMultiblock(
			PATTERN,
			'P', BotaniaBlocks.manaPylon,
			'L', class_2246.field_10441,
			'B', OBSIDIAN_MATCHER.get(),
			'0', OBSIDIAN_MATCHER.get(),
			'F', PatchouliAPI.get().tagMatcher(BotaniaTags.Blocks.ENCHANTER_FLOWERS)
	));

	private static final Supplier<IMultiblock> FORMED_MULTIBLOCK = Suppliers.memoize(() -> PatchouliAPI.get().makeMultiblock(
			PATTERN,
			'P', BotaniaBlocks.manaPylon,
			'L', BotaniaBlocks.enchanter,
			'B', OBSIDIAN_MATCHER.get(),
			'0', OBSIDIAN_MATCHER.get(),
			'F', PatchouliAPI.get().predicateMatcher(BotaniaBlocks.whiteFlower, state -> state.method_26164(BotaniaTags.Blocks.ENCHANTER_FLOWERS))
	));

	public State stage = State.IDLE;
	public int stageTicks = 0;

	public int stage3EndTicks = 0;

	private int manaRequired = -1;
	private int mana = 0;

	public class_1799 itemToEnchant = class_1799.field_8037;
	private final List<class_1889> enchants = new ArrayList<>();

	private static final Map<class_2350.class_2351, class_2338[]> PYLON_LOCATIONS = new EnumMap<>(class_2350.class_2351.class);

	static {
		PYLON_LOCATIONS.put(class_2350.class_2351.field_11048, new class_2338[] { new class_2338(-5, 1, 0), new class_2338(5, 1, 0), new class_2338(-4, 1, 3), new class_2338(4, 1, 3), new class_2338(-4, 1, -3), new class_2338(4, 1, -3) });
		PYLON_LOCATIONS.put(class_2350.class_2351.field_11051, new class_2338[] { new class_2338(0, 1, -5), new class_2338(0, 1, 5), new class_2338(3, 1, -4), new class_2338(3, 1, 4), new class_2338(-3, 1, -4), new class_2338(-3, 1, 4) });
	}

	public ManaEnchanterBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaBlockEntities.ENCHANTER, pos, state);
	}

	@Override
	public boolean onUsedByWand(@Nullable class_1657 player, class_1799 wand, class_2350 side) {
		if (stage != State.IDLE || itemToEnchant.method_7960() || !itemToEnchant.method_7923()) {
			return false;
		}

		List<class_1542> items = field_11863.method_18467(class_1542.class, new class_238(field_11867.method_10263() - 2, field_11867.method_10264(), field_11867.method_10260() - 2, field_11867.method_10263() + 3, field_11867.method_10264() + 1, field_11867.method_10260() + 3));
		int count = items.size();

		if (count > 0 && !field_11863.field_9236) {
			for (class_1542 entity : items) {
				class_1799 item = entity.method_6983();
				if (item.method_31574(class_1802.field_8598)) {
					Map<class_1887, Integer> enchants = class_1890.method_8222(item);
					if (enchants.size() > 0) {
						class_1887 enchant = enchants.keySet().iterator().next();
						if (isEnchantmentValid(enchant)) {
							advanceStage();
							return true;
						}
					}
				}
			}
		}
		return false;
	}

	private void gatherEnchants() {
		if (!field_11863.field_9236 && stageTicks % 20 == 0) {
			List<class_1542> items = field_11863.method_18467(class_1542.class, new class_238(field_11867.method_10263() - 2, field_11867.method_10264(), field_11867.method_10260() - 2, field_11867.method_10263() + 3, field_11867.method_10264() + 1, field_11867.method_10260() + 3));
			boolean addedEnch = false;

			for (class_1542 entity : items) {
				class_1799 item = entity.method_6983();
				if (item.method_31574(class_1802.field_8598)) {
					Map<class_1887, Integer> enchants = class_1890.method_8222(item);
					if (enchants.size() > 0) {
						Map.Entry<class_1887, Integer> e = enchants.entrySet().iterator().next();
						class_1887 ench = e.getKey();
						int enchantLvl = e.getValue();
						if (!hasEnchantAlready(ench) && isEnchantmentValid(ench)) {
							this.enchants.add(new class_1889(ench, enchantLvl));
							field_11863.method_8396(null, field_11867, BotaniaSounds.ding, class_3419.field_15245, 1F, 1F);
							addedEnch = true;
							break;
						}
					}
				}
			}

			if (!addedEnch) {
				if (enchants.isEmpty()) {
					stage = State.IDLE;
				} else {
					advanceStage();
				}
			}
		}
	}

	private void gatherMana(class_2350.class_2351 axis) {
		if (manaRequired == -1) {
			manaRequired = 0;
			for (class_1889 data : enchants) {
				manaRequired += (int) (5000F * ((15 - Math.min(15, data.field_9093.method_8186().method_8197()))
						* 1.05F)
						* ((3F + data.field_9094 * data.field_9094)
								* 0.25F)
						* (0.9F + enchants.size() * 0.05F)
						* (data.field_9093.method_8193() ? 1.25F : 1F));
			}
		} else if (mana >= manaRequired) {
			manaRequired = 0;
			for (class_2338 offset : PYLON_LOCATIONS.get(axis)) {
				class_2586 te = field_11863.method_8321(field_11867.method_10081(offset));
				if (te instanceof PylonBlockEntity pylon) {
					pylon.activated = false;
				}
			}

			advanceStage();
		} else {
			ManaSpark spark = getAttachedSpark();
			if (spark != null) {
				var otherSparks = SparkHelper.getSparksAround(field_11863, field_11867.method_10263() + 0.5, field_11867.method_10264() + 0.5, field_11867.method_10260() + 0.5, spark.getNetwork());
				for (var otherSpark : otherSparks) {
					if (spark != otherSpark && otherSpark.getAttachedManaReceiver() instanceof ManaPool) {
						otherSpark.registerTransfer(spark);
					}
				}
			}
			if (stageTicks % 5 == 0) {
				sync();
			}
		}
	}

	public static void commonTick(class_1937 level, class_2338 worldPosition, class_2680 state, ManaEnchanterBlockEntity self) {
		class_2350.class_2351 axis = state.method_11654(BotaniaStateProperties.ENCHANTER_DIRECTION);

		for (class_2338 offset : PYLON_LOCATIONS.get(axis)) {
			class_2586 tile = level.method_8321(worldPosition.method_10081(offset));
			if (tile instanceof PylonBlockEntity pylon) {
				pylon.activated = self.stage == State.GATHER_MANA;
				if (self.stage == State.GATHER_MANA) {
					pylon.centerPos = worldPosition;
				}
			}
		}

		if (self.stage != State.IDLE) {
			self.stageTicks++;
		}

		if (level.field_9236) {
			return;
		}

		if (FORMED_MULTIBLOCK.get().validate(level, worldPosition.method_10074()) == null) {
			level.method_8501(worldPosition, class_2246.field_10441.method_9564());
			XplatAbstractions.INSTANCE.sendToNear(level, worldPosition, new BotaniaEffectPacket(EffectType.ENCHANTER_DESTROY,
					worldPosition.method_10263() + 0.5, worldPosition.method_10264() + 0.5, worldPosition.method_10260() + 0.5));
			level.method_8396(null, worldPosition, BotaniaSounds.enchanterFade, class_3419.field_15245, 1F, 1F);
		}

		switch (self.stage) {
			case GATHER_ENCHANTS -> self.gatherEnchants();
			case GATHER_MANA -> self.gatherMana(axis);
			case DO_ENCHANT -> { // Enchant
				if (self.stageTicks >= 100) {
					for (class_1889 data : self.enchants) {
						if (class_1890.method_8225(data.field_9093, self.itemToEnchant) == 0) {
							self.itemToEnchant.method_7978(data.field_9093, data.field_9094);
						}
					}

					self.enchants.clear();
					self.manaRequired = -1;
					self.mana = 0;

					level.method_8427(worldPosition, BotaniaBlocks.enchanter, CRAFT_EFFECT_EVENT, 0);
					self.advanceStage();
				}
			}
			case RESET -> { // Reset
				if (self.stageTicks >= 20) {
					self.advanceStage();
				}

			}
			default -> {}
		}
	}

	private void advanceStage() {
		switch (stage) {
			case IDLE -> stage = State.GATHER_ENCHANTS;
			case GATHER_ENCHANTS -> stage = State.GATHER_MANA;
			case GATHER_MANA -> stage = State.DO_ENCHANT;
			case DO_ENCHANT -> {
				stage = State.RESET;
				stage3EndTicks = stageTicks;
			}
			case RESET -> {
				stage = State.IDLE;
				stage3EndTicks = 0;
			}
		}

		stageTicks = 0;
		sync();
	}

	@Override
	public boolean method_11004(int event, int param) {
		switch (event) {
			case CRAFT_EFFECT_EVENT: {
				if (field_11863.field_9236) {
					for (int i = 0; i < 25; i++) {
						float red = (float) Math.random();
						float green = (float) Math.random();
						float blue = (float) Math.random();
						SparkleParticleData data = SparkleParticleData.sparkle((float) Math.random(), red, green, blue, 10);
						field_11863.method_8406(data, method_11016().method_10263() + Math.random() * 0.4 - 0.2, method_11016().method_10264(), method_11016().method_10260() + Math.random() * 0.4 - 0.2, 0, 0, 0);
					}
					field_11863.method_8486(field_11867.method_10263(), field_11867.method_10264(), field_11867.method_10260(), BotaniaSounds.enchanterEnchant, class_3419.field_15245, 1F, 1F, false);
				}
				return true;
			}
			default:
				return super.method_11004(event, param);
		}
	}

	@Override
	public class_1937 getManaReceiverLevel() {
		return method_10997();
	}

	@Override
	public class_2338 getManaReceiverPos() {
		return method_11016();
	}

	@Override
	public int getCurrentMana() {
		return mana;
	}

	@Override
	public boolean isFull() {
		return mana >= manaRequired;
	}

	@Override
	public void receiveMana(int mana) {
		this.mana = Math.min(manaRequired, this.mana + mana);
	}

	@Override
	public boolean canReceiveManaFromBursts() {
		return manaRequired > 0;
	}

	public void sync() {
		VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
	}

	@Override
	public void writePacketNBT(class_2487 cmp) {
		cmp.method_10569(TAG_MANA, mana);
		cmp.method_10569(TAG_MANA_REQUIRED, manaRequired);
		cmp.method_10569(TAG_STAGE, stage.ordinal());
		cmp.method_10569(TAG_STAGE_TICKS, stageTicks);
		cmp.method_10569(TAG_STAGE_3_END_TICKS, stage3EndTicks);

		class_2487 itemCmp = new class_2487();
		if (!itemToEnchant.method_7960()) {
			cmp.method_10566(TAG_ITEM, itemToEnchant.method_7953(itemCmp));
		}

		String enchStr = enchants.stream()
				.map(e -> class_2378.field_11160.method_10221(e.field_9093) + "=" + e.field_9094)
				.collect(Collectors.joining(","));
		cmp.method_10582(TAG_ENCHANTS, enchStr);
	}

	@Override
	public void readPacketNBT(class_2487 cmp) {
		mana = cmp.method_10550(TAG_MANA);
		manaRequired = cmp.method_10550(TAG_MANA_REQUIRED);
		stage = State.values()[cmp.method_10550(TAG_STAGE)];
		stageTicks = cmp.method_10550(TAG_STAGE_TICKS);
		stage3EndTicks = cmp.method_10550(TAG_STAGE_3_END_TICKS);

		class_2487 itemCmp = cmp.method_10562(TAG_ITEM);
		itemToEnchant = class_1799.method_7915(itemCmp);

		enchants.clear();
		String enchStr = cmp.method_10558(TAG_ENCHANTS);
		if (!enchStr.isEmpty()) {
			String[] enchTokens = enchStr.split(",");
			for (String token : enchTokens) {
				try {
					String[] entryTokens = token.split("=");
					int lvl = Integer.parseInt(entryTokens[1]);
					class_2378.field_11160.method_17966(new class_2960(entryTokens[0])).ifPresent(ench -> {
						enchants.add(new class_1889(ench, lvl));
					});
				} catch (class_151 ignored) {}
			}
		}
	}

	private boolean hasEnchantAlready(class_1887 enchant) {
		for (class_1889 data : enchants) {
			if (data.field_9093 == enchant) {
				return true;
			}
		}

		return false;
	}

	private boolean isEnchantmentValid(@Nullable class_1887 ench) {
		if (ench == null || !ench.method_8192(itemToEnchant)) {
			return false;
		}

		for (class_1889 data : enchants) {
			class_1887 otherEnch = data.field_9093;
			if (!ench.method_8188(otherEnch)) {
				return false;
			}
		}

		return true;
	}

	@Nullable
	public static class_2350.class_2351 canEnchanterExist(class_1937 world, class_2338 pos) {
		class_2470 rot = MULTIBLOCK.get().validate(world, pos.method_10074());
		if (rot == null) {
			return null;
		}

		return switch (rot) {
			case field_11467, field_11464 -> class_2350.class_2351.field_11051;
			case field_11463, field_11465 -> class_2350.class_2351.field_11048;
		};
	}

	@Override
	public boolean canAttachSpark(class_1799 stack) {
		return true;
	}

	@Override
	public ManaSpark getAttachedSpark() {
		List<class_1297> sparks = field_11863.method_8390(class_1297.class, new class_238(field_11867.method_10263(), field_11867.method_10264() + 1, field_11867.method_10260(), field_11867.method_10263() + 1, field_11867.method_10264() + 2, field_11867.method_10260() + 1), Predicates.instanceOf(ManaSpark.class));
		if (sparks.size() == 1) {
			class_1297 e = sparks.get(0);
			return (ManaSpark) e;
		}

		return null;
	}

	@Override
	public boolean areIncomingTranfersDone() {
		return stage == State.DO_ENCHANT;
	}

	@Override
	public int getAvailableSpaceForMana() {
		return Math.max(0, manaRequired - getCurrentMana());
	}

	@Override
	public void method_5448() {
		this.itemToEnchant = class_1799.field_8037;
		this.stage = State.IDLE;
	}

	public static class WandHud implements WandHUD {
		private final ManaEnchanterBlockEntity enchanter;

		public WandHud(ManaEnchanterBlockEntity enchanter) {
			this.enchanter = enchanter;
		}

		@Override
		public void renderHUD(class_4587 ms, class_310 mc) {
			if (enchanter.manaRequired > 0 && !enchanter.itemToEnchant.method_7960()) {
				int x = mc.method_22683().method_4486() / 2 + 20;
				int y = mc.method_22683().method_4502() / 2 - 8;

				RenderHelper.renderProgressPie(ms, x, y, (float) enchanter.mana / (float) enchanter.manaRequired,
						enchanter.itemToEnchant);
			}
		}
	}

	public enum State {
		IDLE,
		GATHER_ENCHANTS,
		GATHER_MANA,
		DO_ENCHANT,
		RESET
	}

}
