/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.block_entity;

import com.google.common.base.Suppliers;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.block.Wandable;
import vazkii.botania.api.recipe.ElvenTradeRecipe;
import vazkii.botania.api.state.BotaniaStateProperties;
import vazkii.botania.api.state.enums.AlfheimPortalState;
import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.common.advancements.AlfheimPortalBreadTrigger;
import vazkii.botania.common.advancements.AlfheimPortalTrigger;
import vazkii.botania.common.block.BotaniaBlocks;
import vazkii.botania.common.block.block_entity.mana.ManaPoolBlockEntity;
import vazkii.botania.common.block.mana.ManaPoolBlock;
import vazkii.botania.common.crafting.BotaniaRecipeTypes;
import vazkii.botania.common.lib.BotaniaTags;
import vazkii.botania.xplat.BotaniaConfig;
import vazkii.botania.xplat.XplatAbstractions;
import vazkii.patchouli.api.IMultiblock;
import vazkii.patchouli.api.IStateMatcher;
import vazkii.patchouli.api.PatchouliAPI;
import vazkii.patchouli.api.TriPredicate;

import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1860;
import net.minecraft.class_1922;
import net.minecraft.class_1927;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2378;
import net.minecraft.class_238;
import net.minecraft.class_2470;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_3222;
import net.minecraft.class_6862;
import net.minecraft.class_6880;

public class AlfheimPortalBlockEntity extends BotaniaBlockEntity implements Wandable {
	public static final Supplier<IMultiblock> MULTIBLOCK = Suppliers.memoize(() -> {
		record Matcher(class_6862<class_2248> tag, class_2350.class_2351 displayedRotation, class_2248 defaultBlock) implements IStateMatcher {
			@Override
			public class_2680 getDisplayedState(long ticks) {
				var blocks = StreamSupport.stream(class_2378.field_11146.method_40286(this.tag).spliterator(), false)
						.map(class_6880::comp_349)
						.toList();
				if (blocks.isEmpty()) {
					return class_2246.field_9987.method_9564();
				}

				class_2680 block = blocks.contains(defaultBlock)
						? defaultBlock.method_9564()
						: blocks.get((int) ((ticks / 20) % blocks.size())).method_9564();

				return block.method_28498(class_2741.field_12496)
						? block.method_11657(class_2741.field_12496, displayedRotation())
						: block;
			}

			@Override
			public TriPredicate<class_1922, class_2338, class_2680> getStatePredicate() {
				return (blockGetter, pos, state) -> state.method_26164(tag());
			}
		}
		var horizontal = new Matcher(BotaniaTags.Blocks.LIVINGWOOD_LOGS, class_2350.class_2351.field_11048, BotaniaBlocks.livingwoodLog);
		var vertical = new Matcher(BotaniaTags.Blocks.LIVINGWOOD_LOGS, class_2350.class_2351.field_11052, BotaniaBlocks.livingwoodLog);
		var horizontalGlimmer = new Matcher(BotaniaTags.Blocks.LIVINGWOOD_LOGS_GLIMMERING, class_2350.class_2351.field_11048, BotaniaBlocks.livingwoodLogGlimmering);
		var verticalGlimmer = new Matcher(BotaniaTags.Blocks.LIVINGWOOD_LOGS_GLIMMERING, class_2350.class_2351.field_11052, BotaniaBlocks.livingwoodLogGlimmering);

		return PatchouliAPI.get().makeMultiblock(
				new String[][] {
						{ "_", "w", "g", "w", "_" },
						{ "W", " ", " ", " ", "W" },
						{ "G", " ", " ", " ", "G" },
						{ "W", " ", " ", " ", "W" },
						{ "_", "w", "0", "w", "_" }
				},
				'W', vertical,
				'w', horizontal,
				'G', verticalGlimmer,
				'g', horizontalGlimmer,
				'0', BotaniaBlocks.alfPortal
		);
	});

	public static final int MANA_COST = 500;
	private static final String TAG_TICKS_OPEN = "ticksOpen";
	private static final String TAG_TICKS_SINCE_LAST_ITEM = "ticksSinceLastItem";
	private static final String TAG_STACK_COUNT = "stackCount";
	private static final String TAG_STACK = "portalStack";
	public static final String TAG_PORTAL_FLAG = "_elvenPortal";

	private final List<class_1799> stacksIn = new ArrayList<>();

	public int ticksOpen = 0;
	private int ticksSinceLastItem = 0;
	private boolean closeNow = false;
	private boolean explode = false;
	@Nullable
	private UUID breadPlayer = null;

	public AlfheimPortalBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaBlockEntities.ALF_PORTAL, pos, state);
	}

	public static void commonTick(class_1937 level, class_2338 worldPosition, class_2680 blockState, AlfheimPortalBlockEntity self) {
		if (blockState.method_11654(BotaniaStateProperties.ALFPORTAL_STATE) == AlfheimPortalState.OFF) {
			self.ticksOpen = 0;
			return;
		}
		AlfheimPortalState state = blockState.method_11654(BotaniaStateProperties.ALFPORTAL_STATE);
		AlfheimPortalState newState = self.getValidState();

		self.ticksOpen++;

		class_238 aabb = self.getPortalAABB();
		boolean open = self.ticksOpen > 60;
		XplatAbstractions.INSTANCE.fireElvenPortalUpdateEvent(self, aabb, open, self.stacksIn);

		if (self.ticksOpen > 60) {
			self.ticksSinceLastItem++;
			if (level.field_9236 && BotaniaConfig.client().elfPortalParticlesEnabled()) {
				self.blockParticle(state);
			}

			List<class_1542> items = level.method_18467(class_1542.class, aabb);
			if (!level.field_9236) {
				for (class_1542 item : items) {
					if (!item.method_5805()) {
						continue;
					}

					class_1799 stack = item.method_6983();
					if (XplatAbstractions.INSTANCE.itemFlagsComponent(item).alfPortalSpawned) {
						continue;
					}

					item.method_31472();
					if (self.validateItemUsage(item)) {
						self.addItem(stack);
					}
					self.ticksSinceLastItem = 0;
				}
			}

			if (!level.field_9236 && !self.stacksIn.isEmpty() && self.ticksSinceLastItem >= 4) {
				self.resolveRecipes();
			}
		}

		if (self.closeNow) {
			if (!level.field_9236) {
				level.method_8501(worldPosition, BotaniaBlocks.alfPortal.method_9564());
			}
			for (int i = 0; i < 36; i++) {
				self.blockParticle(state);
			}
			self.closeNow = false;
		} else if (newState != state) {
			if (newState == AlfheimPortalState.OFF) {
				for (int i = 0; i < 36; i++) {
					self.blockParticle(state);
				}
			}

			if (!level.field_9236) {
				level.method_8501(worldPosition, blockState.method_11657(BotaniaStateProperties.ALFPORTAL_STATE, newState));
			}
		} else if (self.explode) {
			level.method_8437(null, worldPosition.method_10263() + .5, worldPosition.method_10264() + 2.0, worldPosition.method_10260() + .5,
					3f, class_1927.class_4179.field_18686);
			self.explode = false;

			if (!level.field_9236 && self.breadPlayer != null) {
				class_1657 entity = level.method_18470(self.breadPlayer);
				if (entity instanceof class_3222 serverPlayer) {
					AlfheimPortalBreadTrigger.INSTANCE.trigger(serverPlayer, worldPosition);
				}
			}
			self.breadPlayer = null;
		}
	}

	private boolean validateItemUsage(class_1542 entity) {
		class_1799 inputStack = entity.method_6983();
		for (class_1860<?> recipe : BotaniaRecipeTypes.getRecipes(field_11863, BotaniaRecipeTypes.ELVEN_TRADE_TYPE).values()) {
			if (recipe instanceof ElvenTradeRecipe tradeRecipe && tradeRecipe.containsItem(inputStack)) {
				return true;
			}
		}
		if (inputStack.method_31574(class_1802.field_8229)) {
			//Don't teleport bread. (See also: #2403)
			explode = true;
			breadPlayer = entity.method_6978();
		}

		return false;
	}

	private void blockParticle(AlfheimPortalState state) {
		double dh, dy;

		// Pick one of the inner positions
		switch (field_11863.field_9229.method_43048(9)) {
			case 0 -> {
				dh = 0;
				dy = 1;
			}
			case 1 -> {
				dh = 0;
				dy = 2;
			}
			case 2 -> {
				dh = 0;
				dy = 3;
			}
			case 3 -> {
				dh = -1;
				dy = 1;
			}
			case 4 -> {
				dh = -1;
				dy = 2;
			}
			case 5 -> {
				dh = -1;
				dy = 3;
			}
			case 6 -> {
				dh = 1;
				dy = 1;
			}
			case 7 -> {
				dh = 1;
				dy = 2;
			}
			case 8 -> {
				dh = 1;
				dy = 3;
			}
			default -> throw new AssertionError();
		}
		double dx = state == AlfheimPortalState.ON_X ? 0 : dh;
		double dz = state == AlfheimPortalState.ON_Z ? 0 : dh;

		float motionMul = 0.2F;
		WispParticleData data = WispParticleData.wisp((float) (Math.random() * 0.15F + 0.1F), (float) (Math.random() * 0.25F), (float) (Math.random() * 0.5F + 0.5F), (float) (Math.random() * 0.25F));
		field_11863.method_8406(data, method_11016().method_10263() + dx, method_11016().method_10264() + dy, method_11016().method_10260() + dz, (float) (Math.random() - 0.5F) * motionMul, (float) (Math.random() - 0.5F) * motionMul, (float) (Math.random() - 0.5F) * motionMul);
	}

	@Override
	public boolean onUsedByWand(@Nullable class_1657 player, class_1799 stack, class_2350 side) {
		AlfheimPortalState state = method_11010().method_11654(BotaniaStateProperties.ALFPORTAL_STATE);
		if (state == AlfheimPortalState.OFF) {
			AlfheimPortalState newState = getValidState();
			if (newState != AlfheimPortalState.OFF) {
				field_11863.method_8501(method_11016(), method_11010().method_11657(BotaniaStateProperties.ALFPORTAL_STATE, newState));
				if (player instanceof class_3222 serverPlayer) {
					AlfheimPortalTrigger.INSTANCE.trigger(serverPlayer, serverPlayer.method_14220(), method_11016(), stack);
				}
				return true;
			}
		}

		return false;
	}

	private class_238 getPortalAABB() {
		class_238 aabb = new class_238(field_11867.method_10069(-1, 1, 0), field_11867.method_10069(2, 4, 1));
		if (method_11010().method_11654(BotaniaStateProperties.ALFPORTAL_STATE) == AlfheimPortalState.ON_X) {
			aabb = new class_238(field_11867.method_10069(0, 1, -1), field_11867.method_10069(1, 4, 2));
		}

		return aabb;
	}

	private void addItem(class_1799 stack) {
		int size = stack.method_7947();
		stack.method_7939(1);
		for (int i = 0; i < size; i++) {
			stacksIn.add(stack.method_7972());
		}
	}

	@SuppressWarnings("unchecked")
	public static Collection<ElvenTradeRecipe> elvenTradeRecipes(class_1937 world) {
		// By virtue of IRecipeType's type parameter,
		// we know all the recipes in the map must be ElvenTradeRecipe.
		// However, vanilla's signature on this method is dumb (should be Map<ResourceLocation, T>)
		return (Collection<ElvenTradeRecipe>) (Collection<?>) BotaniaRecipeTypes.getRecipes(world, BotaniaRecipeTypes.ELVEN_TRADE_TYPE).values();
	}

	private void resolveRecipes() {
		List<class_2338> pylons = locatePylons();
		for (class_1860<?> r : BotaniaRecipeTypes.getRecipes(field_11863, BotaniaRecipeTypes.ELVEN_TRADE_TYPE).values()) {
			if (!(r instanceof ElvenTradeRecipe recipe)) {
				continue;
			}
			Optional<List<class_1799>> match = recipe.match(stacksIn);
			if (match.isPresent()) {
				if (consumeMana(pylons, MANA_COST, false)) {
					List<class_1799> inputs = match.get();
					for (class_1799 stack : inputs) {
						stacksIn.remove(stack);
					}
					for (class_1799 output : recipe.getOutputs(inputs)) {
						spawnItem(output.method_7972());
					}
				}
				break;
			}
		}
	}

	private void spawnItem(class_1799 stack) {
		class_1542 item = new class_1542(field_11863, field_11867.method_10263() + 0.5, field_11867.method_10264() + 1.5, field_11867.method_10260() + 0.5, stack);
		XplatAbstractions.INSTANCE.itemFlagsComponent(item).alfPortalSpawned = true;
		field_11863.method_8649(item);
		ticksSinceLastItem = 0;
	}

	@Override
	public void method_11007(class_2487 cmp) {
		super.method_11007(cmp);

		cmp.method_10569(TAG_STACK_COUNT, stacksIn.size());
		int i = 0;
		for (class_1799 stack : stacksIn) {
			class_2487 stackcmp = stack.method_7953(new class_2487());
			cmp.method_10566(TAG_STACK + i, stackcmp);
			i++;
		}
	}

	@Override
	public void method_11014(@NotNull class_2487 cmp) {
		super.method_11014(cmp);

		int count = cmp.method_10550(TAG_STACK_COUNT);
		stacksIn.clear();
		for (int i = 0; i < count; i++) {
			class_2487 stackcmp = cmp.method_10562(TAG_STACK + i);
			class_1799 stack = class_1799.method_7915(stackcmp);
			stacksIn.add(stack);
		}
	}

	@Override
	public void writePacketNBT(class_2487 cmp) {
		cmp.method_10569(TAG_TICKS_OPEN, ticksOpen);
		cmp.method_10569(TAG_TICKS_SINCE_LAST_ITEM, ticksSinceLastItem);
	}

	@Override
	public void readPacketNBT(class_2487 cmp) {
		ticksOpen = cmp.method_10550(TAG_TICKS_OPEN);
		ticksSinceLastItem = cmp.method_10550(TAG_TICKS_SINCE_LAST_ITEM);
	}

	private AlfheimPortalState getValidState() {
		class_2470 rot = MULTIBLOCK.get().validate(field_11863, method_11016());
		if (rot == null) {
			return AlfheimPortalState.OFF;
		}

		lightPylons();
		return switch (rot) {
			case field_11467, field_11464 -> AlfheimPortalState.ON_Z;
			case field_11463, field_11465 -> AlfheimPortalState.ON_X;
		};
	}

	public List<class_2338> locatePylons() {
		int range = 5;

		return class_2338.method_20437(method_11016().method_10069(-range, -range, -range), method_11016().method_10069(range, range, range))
				.filter(field_11863::method_22340)
				.filter(p -> field_11863.method_8320(p).method_27852(BotaniaBlocks.naturaPylon) && field_11863.method_8320(p.method_10074()).method_26204() instanceof ManaPoolBlock)
				.map(class_2338::method_10062)
				.collect(Collectors.toList());
	}

	public void lightPylons() {
		if (ticksOpen < 50) {
			return;
		}

		List<class_2338> pylons = locatePylons();
		for (class_2338 pos : pylons) {
			class_2586 tile = field_11863.method_8321(pos);
			if (tile instanceof PylonBlockEntity pylon) {
				pylon.activated = true;
				pylon.centerPos = method_11016();
			}
		}

		if (ticksOpen == 50) {
			consumeMana(pylons, 200000, true);
		}
	}

	public boolean consumeMana(List<class_2338> pylons, int totalCost, boolean close) {
		List<ManaPoolBlockEntity> consumePools = new ArrayList<>();
		int consumed = 0;

		if (pylons.size() < 2) {
			closeNow = true;
			return false;
		}

		int costPer = Math.max(1, totalCost / pylons.size());
		int expectedConsumption = costPer * pylons.size();

		for (class_2338 pos : pylons) {
			class_2586 tile = field_11863.method_8321(pos);
			if (tile instanceof PylonBlockEntity pylon) {
				pylon.activated = true;
				pylon.centerPos = method_11016();
			}

			tile = field_11863.method_8321(pos.method_10074());
			if (tile instanceof ManaPoolBlockEntity pool) {
				if (pool.getCurrentMana() < costPer) {
					closeNow = closeNow || close;
					return false;
				} else if (!field_11863.field_9236) {
					consumePools.add(pool);
					consumed += costPer;
				}
			}
		}

		if (consumed >= expectedConsumption) {
			for (ManaPoolBlockEntity pool : consumePools) {
				pool.receiveMana(-costPer);
			}
			return true;
		}

		return false;
	}
}
