/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.client.patchouli.processor;

import com.google.common.collect.ImmutableList;
import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.recipe.ManaInfusionRecipe;
import vazkii.botania.client.patchouli.PatchouliUtils;
import vazkii.botania.common.crafting.BotaniaRecipeTypes;
import vazkii.patchouli.api.IComponentProcessor;
import vazkii.patchouli.api.IVariable;
import vazkii.patchouli.api.IVariableProvider;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.class_124;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_2960;

public class ManaInfusionProcessor implements IComponentProcessor {
	private List<ManaInfusionRecipe> recipes;
	private boolean hasCustomHeading;

	@Override
	public void setup(IVariableProvider variables) {
		if (variables.has("recipes") && variables.has("group")) {
			BotaniaAPI.LOGGER.warn("Mana infusion template has both 'recipes' and 'group', ignoring 'recipes'");
		}

		ImmutableList.Builder<ManaInfusionRecipe> builder = ImmutableList.builder();
		if (variables.has("group")) {
			String group = variables.get("group").asString();
			builder.addAll(PatchouliUtils.getRecipeGroup(BotaniaRecipeTypes.MANA_INFUSION_TYPE, group));
		} else {
			for (IVariable s : variables.get("recipes").asListOrSingleton()) {
				ManaInfusionRecipe recipe = PatchouliUtils.getRecipe(BotaniaRecipeTypes.MANA_INFUSION_TYPE, new class_2960(s.asString()));
				if (recipe != null) {
					builder.add(recipe);
				}
			}
		}

		this.recipes = builder.build();
		this.hasCustomHeading = variables.has("heading");
	}

	@Override
	public IVariable process(String key) {
		if (recipes.isEmpty()) {
			return null;
		}
		switch (key) {
			case "heading":
				if (!hasCustomHeading) {
					return IVariable.from(recipes.get(0).method_8110().method_7964());
				}
				return null;
			case "input":
				return PatchouliUtils.interweaveIngredients(recipes.stream().map(r -> r.method_8117().get(0)).collect(Collectors.toList()));
			case "output":
				return IVariable.wrapList(recipes.stream().map(ManaInfusionRecipe::method_8110).map(IVariable::from).collect(Collectors.toList()));
			case "catalyst":
				return IVariable.wrapList(recipes.stream().map(ManaInfusionRecipe::getRecipeCatalyst)
						.flatMap(ingr -> {
							if (ingr == null) {
								return Stream.of(class_1799.field_8037);
							}
							return ingr.getDisplayedStacks().stream();
						})
						.map(IVariable::from)
						.collect(Collectors.toList()));
			case "mana":
				return IVariable.wrapList(recipes.stream().mapToInt(ManaInfusionRecipe::getManaToConsume).mapToObj(IVariable::wrap).collect(Collectors.toList()));
			case "drop":
				class_2561 q = class_2561.method_43470("(?)").method_27692(class_124.field_1067);
				return IVariable.from(class_2561.method_43471("botaniamisc.drop").method_27693(" ").method_10852(q));
			case "dropTip2":
			case "dropTip1":
				class_2561 drop = class_2561.method_43472("key.drop").method_27692(class_124.field_1060);
				return IVariable.from(class_2561.method_43469("botaniamisc." + key, drop));
			case "dropTip3":
				return IVariable.from(class_2561.method_43471("botaniamisc." + key));
		}
		return null;
	}
}
