/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.client.patchouli;

import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.common.crafting.BotaniaRecipeTypes;
import vazkii.botania.xplat.XplatAbstractions;
import vazkii.patchouli.api.IVariable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import net.minecraft.class_1263;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_1860;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3956;

public class PatchouliUtils {
	private static boolean crafttweakerInfoNote = false;

	/**
	 * Gets a recipe of a specified type and ID, and replaces the namespace
	 * with {@code crafttweaker} to try and find replacements if the recipe doesn't exist.
	 *
	 * If the recipe has no replacement, it will be logged.
	 */
	public static <T extends class_1860<C>, C extends class_1263> T getRecipe(class_3956<T> type, class_2960 id) {
		@SuppressWarnings("unchecked")
		Map<class_2960, T> map = (Map<class_2960, T>) BotaniaRecipeTypes.getRecipes(class_310.method_1551().field_1687, type);
		T r = map.get(id);
		if (r != null) {
			return r;
		}
		r = map.get(new class_2960("crafttweaker", id.method_12832()));
		if (r != null) {
			return r;
		}
		r = map.get(new class_2960("crafttweaker", "autogenerated/" + id.method_12836() + "." + id.method_12832()));
		if (r != null) {
			return r;
		}

		BotaniaAPI.LOGGER.warn("Template references nonexistent recipe {} of type {}", id, type);
		if (!crafttweakerInfoNote) {
			crafttweakerInfoNote = true;
			if (XplatAbstractions.INSTANCE.isModLoaded("crafttweaker")) {
				BotaniaAPI.LOGGER.info("""
					To add a recipe that replaces a builtin recipe with CT,\s
					add one with the same type, named the same as the path of the missing recipe.
					eg. for recipe {}, add a recipe named "{}".""", id, id.method_12832());
			}
		}
		return null;
	}

	/**
	 * Get all recipes of the specified type that belong to the specified recipe group.
	 */
	public static <T extends class_1860<C>, C extends class_1263> List<T> getRecipeGroup(class_3956<T> type, String group) {
		@SuppressWarnings("unchecked")
		Map<class_2960, T> map = (Map<class_2960, T>) BotaniaRecipeTypes.getRecipes(class_310.method_1551().field_1687, type);
		List<T> list = new ArrayList<>();
		for (T value : map.values()) {
			if (group.equals(value.method_8112())) {
				list.add(value);
			}
		}
		if (list.isEmpty()) {
			BotaniaAPI.LOGGER.warn("Template references empty group {} of recipe type {}", group, type);
		}
		return list;
	}

	/**
	 * Combines the ingredients, returning the first matching stack of each, then the second stack of each, etc.
	 * looping back ingredients that run out of matched stacks, until the ingredients reach the length
	 * of the longest ingredient in the recipe set.
	 *
	 * @param ingredients           List of ingredients in the specific slot
	 * @param longestIngredientSize Longest ingredient in the entire recipe
	 * @return Serialized Patchouli ingredient string
	 */
	public static IVariable interweaveIngredients(List<class_1856> ingredients, int longestIngredientSize) {
		if (ingredients.size() == 1) {
			return IVariable.wrapList(Arrays.stream(ingredients.get(0).method_8105()).map(IVariable::from).collect(Collectors.toList()));
		}

		class_1799[] empty = { class_1799.field_8037 };
		List<class_1799[]> stacks = new ArrayList<>();
		for (class_1856 ingredient : ingredients) {
			if (ingredient != null && !ingredient.method_8103()) {
				stacks.add(ingredient.method_8105());
			} else {
				stacks.add(empty);
			}
		}
		List<IVariable> list = new ArrayList<>(stacks.size() * longestIngredientSize);
		for (int i = 0; i < longestIngredientSize; i++) {
			for (class_1799[] stack : stacks) {
				list.add(IVariable.from(stack[i % stack.length]));
			}
		}
		return IVariable.wrapList(list);
	}

	/**
	 * Overload of the method above that uses the provided list's longest ingredient size.
	 */
	public static IVariable interweaveIngredients(List<class_1856> ingredients) {
		return interweaveIngredients(ingredients, ingredients.stream().mapToInt(ingr -> ingr.method_8105().length).max().orElse(1));
	}
}
