/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.client.fx;
/*
 * This class is derived from the Mekanism project.
 * See ALTERNATE_LICENSES.txt.
 * Imported October 2021 and heavily modified.
 */

import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import org.apache.commons.lang3.tuple.Pair;

import vazkii.botania.client.core.helper.RenderHelper;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Set;
import net.minecraft.class_1159;
import net.minecraft.class_1937;
import net.minecraft.class_243;
import net.minecraft.class_310;
import net.minecraft.class_4184;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4597;
import net.minecraft.class_4597.class_4598;
import net.minecraft.class_4599;

// todo(williewillus) integrate this properly into the particle system
public class BoltRenderer {

	public static final BoltRenderer INSTANCE = new BoltRenderer();
	/** Amount of times per tick we refresh. 3 implies 60 Hz. */
	private static final float REFRESH_TIME = 3F;
	/** After there are no more bolts in an emitter, how much longer in ticks it should stay alive. */
	private static final double LIFETIME_AFTER_LAST_BOLT = 100;

	private Timestamp refreshTimestamp = Timestamp.ZERO;

	private final Random random = new Random();

	private final List<BoltEmitter> boltEmitters = new LinkedList<>();

	public static void onWorldRenderLast(class_4184 camera, float partialTicks, class_4587 ps, class_4599 buffers) {
		ps.method_22903();
		// here we translate based on the inverse position of the client viewing camera to get back to 0, 0, 0
		class_243 camVec = camera.method_19326();
		ps.method_22904(-camVec.field_1352, -camVec.field_1351, -camVec.field_1350);
		var bufferSource = buffers.method_23000();
		BoltRenderer.INSTANCE.render(partialTicks, ps, bufferSource);
		bufferSource.method_22994(RenderHelper.LIGHTNING);
		ps.method_22909();
	}

	public void render(float partialTicks, class_4587 matrixStack, class_4597 buffers) {
		class_4588 buffer = buffers.getBuffer(RenderHelper.LIGHTNING);
		class_1159 matrix = matrixStack.method_23760().method_23761();
		Timestamp timestamp = new Timestamp(class_310.method_1551().field_1687.method_8510(), partialTicks);
		boolean refresh = timestamp.isPassed(refreshTimestamp, (1 / REFRESH_TIME));
		if (refresh) {
			refreshTimestamp = timestamp;
		}

		for (Iterator<BoltEmitter> iter = boltEmitters.iterator(); iter.hasNext();) {
			BoltEmitter emitter = iter.next();
			emitter.renderTick(timestamp, refresh, matrix, buffer);
			if (emitter.shouldRemove(timestamp)) {
				iter.remove();
			}
		}
	}

	public void add(class_1937 level, BoltParticleOptions options, float partialTicks) {
		if (!level.field_9236) {
			return;
		}
		var emitter = new BoltEmitter(options);
		Timestamp timestamp = new Timestamp(level.method_8510(), partialTicks);
		if ((!emitter.options.getSpawnFunction().isConsecutive() || emitter.bolts.isEmpty()) && timestamp.isPassed(emitter.lastBoltTimestamp, emitter.lastBoltDelay)) {
			emitter.addBolt(new BoltInstance(options, timestamp), timestamp);
		}
		emitter.lastUpdateTimestamp = timestamp;
		boltEmitters.add(emitter);
	}

	public class BoltEmitter {

		private final Set<BoltInstance> bolts = new ObjectOpenHashSet<>();
		private final BoltParticleOptions options;
		private Timestamp lastBoltTimestamp = Timestamp.ZERO;
		private Timestamp lastUpdateTimestamp = Timestamp.ZERO;
		private double lastBoltDelay;

		public BoltEmitter(BoltParticleOptions options) {
			this.options = options;
		}

		private void addBolt(BoltInstance instance, Timestamp timestamp) {
			bolts.add(instance);
			lastBoltDelay = instance.options.getSpawnFunction().getSpawnDelay(random);
			lastBoltTimestamp = timestamp;
		}

		public void renderTick(Timestamp timestamp, boolean refresh, class_1159 matrix, class_4588 buffer) {
			// tick our bolts based on the refresh rate, removing if they're now finished
			if (refresh) {
				bolts.removeIf(bolt -> bolt.tick(timestamp));
			}
			if (bolts.isEmpty() && options != null && options.getSpawnFunction().isConsecutive()) {
				addBolt(new BoltInstance(options, timestamp), timestamp);
			}
			bolts.forEach(bolt -> bolt.render(matrix, buffer, timestamp));
		}

		public boolean shouldRemove(Timestamp timestamp) {
			return bolts.isEmpty() && timestamp.isPassed(lastUpdateTimestamp, LIFETIME_AFTER_LAST_BOLT);
		}
	}

	private static class BoltInstance {

		private final BoltParticleOptions options;
		private final List<BoltParticleOptions.BoltQuads> renderQuads;
		private final Timestamp createdTimestamp;

		public BoltInstance(BoltParticleOptions options, Timestamp timestamp) {
			this.options = options;
			this.renderQuads = options.generate();
			this.createdTimestamp = timestamp;
		}

		public void render(class_1159 matrix, class_4588 buffer, Timestamp timestamp) {
			float lifeScale = timestamp.subtract(createdTimestamp).value() / options.getLifespan();
			Pair<Integer, Integer> bounds = options.getFadeFunction().getRenderBounds(renderQuads.size(), lifeScale);
			for (int i = bounds.getLeft(); i < bounds.getRight(); i++) {
				renderQuads.get(i).getVecs().forEach(v -> buffer.method_22918(matrix, (float) v.field_1352, (float) v.field_1351, (float) v.field_1350)
						.method_22915(options.getColor().method_4953(), options.getColor().method_4956(), options.getColor().method_4957(), options.getColor().method_23853())
						.method_1344());
			}
		}

		public boolean tick(Timestamp timestamp) {
			return timestamp.isPassed(createdTimestamp, options.getLifespan());
		}
	}

	private static class Timestamp {

		public static final Timestamp ZERO = new Timestamp(0, 0);
		private final long ticks;
		private final float partial;

		public Timestamp(long ticks, float partial) {
			this.ticks = ticks;
			this.partial = partial;
		}

		public Timestamp subtract(Timestamp other) {
			long newTicks = ticks - other.ticks;
			float newPartial = partial - other.partial;
			if (newPartial < 0) {
				newPartial += 1;
				newTicks -= 1;
			}
			return new Timestamp(newTicks, newPartial);
		}

		public float value() {
			return ticks + partial;
		}

		public boolean isPassed(Timestamp prev, double duration) {
			long ticksPassed = ticks - prev.ticks;
			if (ticksPassed > duration) {
				return true;
			}
			duration -= ticksPassed;
			if (duration >= 1) {
				return false;
			}
			return (partial - prev.partial) >= duration;
		}
	}
}
