/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.tile.mana;

import com.google.common.base.Predicates;
import com.mojang.blaze3d.systems.RenderSystem;

import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1297;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1767;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3000;
import net.minecraft.class_310;
import net.minecraft.class_3419;
import net.minecraft.class_4587;
import org.lwjgl.opengl.GL11;

import vazkii.botania.api.BotaniaAPIClient;
import vazkii.botania.api.internal.VanillaPacketDispatcher;
import vazkii.botania.api.item.IManaDissolvable;
import vazkii.botania.api.mana.*;
import vazkii.botania.api.mana.spark.ISparkAttachable;
import vazkii.botania.api.mana.spark.ISparkEntity;
import vazkii.botania.api.recipe.IManaInfusionRecipe;
import vazkii.botania.client.core.handler.HUDHandler;
import vazkii.botania.client.core.helper.RenderHelper;
import vazkii.botania.client.fx.SparkleParticleData;
import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.common.Botania;
import vazkii.botania.common.block.ModBlocks;
import vazkii.botania.common.block.mana.BlockPool;
import vazkii.botania.common.block.tile.ModTiles;
import vazkii.botania.common.block.tile.TileMod;
import vazkii.botania.common.core.handler.ConfigHandler;
import vazkii.botania.common.core.handler.ManaNetworkHandler;
import vazkii.botania.common.core.handler.ModSounds;
import vazkii.botania.common.core.helper.Vector3;
import vazkii.botania.common.crafting.ModRecipeTypes;
import vazkii.botania.common.item.ItemManaTablet;
import vazkii.botania.common.item.ModItems;
import vazkii.botania.mixin.AccessorItemEntity;

import javax.annotation.Nonnull;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

public class TilePool extends TileMod implements IManaPool, IKeyLocked, ISparkAttachable, IThrottledPacket, class_3000 {
	public static final int PARTICLE_COLOR = 0x00C6FF;
	public static final int MAX_MANA = 1000000;
	private static final int MAX_MANA_DILLUTED = 10000;

	private static final String TAG_MANA = "mana";
	private static final String TAG_OUTPUTTING = "outputting";
	private static final String TAG_COLOR = "color";
	private static final String TAG_MANA_CAP = "manaCap";
	private static final String TAG_CAN_ACCEPT = "canAccept";
	private static final String TAG_CAN_SPARE = "canSpare";
	private static final String TAG_FRAGILE = "fragile";
	private static final String TAG_INPUT_KEY = "inputKey";
	private static final String TAG_OUTPUT_KEY = "outputKey";
	private static final int CRAFT_EFFECT_EVENT = 0;
	private static final int CHARGE_EFFECT_EVENT = 1;

	private boolean outputting = false;

	public class_1767 color = class_1767.field_7952;
	private int mana;

	public int manaCap = -1;
	private int soundTicks = 0;
	private boolean canAccept = true;
	private boolean canSpare = true;
	public boolean fragile = false;
	boolean isDoingTransfer = false;
	int ticksDoingTransfer = 0;

	private String inputKey = "";
	private final String outputKey = "";

	private int ticks = 0;
	private boolean sendPacket = false;

	public TilePool() {
		super(ModTiles.POOL);
	}

	@Override
	public boolean isFull() {
		class_2248 blockBelow = field_11863.method_8320(field_11867.method_10074()).method_26204();
		return blockBelow != ModBlocks.manaVoid && getCurrentMana() >= manaCap;
	}

	@Override
	public void receiveMana(int mana) {
		int old = this.mana;
		this.mana = Math.max(0, Math.min(getCurrentMana() + mana, manaCap));
		if (old != this.mana) {
			method_5431();
			markDispatchable();
		}
	}

	@Override
	public void method_11012() {
		super.method_11012();
		ManaNetworkCallback.removePool(this);
	}

	public static int calculateComparatorLevel(int mana, int max) {
		int val = (int) ((double) mana / (double) max * 15.0);
		if (mana > 0) {
			val = Math.max(val, 1);
		}
		return val;
	}

	public static List<IManaInfusionRecipe> manaInfusionRecipes(class_1937 world) {
		return ModRecipeTypes.getRecipes(world, ModRecipeTypes.MANA_INFUSION_TYPE).values().stream()
				.filter(r -> r instanceof IManaInfusionRecipe)
				.map(r -> (IManaInfusionRecipe) r)
				.collect(Collectors.toList());
	}

	public IManaInfusionRecipe getMatchingRecipe(@Nonnull class_1799 stack, @Nonnull class_2680 state) {
		List<IManaInfusionRecipe> matchingNonCatRecipes = new ArrayList<>();
		List<IManaInfusionRecipe> matchingCatRecipes = new ArrayList<>();

		for (IManaInfusionRecipe recipe : manaInfusionRecipes(field_11863)) {
			if (recipe.matches(stack)) {
				if (recipe.getCatalyst() == null) {
					matchingNonCatRecipes.add(recipe);
				} else if (recipe.getCatalyst() == state) {
					matchingCatRecipes.add(recipe);
				}
			}
		}

		// Recipes with matching catalyst take priority above recipes with no catalyst specified
		return !matchingCatRecipes.isEmpty() ? matchingCatRecipes.get(0) : !matchingNonCatRecipes.isEmpty() ? matchingNonCatRecipes.get(0) : null;
	}

	public boolean collideEntityItem(class_1542 item) {
		if (field_11863.field_9236 || !item.method_5805() || item.method_6983().method_7960()) {
			return false;
		}

		class_1799 stack = item.method_6983();

		if (stack.method_7909() instanceof IManaDissolvable) {
			((IManaDissolvable) stack.method_7909()).onDissolveTick(this, stack, item);
		}

		int age = ((AccessorItemEntity) item).getAge();
		if (age > 100 && age < 130) {
			return false;
		}

		IManaInfusionRecipe recipe = getMatchingRecipe(stack, field_11863.method_8320(field_11867.method_10074()));

		if (recipe != null) {
			int mana = recipe.getManaToConsume();
			if (getCurrentMana() >= mana) {
				receiveMana(-mana);

				stack.method_7934(1);
				item.method_24830(false); //Force entity collision update to run every tick if crafting is in progress

				class_1799 output = recipe.method_8110().method_7972();
				class_1542 outputItem = new class_1542(field_11863, field_11867.method_10263() + 0.5, field_11867.method_10264() + 1.5, field_11867.method_10260() + 0.5, output);
				((AccessorItemEntity) outputItem).setAge(105);
				field_11863.method_8649(outputItem);

				craftingFanciness();
				return true;
			}
		}

		return false;
	}

	private void craftingFanciness() {
		if (soundTicks == 0) {
			field_11863.method_8396(null, field_11867, ModSounds.manaPoolCraft, class_3419.field_15245, 0.4F, 4F);
			soundTicks = 6;
		}

		field_11863.method_8427(method_11016(), method_11010().method_26204(), CRAFT_EFFECT_EVENT, 0);
	}

	@Override
	public boolean method_11004(int event, int param) {
		switch (event) {
		case CRAFT_EFFECT_EVENT: {
			if (field_11863.field_9236) {
				for (int i = 0; i < 25; i++) {
					float red = (float) Math.random();
					float green = (float) Math.random();
					float blue = (float) Math.random();
					SparkleParticleData data = SparkleParticleData.sparkle((float) Math.random(), red, green, blue, 10);
					field_11863.method_8406(data, field_11867.method_10263() + 0.5 + Math.random() * 0.4 - 0.2, field_11867.method_10264() + 0.75, field_11867.method_10260() + 0.5 + Math.random() * 0.4 - 0.2, 0, 0, 0);
				}
			}

			return true;
		}
		case CHARGE_EFFECT_EVENT: {
			if (field_11863.field_9236) {
				if (ConfigHandler.COMMON.chargingAnimationEnabled.getValue()) {
					boolean outputting = param == 1;
					Vector3 itemVec = Vector3.fromBlockPos(field_11867).add(0.5, 0.5 + Math.random() * 0.3, 0.5);
					Vector3 tileVec = Vector3.fromBlockPos(field_11867).add(0.2 + Math.random() * 0.6, 0, 0.2 + Math.random() * 0.6);
					Botania.proxy.lightningFX(outputting ? tileVec : itemVec,
							outputting ? itemVec : tileVec, 80, field_11863.field_9229.nextLong(), 0x4400799c, 0x4400C6FF);
				}
			}
			return true;
		}
		default:
			return super.method_11004(event, param);
		}
	}

	@Override
	public void method_16896() {
		if (manaCap == -1) {
			manaCap = ((BlockPool) method_11010().method_26204()).variant == BlockPool.Variant.DILUTED ? MAX_MANA_DILLUTED : MAX_MANA;
		}

		if (!ManaNetworkHandler.instance.isPoolIn(this) && !method_11015()) {
			ManaNetworkCallback.addPool(this);
		}

		if (field_11863.field_9236) {
			double particleChance = 1F - (double) getCurrentMana() / (double) manaCap * 0.1;
			if (Math.random() > particleChance) {
				float red = (PARTICLE_COLOR >> 16 & 0xFF) / 255F;
				float green = (PARTICLE_COLOR >> 8 & 0xFF) / 255F;
				float blue = (PARTICLE_COLOR & 0xFF) / 255F;
				WispParticleData data = WispParticleData.wisp((float) Math.random() / 3F, red, green, blue, 2F);
				field_11863.method_8406(data, field_11867.method_10263() + 0.3 + Math.random() * 0.5, field_11867.method_10264() + 0.6 + Math.random() * 0.25, field_11867.method_10260() + Math.random(), 0, (float) Math.random() / 25F, 0);
			}
			return;
		}

		boolean wasDoingTransfer = isDoingTransfer;
		isDoingTransfer = false;

		if (soundTicks > 0) {
			soundTicks--;
		}

		if (sendPacket && ticks % 10 == 0) {
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
			sendPacket = false;
		}

		List<class_1542> items = field_11863.method_18467(class_1542.class, new class_238(field_11867, field_11867.method_10069(1, 1, 1)));
		for (class_1542 item : items) {
			if (!item.method_5805()) {
				continue;
			}

			class_1799 stack = item.method_6983();
			if (!stack.method_7960() && stack.method_7909() instanceof IManaItem) {
				IManaItem mana = (IManaItem) stack.method_7909();
				if (outputting && mana.canReceiveManaFromPool(stack, this) || !outputting && mana.canExportManaToPool(stack, this)) {
					boolean didSomething = false;

					int bellowCount = 0;
					if (outputting) {
						for (class_2350 dir : class_2350.class_2353.field_11062) {
							class_2586 tile = field_11863.method_8321(field_11867.method_10093(dir));
							if (tile instanceof TileBellows && ((TileBellows) tile).getLinkedTile() == this) {
								bellowCount++;
							}
						}
					}
					int transfRate = 1000 * (bellowCount + 1);

					if (outputting) {
						if (canSpare) {
							if (getCurrentMana() > 0 && mana.getMana(stack) < mana.getMaxMana(stack)) {
								didSomething = true;
							}

							int manaVal = Math.min(transfRate, Math.min(getCurrentMana(), mana.getMaxMana(stack) - mana.getMana(stack)));
							mana.addMana(stack, manaVal);
							receiveMana(-manaVal);
						}
					} else {
						if (canAccept) {
							if (mana.getMana(stack) > 0 && !isFull()) {
								didSomething = true;
							}

							int manaVal = Math.min(transfRate, Math.min(manaCap - getCurrentMana(), mana.getMana(stack)));
							mana.addMana(stack, -manaVal);
							receiveMana(manaVal);
						}
					}

					if (didSomething) {
						if (ConfigHandler.COMMON.chargingAnimationEnabled.getValue() && field_11863.field_9229.nextInt(20) == 0) {
							field_11863.method_8427(method_11016(), method_11010().method_26204(), CHARGE_EFFECT_EVENT, outputting ? 1 : 0);
						}
						isDoingTransfer = outputting;
					}
				}
			}
		}

		if (isDoingTransfer) {
			ticksDoingTransfer++;
		} else {
			ticksDoingTransfer = 0;
			if (wasDoingTransfer) {
				VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
			}
		}

		ticks++;
	}

	@Override
	public void writePacketNBT(class_2487 cmp) {
		cmp.method_10569(TAG_MANA, mana);
		cmp.method_10556(TAG_OUTPUTTING, outputting);
		cmp.method_10569(TAG_COLOR, color.method_7789());

		cmp.method_10569(TAG_MANA_CAP, manaCap);
		cmp.method_10556(TAG_CAN_ACCEPT, canAccept);
		cmp.method_10556(TAG_CAN_SPARE, canSpare);
		cmp.method_10556(TAG_FRAGILE, fragile);

		cmp.method_10582(TAG_INPUT_KEY, inputKey);
		cmp.method_10582(TAG_OUTPUT_KEY, outputKey);
	}

	@Override
	public void readPacketNBT(class_2487 cmp) {
		mana = cmp.method_10550(TAG_MANA);
		outputting = cmp.method_10577(TAG_OUTPUTTING);
		color = class_1767.method_7791(cmp.method_10550(TAG_COLOR));

		if (cmp.method_10545(TAG_MANA_CAP)) {
			manaCap = cmp.method_10550(TAG_MANA_CAP);
		}
		if (cmp.method_10545(TAG_CAN_ACCEPT)) {
			canAccept = cmp.method_10577(TAG_CAN_ACCEPT);
		}
		if (cmp.method_10545(TAG_CAN_SPARE)) {
			canSpare = cmp.method_10577(TAG_CAN_SPARE);
		}
		fragile = cmp.method_10577(TAG_FRAGILE);

		if (cmp.method_10545(TAG_INPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_INPUT_KEY);
		}
		if (cmp.method_10545(TAG_OUTPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_OUTPUT_KEY);
		}

	}

	public void onWanded(class_1657 player) {
		if (player == null || player.method_5715()) {
			outputting = !outputting;
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
		}
	}

	@Environment(EnvType.CLIENT)
	public void renderHUD(class_4587 ms, class_310 mc) {
		class_1799 pool = new class_1799(method_11010().method_26204());
		String name = pool.method_7964().getString();
		int color = 0x4444FF;
		BotaniaAPIClient.instance().drawSimpleManaHUD(ms, color, getCurrentMana(), manaCap, name);

		int x = class_310.method_1551().method_22683().method_4486() / 2 - 11;
		int y = class_310.method_1551().method_22683().method_4502() / 2 + 30;

		int u = outputting ? 22 : 0;
		int v = 38;

		RenderSystem.enableBlend();
		RenderSystem.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);

		mc.method_1531().method_22813(HUDHandler.manaBar);
		RenderHelper.drawTexturedModalRect(ms, x, y, u, v, 22, 15);
		RenderSystem.color4f(1F, 1F, 1F, 1F);

		class_1799 tablet = new class_1799(ModItems.manaTablet);
		ItemManaTablet.setStackCreative(tablet);

		mc.method_1480().method_4023(tablet, x - 20, y);
		mc.method_1480().method_4023(pool, x + 26, y);

		RenderSystem.disableLighting();
		RenderSystem.disableBlend();
	}

	@Override
	public boolean canReceiveManaFromBursts() {
		return true;
	}

	@Override
	public boolean isOutputtingPower() {
		return outputting;
	}

	@Override
	public int getCurrentMana() {
		if (method_11010().method_26204() instanceof BlockPool) {
			return ((BlockPool) method_11010().method_26204()).variant == BlockPool.Variant.CREATIVE ? MAX_MANA : mana;
		}
		return 0;
	}

	@Override
	public String getInputKey() {
		return inputKey;
	}

	@Override
	public String getOutputKey() {
		return outputKey;
	}

	@Override
	public boolean canAttachSpark(class_1799 stack) {
		return true;
	}

	@Override
	public void attachSpark(ISparkEntity entity) {}

	@Override
	public ISparkEntity getAttachedSpark() {
		List<class_1297> sparks = field_11863.method_8390(class_1297.class, new class_238(field_11867.method_10084(), field_11867.method_10084().method_10069(1, 1, 1)), Predicates.instanceOf(ISparkEntity.class));
		if (sparks.size() == 1) {
			class_1297 e = sparks.get(0);
			return (ISparkEntity) e;
		}

		return null;
	}

	@Override
	public boolean areIncomingTranfersDone() {
		return false;
	}

	@Override
	public int getAvailableSpaceForMana() {
		int space = Math.max(0, manaCap - getCurrentMana());
		if (space > 0) {
			return space;
		} else if (field_11863.method_8320(field_11867.method_10074()).method_26204() == ModBlocks.manaVoid) {
			return manaCap;
		} else {
			return 0;
		}
	}

	@Override
	public class_1767 getColor() {
		return color;
	}

	@Override
	public void setColor(class_1767 color) {
		this.color = color;
		field_11863.method_8413(field_11867, method_11010(), method_11010(), 3);
	}

	@Override
	public void markDispatchable() {
		sendPacket = true;
	}
}
