/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.tile.corporea;

import vazkii.botania.api.corporea.*;
import vazkii.botania.common.advancements.CorporeaRequestTrigger;
import vazkii.botania.common.block.tile.ModTiles;
import vazkii.botania.common.core.ModStats;
import vazkii.botania.common.core.helper.MathHelper;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import net.minecraft.class_124;
import net.minecraft.class_1542;
import net.minecraft.class_156;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_238;
import net.minecraft.class_2588;
import net.minecraft.class_3000;
import net.minecraft.class_3222;

public class TileCorporeaIndex extends TileCorporeaBase implements ICorporeaRequestor, class_3000 {
	public static final double RADIUS = 2.5;

	private static InputHandler input;
	private static final Set<TileCorporeaIndex> serverIndexes = Collections.newSetFromMap(new WeakHashMap<>());
	private static final Set<TileCorporeaIndex> clientIndexes = Collections.newSetFromMap(new WeakHashMap<>());

	private static final Map<Pattern, IRegexStacker> patterns = new LinkedHashMap<>();

	/**
	 * (name) = Item name, or "this" for the name of the item in your hand
	 * (n), (n1), (n2), etc = Numbers
	 * [text] = Optional
	 * <a/b> = Either a or b
	 */
	static {
		// (name) = 1
		addPattern("(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 1;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a][n] (name) = 1
		addPattern("a??n?? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 1;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		//(n)[x][ of] (name) = n
		addPattern("(\\d+)x?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// [a ]stack[ of] (name) = 64
		addPattern("(?:a )?stack(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// (n)[x] stack[s][ of] (name) = n * 64
		addPattern("(\\d+)x?? stacks?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64 * i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// [a ]stack <and/+> (n)[x][ of] (name) = 64 + n
		addPattern("(?:a )?stack (?:(?:and)|(?:\\+)) (\\d+)(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64 + i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// (n1)[x] stack[s] <and/+> (n2)[x][ of] (name) = n1 * 64 + n2
		addPattern("(\\d+)x?? stacks? (?:(?:and)|(?:\\+)) (\\d+)x?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 64 * i(m, 1) + i(m, 2);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(3);
			}
		});

		// [a ]half [of ][a ]stack[ of] (name) = 32
		addPattern("(?:a )?half (?:of )?(?:a )?stack(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 32;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a ]quarter [of ][a ]stack[ of] (name) = 16
		addPattern("(?:a )?quarter (?:of )?(?:a )?stack(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 16;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a ]dozen[ of] (name) = 12
		addPattern("(?:a )?dozen(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 12;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// (n)[x] dozen[s][ of] (name) = n * 12
		addPattern("(\\d+)x?? dozens?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 12 * i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// <all/every> [<of/the> ](name) = 2147483647
		addPattern("(?:all|every) (?:(?:of|the) )?(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return Integer.MAX_VALUE;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [the ]answer to life[,] the universe and everything [of ](name) = 42
		addPattern("(?:the )?answer to life,? the universe and everything (?:of )?(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 42;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// [a ]nice [of ](name) = 69 
		addPattern("(?:a )?nice (?:of )?(.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 69;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});

		// (n)[x] nice[s][ of] (name) = n * 69
		addPattern("(\\d+)x?? nices?(?: of)? (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 69 * i(m, 1);
			}

			@Override
			public String getName(Matcher m) {
				return m.group(2);
			}
		});

		// <count/show/display/tell> (name) = 0 (display only)
		addPattern("(?:count|show|display|tell) (.+)", new IRegexStacker() {
			@Override
			public int getCount(Matcher m) {
				return 0;
			}

			@Override
			public String getName(Matcher m) {
				return m.group(1);
			}
		});
	}

	public int ticks = 0;
	public int ticksWithCloseby = 0;
	public float closeby = 0F;
	public boolean hasCloseby;

	public TileCorporeaIndex() {
		super(ModTiles.CORPOREA_INDEX);
	}

	@Override
	public void method_16896() {
		double x = field_11867.method_10263() + 0.5;
		double y = field_11867.method_10264() + 0.5;
		double z = field_11867.method_10260() + 0.5;

		List<class_1657> players = field_11863.method_18467(class_1657.class, new class_238(x - RADIUS, y - RADIUS, z - RADIUS, x + RADIUS, y + RADIUS, z + RADIUS));
		hasCloseby = false;
		for (class_1657 player : players) {
			if (isInRangeOfIndex(player, this)) {
				hasCloseby = true;
				break;
			}
		}

		float step = 0.2F;
		ticks++;
		if (hasCloseby) {
			ticksWithCloseby++;
			if (closeby < 1F) {
				closeby += step;
			}
		} else if (closeby > 0F) {
			closeby -= step;
		}

		if (!method_11015()) {
			addIndex(this);
		}
	}

	@Override
	public void method_11012() {
		super.method_11012();
		removeIndex(this);
	}

	@Override
	public void doCorporeaRequest(ICorporeaRequestMatcher request, int count, ICorporeaSpark spark) {
		doRequest(request, count, spark);
	}

	private ICorporeaResult doRequest(ICorporeaRequestMatcher matcher, int count, ICorporeaSpark spark) {
		ICorporeaResult result = CorporeaHelper.instance().requestItem(matcher, count, spark, true);
		List<class_1799> stacks = result.getStacks();
		spark.onItemsRequested(stacks);
		for (class_1799 stack : stacks) {
			if (!stack.method_7960()) {
				class_1542 item = new class_1542(field_11863, field_11867.method_10263() + 0.5, field_11867.method_10264() + 1.5, field_11867.method_10260() + 0.5, stack);
				field_11863.method_8649(item);
			}
		}
		return result;
	}

	public static boolean isInRangeOfIndex(class_1657 player, TileCorporeaIndex index) {
		return player.field_6002.method_27983() == index.field_11863.method_27983() && MathHelper.pointDistancePlane(index.method_11016().method_10263() + 0.5, index.method_11016().method_10260() + 0.5, player.method_23317(), player.method_23321()) < RADIUS && Math.abs(index.method_11016().method_10264() + 0.5 - player.method_23318() + (player.field_6002.field_9236 ? 0 : 1.6)) < 5;
	}

	public static void addPattern(String pattern, IRegexStacker stacker) {
		patterns.put(Pattern.compile(pattern), stacker);
	}

	public static int i(Matcher m, int g) {
		try {
			int i = Math.abs(Integer.parseInt(m.group(g)));
			return i;
		} catch (NumberFormatException e) {
			return 0;
		}
	}

	public static InputHandler getInputHandler() {
		if (input == null) {
			input = new InputHandler();
		}
		return input;
	}

	private static void addIndex(TileCorporeaIndex index) {
		Set<TileCorporeaIndex> set = index.field_11863.field_9236 ? clientIndexes : serverIndexes;
		set.add(index);
	}

	private static void removeIndex(TileCorporeaIndex index) {
		Set<TileCorporeaIndex> set = index.field_11863.field_9236 ? clientIndexes : serverIndexes;
		set.remove(index);
	}

	public static void clearCache() {
		clientIndexes.clear();
		serverIndexes.clear();
	}

	public void performPlayerRequest(class_3222 player, ICorporeaRequestMatcher request, int count) {
		if (!CorporeaIndexRequestCallback.EVENT.invoker().onIndexRequest(player, request, count, this.getSpark())) {
			ICorporeaResult res = this.doRequest(request, count, this.getSpark());

			player.method_9203(new class_2588("botaniamisc.requestMsg", count, request.getRequestName(), res.getMatchedCount(), res.getExtractedCount()).method_27692(class_124.field_1076), class_156.field_25140);
			player.method_7339(ModStats.CORPOREA_ITEMS_REQUESTED, res.getExtractedCount());
			CorporeaRequestTrigger.INSTANCE.trigger(player, player.method_14220(), this.method_11016(), res.getExtractedCount());
		}
	}

	public static final class InputHandler {
		public boolean onChatMessage(class_3222 player, String message) {
			if (player.method_7325()) {
				return false;
			}

			List<TileCorporeaIndex> nearbyIndexes = getNearbyIndexes(player);
			if (!nearbyIndexes.isEmpty()) {
				String msg = message.toLowerCase().trim();
				for (TileCorporeaIndex index : nearbyIndexes) {
					ICorporeaSpark spark = index.getSpark();
					if (spark != null) {
						String name = "";
						int count = 0;
						for (Pattern pattern : patterns.keySet()) {
							Matcher matcher = pattern.matcher(msg);
							if (matcher.matches()) {
								IRegexStacker stacker = patterns.get(pattern);
								count = stacker.getCount(matcher);
								name = stacker.getName(matcher).toLowerCase().trim();
							}
						}

						if (name.equals("this")) {
							class_1799 stack = player.method_6047();
							if (!stack.method_7960()) {
								name = stack.method_7964().getString().toLowerCase().trim();
							}
						}

						index.performPlayerRequest(player, CorporeaHelper.instance().createMatcher(name), count);
					}
				}

				return true;
			}

			return false;
		}

		public static List<TileCorporeaIndex> getNearbyIndexes(class_1657 player) {
			return (player.field_6002.field_9236 ? clientIndexes : serverIndexes)
					.stream().filter(i -> isInRangeOfIndex(player, i))
					.collect(Collectors.toList());
		}
	}

	public interface IRegexStacker {

		int getCount(Matcher m);

		String getName(Matcher m);

	}

}
