/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.client.core.helper;

import net.fabricmc.fabric.api.resource.IdentifiableResourceReloadListener;
import net.fabricmc.fabric.api.resource.ResourceManagerHelper;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_281;
import net.minecraft.class_285;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3302;
import net.minecraft.class_3679;
import net.minecraft.class_3695;
import net.minecraft.class_4013;
import net.minecraft.class_4493;
import net.minecraft.resource.*;
import org.lwjgl.system.MemoryUtil;

import vazkii.botania.client.core.handler.ClientTickHandler;
import vazkii.botania.client.lib.LibResources;
import vazkii.botania.common.Botania;
import vazkii.botania.common.core.handler.ConfigHandler;

import javax.annotation.Nullable;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.FloatBuffer;
import java.util.EnumMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import static vazkii.botania.common.lib.ResourceLocationHelper.prefix;

public final class ShaderHelper {
	public enum BotaniaShader {
		PYLON_GLOW(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_PYLON_GLOW_FRAG),
		ENCHANTER_RUNE(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_ENCHANTER_RUNE_FRAG),
		MANA_POOL(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_MANA_POOL_FRAG),
		DOPPLEGANGER(LibResources.SHADER_DOPLLEGANGER_VERT, LibResources.SHADER_DOPLLEGANGER_FRAG),
		HALO(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_HALO_FRAG),
		DOPPLEGANGER_BAR(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_DOPLLEGANGER_BAR_FRAG),
		TERRA_PLATE(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_TERRA_PLATE_RUNE_FRAG),
		FILM_GRAIN(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_FILM_GRAIN_FRAG),
		GOLD(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_GOLD_FRAG),
		ALPHA(LibResources.SHADER_PASSTHROUGH_VERT, LibResources.SHADER_ALPHA_FRAG);

		public final String vertexShaderPath;
		public final String fragmentShaderPath;

		BotaniaShader(String vertexShaderPath, String fragmentShaderPath) {
			this.vertexShaderPath = vertexShaderPath;
			this.fragmentShaderPath = fragmentShaderPath;
		}
	}

	// Scratch buffer to use for uniforms
	public static final FloatBuffer FLOAT_BUF = MemoryUtil.memAllocFloat(1);
	private static final Map<BotaniaShader, ShaderProgram> PROGRAMS = new EnumMap<>(BotaniaShader.class);

	private static boolean hasIncompatibleMods = false;
	private static boolean checkedIncompatibility = false;

	public static void initShaders() {
		ResourceManagerHelper.get(class_3264.field_14188).registerReloadListener(new IdentifiableResourceReloadListener() {
			private final class_3302 inner = (class_4013) manager -> {
				PROGRAMS.values().forEach(class_285::method_1304);
				PROGRAMS.clear();
				loadShaders(manager);
			};

			@Override
			public class_2960 getFabricId() {
				return prefix("shader_loader");
			}

			@Override
			public CompletableFuture<Void> method_25931(class_4045 synchronizer, class_3300 manager, class_3695 prepareProfiler, class_3695 applyProfiler, Executor prepareExecutor, Executor applyExecutor) {
				return inner.method_25931(synchronizer, manager, prepareProfiler, applyProfiler, prepareExecutor, applyExecutor);
			}
		});
	}

	private static void loadShaders(class_3300 manager) {
		if (!useShaders()) {
			return;
		}

		for (BotaniaShader shader : BotaniaShader.values()) {
			createProgram(manager, shader);
		}
	}

	public static void useShader(BotaniaShader shader, @Nullable ShaderCallback callback) {
		if (!useShaders()) {
			return;
		}

		ShaderProgram prog = PROGRAMS.get(shader);
		if (prog == null) {
			return;
		}

		int program = prog.method_1270();
		class_285.method_22094(program);

		int time = class_4493.method_21990(program, "time");
		class_4493.method_22030(time, ClientTickHandler.ticksInGame);

		if (callback != null) {
			callback.call(program);
		}
	}

	public static void useShader(BotaniaShader shader) {
		useShader(shader, null);
	}

	public static void releaseShader() {
		class_285.method_22094(0);
	}

	public static boolean useShaders() {
		return ConfigHandler.CLIENT.useShaders.getValue() && checkIncompatibleMods();
	}

	private static boolean checkIncompatibleMods() {
		if (!checkedIncompatibility) {
			hasIncompatibleMods = FabricLoader.getInstance().isModLoaded("optifabric");
			checkedIncompatibility = true;
		}

		return !hasIncompatibleMods;
	}

	private static void createProgram(class_3300 manager, BotaniaShader shader) {
		try {
			class_281 vert = createShader(manager, shader.vertexShaderPath, class_281.class_282.field_1530);
			class_281 frag = createShader(manager, shader.fragmentShaderPath, class_281.class_282.field_1531);
			int progId = class_285.method_1306();
			ShaderProgram prog = new ShaderProgram(progId, vert, frag);
			class_285.method_1307(prog);
			PROGRAMS.put(shader, prog);
		} catch (IOException ex) {
			Botania.LOGGER.error("Failed to load program {}", shader.name(), ex);
		}
	}

	private static class_281 createShader(class_3300 manager, String filename, class_281.class_282 shaderType) throws IOException {
		class_2960 loc = prefix(filename);
		try (InputStream is = new BufferedInputStream(manager.method_14486(loc).method_14482())) {
			return class_281.method_1283(shaderType, loc.toString(), is, shaderType.name().toLowerCase(Locale.ROOT));
		}
	}

	private static class ShaderProgram implements class_3679 {
		private final int program;
		private final class_281 vert;
		private final class_281 frag;

		private ShaderProgram(int program, class_281 vert, class_281 frag) {
			this.program = program;
			this.vert = vert;
			this.frag = frag;
		}

		@Override
		public int method_1270() {
			return program;
		}

		@Override
		public void method_1279() {

		}

		@Override
		public class_281 method_1274() {
			return vert;
		}

		@Override
		public class_281 method_1278() {
			return frag;
		}
	}

}
