package rearth.oritech.block.entity.interaction;

import dev.architectury.fluid.FluidStack;
import dev.architectury.hooks.fluid.FluidStackHooks;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.EnergyApi;
import rearth.oritech.api.energy.containers.SimpleEnergyStorage;
import rearth.oritech.api.fluid.FluidApi;
import rearth.oritech.api.fluid.containers.SimpleFluidStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.base.entity.MachineBlockEntity;
import rearth.oritech.init.BlockContent;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.util.ColorableMachine;
import software.bernie.geckolib.animatable.GeoBlockEntity;
import software.bernie.geckolib.animatable.instance.AnimatableInstanceCache;
import software.bernie.geckolib.animation.AnimatableManager;
import software.bernie.geckolib.animation.AnimationController;
import software.bernie.geckolib.util.GeckoLibUtil;

import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3610;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_7225;

public class PumpBlockEntity extends NetworkedBlockEntity implements FluidApi.BlockProvider, EnergyApi.BlockProvider, GeoBlockEntity, ColorableMachine {
    
    private static final int MAX_SEARCH_COUNT = 100_000;
    private static final int ENERGY_USAGE = 512;   // per block pumped
    private static final int PUMP_RATE = 5; // pump every n ticks
    
    protected final AnimatableInstanceCache animatableInstanceCache = GeckoLibUtil.createInstanceCache(this);
    private final AnimationController<PumpBlockEntity> animationController = getAnimationController();
    
    private final SimpleFluidStorage fluidStorage = new SimpleFluidStorage(16 * FluidStackHooks.bucketAmount(), this::method_5431);
    
    private final SimpleEnergyStorage energyStorage = new SimpleEnergyStorage(1000, 0, 20_000);
    private boolean initialized = false;
    private boolean toolheadLowered = false;
    private boolean searchActive = false;
    private class_2338 toolheadPosition;
    private FloodFillSearch searchInstance;
    private Deque<class_2338> pendingLiquidPositions;
    
    @SyncField(SyncType.TICK)
    private long lastWorkTime;
    
    @SyncField({SyncType.SPARSE_TICK, SyncType.INITIAL})
    public ColorableMachine.ColorVariant currentColor = getDefaultColor();
    
    public PumpBlockEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.PUMP_BLOCK, pos, state);
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        fluidStorage.writeNbt(nbt, "");
        nbt.method_10556("initialized", initialized);
        nbt.method_10544("energy", energyStorage.getAmount());
        addColorToNbt(nbt);
        
        if (pendingLiquidPositions != null)
            nbt.method_10564("pendingTargets", pendingLiquidPositions.stream().mapToLong(class_2338::method_10063).toArray());
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        loadColorFromNbt(nbt);
        initialized = nbt.method_10577("initialized");
        fluidStorage.readNbt(nbt, "");
        energyStorage.setAmount(nbt.method_10537("energy"));
        pendingLiquidPositions = Arrays.stream(nbt.method_10565("pendingTargets")).mapToObj(class_2338::method_10092).collect(Collectors.toCollection(ArrayDeque::new));
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        
        if ((initialized && pendingLiquidPositions.isEmpty() && world.method_8510() % 62 == 0) || (!initialized && toolheadLowered && !searchActive && world.method_8510() % 62 == 0)) {
            // reset
            initialized = false;
            toolheadLowered = false;
            searchActive = false;
            toolheadPosition = pos;
        }
        
        if (!initialized) {
            progressStartup();
            return;
        }
        
        if (world.method_8510() % PUMP_RATE == 0 && hasEnoughEnergy() && world.method_49804(pos) <= 0) {
            
            if (pendingLiquidPositions.isEmpty() || tankIsFull()) return;
            
            var targetBlock = pendingLiquidPositions.peekLast();
            
            // Only drain the source (still) fluid, so it doesn't keep pumping infinitely
            if (!world.method_8320(targetBlock).method_26227().method_15771()) {
                pendingLiquidPositions.pollLast();
                return;
            }
            
            var targetState = world.method_8316(targetBlock);
            if (!targetState.method_15772().method_15780(class_3612.field_15910)) {
                drainSourceBlock(targetBlock);
                pendingLiquidPositions.pollLast();
            }
            
            addLiquidToTank(targetState);
            useEnergy();
            this.method_5431();
            lastWorkTime = world.method_8510();
            
            
            var targetPos = pos.method_46558().method_49272(world.field_9229, 0.5f);
            var targetType = targetState.method_15766();
            
            if (targetType != null && world instanceof class_3218 serverWorld)
                serverWorld.method_14199(targetType, targetPos.method_10216(), targetPos.method_10214(), targetPos.method_10215(), 1, 0, 0, 0, 1);
        }
        
    }
    
    private boolean isBusy() {
        return field_11863.method_8510() - lastWorkTime < 40;
    }
    
    public void onUsed(class_1657 player) {
        
        var message = class_2561.method_43471("message.oritech.pump.starting");
        if (!initialized) {
            if (!toolheadLowered) {
                message = class_2561.method_43471("message.oritech.pump.extending");
            } else if (searchActive) {
                message = class_2561.method_43471("message.oritech.pump.initializing");
            } else {
                message = class_2561.method_43471("message.oritech.pump.no_fluids");
            }
        } else if (isBusy()) {
            message = class_2561.method_43471("message.oritech.pump.busy");
        } else if (!hasEnoughEnergy()) {
            message = class_2561.method_43471("message.oritech.pump.low_energy");
        } else if (pendingLiquidPositions.isEmpty()) {
            message = class_2561.method_43471("message.oritech.pump.pump_finished");
        } else if (tankIsFull()) {
            message = class_2561.method_43471("message.oritech.pump.full");
        }
        
        player.method_7353(message, true);
    }
    
    private boolean hasEnoughEnergy() {
        return energyStorage.getAmount() >= ENERGY_USAGE;
    }
    
    private void useEnergy() {
        energyStorage.extractIgnoringLimit(ENERGY_USAGE, false);
    }
    
    private boolean tankIsFull() {
        return fluidStorage.getAmount() > fluidStorage.getCapacity() - FluidStackHooks.bucketAmount();
    }
    
    private void addLiquidToTank(class_3610 targetState) {
        fluidStorage.insert(FluidStack.create(targetState.method_15772(), FluidStackHooks.bucketAmount()), false);
    }
    
    private void drainSourceBlock(class_2338 targetBlock) {
        field_11863.method_8501(targetBlock, class_2246.field_10124.method_9564());
    }
    
    private void progressStartup() {
        
        // startup sequence is:
        // move down until no longer in air
        // check if target is liquid
        // if liquid is water, consider as infinite
        // if liquid, start flood fill to find all liquid blocks. Add all found blocks to queue so that it can be soaked up in reverse
        // search all neighbors per tick
        // if more than N blocks are found, consider the search finished
        // mark startup as completed
        
        if (toolheadPosition == null) {
            toolheadPosition = field_11867;
        }
        
        if (!toolheadLowered) {
            
            if (field_11863.method_8510() % 10 != 0)
                moveToolheadDown();
            
            return;
        }
        
        if (searchActive) {
            if (searchInstance.nextGeneration()) {
                finishSearch();
                searchActive = false;
            }
        }
    }
    
    private void moveToolheadDown() {
        toolheadLowered = checkToolheadEnd(toolheadPosition);
        if (toolheadLowered) {
            startLiquidSearch(toolheadPosition.method_10074());
            return;
        }
        
        toolheadPosition = toolheadPosition.method_10074();
        field_11863.method_8501(toolheadPosition, BlockContent.PUMP_TRUNK_BLOCK.method_9564());
    }
    
    private boolean checkToolheadEnd(class_2338 newPosition) {
        
        var posBelow = newPosition.method_10074();
        var stateBelow = field_11863.method_8320(posBelow);
        var blockBelow = stateBelow.method_26204();
        
        var isAirOrTrunk = stateBelow.method_45474() || blockBelow.equals(BlockContent.PUMP_TRUNK_BLOCK);
        var isFluid = !stateBelow.method_26227().method_15769();
        
        return isFluid || !isAirOrTrunk;
    }
    
    private void startLiquidSearch(class_2338 start) {
        
        var state = field_11863.method_8316(start);
        if (!state.method_15771()) return;
        
        searchInstance = new FloodFillSearch(start, field_11863, state.method_15772());
        searchActive = true;
        
        Oritech.LOGGER.debug("starting search at: " + start + " " + state.method_15772() + " " + state.method_15771());
    }
    
    private void finishSearch() {
        Oritech.LOGGER.debug("search finished, found: " + searchInstance.foundTargets.size());
        pendingLiquidPositions = searchInstance.foundTargets;
        initialized = true;
        searchInstance = null;
    }
    
    @Override
    public ColorVariant getCurrentColor() {
        return currentColor;
    }
    
    @Override
    public void assignColor(ColorVariant color) {
        this.currentColor = color;
        
        if (this.field_11863 != null && !this.field_11863.method_8608()) {
            this.setChanged(false);
            this.sendUpdate(SyncType.SPARSE_TICK);
        }
    }
    
    
    @Override
    public FluidApi.FluidStorage getFluidStorage(class_2350 direction) {
        return fluidStorage;
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorage(class_2350 direction) {
        return energyStorage;
    }
    
    @Override
    public void registerControllers(AnimatableManager.ControllerRegistrar controllers) {
        controllers.add(animationController);
    }
    
    @Override
    public AnimatableInstanceCache getAnimatableInstanceCache() {
        return animatableInstanceCache;
    }
    
    private AnimationController<PumpBlockEntity> getAnimationController() {
        return new AnimationController<>(this, state -> {
            if (isBusy()) {
                return state.setAndContinue(MachineBlockEntity.WORKING);
            } else {
                return state.setAndContinue(MachineBlockEntity.IDLE);
            }
        });
    }
    
    private static class FloodFillSearch {
        
        final HashSet<class_2338> checkedPositions = new HashSet<>();
        final HashSet<class_2338> nextTargets = new HashSet<>();
        final Deque<class_2338> foundTargets = new ArrayDeque<>();
        final class_1937 world;
        final class_3611 fluidType;
        
        public FloodFillSearch(class_2338 startPosition, class_1937 world, class_3611 fluidType) {
            this.world = world;
            this.fluidType = fluidType;
            nextTargets.add(startPosition);
        }
        
        // returns true when done
        @SuppressWarnings("unchecked")
        public boolean nextGeneration() {
            
            var currentGeneration = (HashSet<class_2338>) nextTargets.clone();
            
            var earlyStop = false;
            
            for (var target : currentGeneration) {
                if (isValidTarget(target)) {
                    foundTargets.addLast(target);
                    addNeighborsToQueue(target);
                    if (checkForEarlyStop(target)) earlyStop = true;
                }
                
                checkedPositions.add(target);
                nextTargets.remove(target);
            }
            
            if (cutoffSearch() || earlyStop) nextTargets.clear();
            
            return nextTargets.isEmpty();
        }
        
        private boolean checkForEarlyStop(class_2338 target) {
            return world.method_8316(target).method_15772().method_15780(class_3612.field_15910);
        }
        
        private boolean cutoffSearch() {
            return foundTargets.size() >= MAX_SEARCH_COUNT;
        }
        
        private boolean isValidTarget(class_2338 target) {
            var state = world.method_8316(target);
            return !state.method_15769() && state.method_39360(fluidType);
        }
        
        private void addNeighborsToQueue(class_2338 self) {
            
            for (var neighbor : getNeighbors(self)) {
                if (checkedPositions.contains(neighbor)) continue;
                nextTargets.add(neighbor);
            }
            
        }
        
        // returns all neighboring positions except up
        private List<class_2338> getNeighbors(class_2338 pos) {
            return List.of(pos.method_10074(), pos.method_10095(), pos.method_10078(), pos.method_10072(), pos.method_10067());
        }
        
    }
}
