package rearth.oritech.block.entity.interaction;

import com.mojang.authlib.GameProfile;
import dev.architectury.registry.menu.ExtendedMenuProvider;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.EnergyApi;
import rearth.oritech.api.energy.containers.DynamicEnergyStorage;
import rearth.oritech.api.item.ItemApi;
import rearth.oritech.api.item.containers.SimpleInventoryStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.api.networking.WorldPacketCodec;
import rearth.oritech.block.base.entity.MachineBlockEntity;
import rearth.oritech.block.behavior.LaserArmBlockBehavior;
import rearth.oritech.block.blocks.interaction.LaserArmBlock;
import rearth.oritech.block.blocks.processing.MachineCoreBlock;
import rearth.oritech.block.entity.MachineCoreEntity;
import rearth.oritech.block.entity.addons.CombiAddonEntity;
import rearth.oritech.block.entity.addons.RedstoneAddonBlockEntity;
import rearth.oritech.client.init.ModScreens;
import rearth.oritech.client.init.ParticleContent;
import rearth.oritech.client.ui.UpgradableMachineScreenHandler;
import rearth.oritech.init.BlockContent;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.init.TagContent;
import rearth.oritech.init.recipes.OritechRecipe;
import rearth.oritech.init.recipes.RecipeContent;
import rearth.oritech.util.*;
import software.bernie.geckolib.animatable.GeoBlockEntity;
import software.bernie.geckolib.animatable.instance.AnimatableInstanceCache;
import software.bernie.geckolib.animation.AnimatableManager;
import software.bernie.geckolib.animation.AnimationController;
import software.bernie.geckolib.util.GeckoLibUtil;

import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_1262;
import net.minecraft.class_1263;
import net.minecraft.class_1301;
import net.minecraft.class_1309;
import net.minecraft.class_1429;
import net.minecraft.class_1480;
import net.minecraft.class_1569;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1893;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_2382;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_2769;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3419;
import net.minecraft.class_3545;
import net.minecraft.class_3917;
import net.minecraft.class_7225;
import net.minecraft.class_7924;
import net.minecraft.class_8786;
import net.minecraft.class_9129;
import net.minecraft.class_9300;
import net.minecraft.class_9334;

import static rearth.oritech.block.base.block.MultiblockMachine.ASSEMBLED;


public class LaserArmBlockEntity extends NetworkedBlockEntity implements
  GeoBlockEntity, EnergyApi.BlockProvider, ScreenProvider, ExtendedMenuProvider, MultiblockMachineController, MachineAddonController,
    ItemApi.BlockProvider, RedstoneAddonBlockEntity.RedstoneControllable, ColorableMachine {
    
    private static final String LASER_PLAYER_NAME = "oritech_laser";
    public static final int BLOCK_BREAK_ENERGY = Oritech.CONFIG.laserArmConfig.blockBreakEnergyBase();
    
    // storage
    @SyncField({SyncType.GUI_OPEN, SyncType.GUI_TICK})
    protected final DynamicEnergyStorage energyStorage = new DynamicEnergyStorage(getDefaultCapacity(), getDefaultInsertRate(), 0, this::method_5431);
    
    public final SimpleInventoryStorage inventory = new SimpleInventoryStorage(3, this::method_5431);
    
    // animation
    protected final AnimatableInstanceCache animatableInstanceCache = GeckoLibUtil.createInstanceCache(this);
    private final AnimationController<LaserArmBlockEntity> animationController = getAnimationController();
    
    // multiblock
    private final ArrayList<class_2338> coreBlocksConnected = new ArrayList<>();
    
    // addons
    @SyncField(SyncType.GUI_OPEN)
    private final List<class_2338> connectedAddons = new ArrayList<>();
    @SyncField(SyncType.GUI_OPEN)
    private final List<class_2338> openSlots = new ArrayList<>();
    @SyncField(SyncType.GUI_OPEN)
    private float coreQuality = 1f;
    @SyncField(SyncType.GUI_OPEN)
    private BaseAddonData addonData = BaseAddonData.DEFAULT_ADDON_DATA;
    @SyncField(SyncType.GUI_OPEN)
    public int areaSize = 1;
    @SyncField(SyncType.GUI_OPEN)
    public int yieldAddons = 0;
    @SyncField(SyncType.GUI_OPEN)
    public int hunterAddons = 0;
    @SyncField(SyncType.GUI_OPEN)
    public boolean hasCropFilterAddon = false;
    @SyncField(SyncType.GUI_OPEN)
    public boolean hasSilkTouchAddon = false;
    
    @SyncField({SyncType.SPARSE_TICK, SyncType.INITIAL})
    public ColorableMachine.ColorVariant currentColor = ColorVariant.ORANGE;
    
    // config
    private final int range = Oritech.CONFIG.laserArmConfig.range();
    
    public class_243 laserHead;
    
    // working data
    private class_2338 targetDirection;
    
    @SyncField
    private class_2338 currentTarget = class_2338.field_10980;
    @SyncField
    public HunterTargetMode hunterTargetMode = HunterTargetMode.HOSTILE_ONLY;
    @SyncField
    private class_1309 currentLivingTarget;
    @SyncField
    private long lastFiredAt;
    @SyncField({SyncType.GUI_OPEN, SyncType.GUI_TICK})
    private boolean redstonePowered;
    private int progress;
    private ArrayDeque<class_2338> pendingArea;
    private final ArrayDeque<class_1309> pendingLivingTargets = new ArrayDeque<>();
    private int targetBlockEnergyNeeded = BLOCK_BREAK_ENERGY;
    
    // needed only on client
    public class_243 lastRenderPosition;
    private class_1657 laserPlayerEntity = null;
    
    public LaserArmBlockEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.LASER_ARM_ENTITY, pos, state);
        laserHead = getLaserHeadPosition().method_46558();
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        if (!isActive(state))
            return;
        
        if (!redstonePowered && energyStorage.getAmount() >= energyRequiredToFire()) {
            if (hunterAddons > 0) {
                fireAtLivingEntities(world, pos, state, this);
            } else if (currentTarget != null && !currentTarget.equals(class_2338.field_10980)) {
                fireAtBlocks(world, pos, state, this);
            } else if (targetDirection != null && !targetDirection.equals(class_2338.field_10980) && (world.method_8510() + pos.method_10260()) % 40 == 0) {
                // target pos is set, but no target is found (e.g. all blocks already mined). Periodically scan again for new blocks.
                findNextBlockBreakTarget();
            }
        }
    }
    
    private void fireAtBlocks(class_1937 world, class_2338 pos, class_2680 state, LaserArmBlockEntity blockEntity) {
        var targetBlockPos = currentTarget;
        var targetBlockState = world.method_8320(targetBlockPos);
        var targetBlock = targetBlockState.method_26204();
        var targetBlockEntity = world.method_8321(targetBlockPos);
        
        LaserArmBlockBehavior behavior = LaserArmBlock.getBehaviorForBlock(targetBlock);
        boolean fired = false;
        if (behavior.fireAtBlock(world, this, targetBlock, targetBlockPos, targetBlockState, targetBlockEntity)) {
            energyStorage.amount -= energyRequiredToFire();
            lastFiredAt = world.method_8510();
        } else {
            findNextBlockBreakTarget();
        }
    }
    
    private void fireAtLivingEntities(class_1937 world, class_2338 pos, class_2680 state, LaserArmBlockEntity blockEntity) {
        // check that there is a target, that is still alive and still in range
        if (currentLivingTarget != null && validTarget(currentLivingTarget)) {
            
            var behavior = LaserArmBlock.getBehaviorForEntity(currentLivingTarget.method_5864());
            if (behavior.fireAtEntity(world, this, currentLivingTarget)) {
                energyStorage.amount -= energyRequiredToFire();
                this.targetDirection = currentLivingTarget.method_24515();
                lastFiredAt = world.method_8510();
            } else {
                pendingLivingTargets.remove(currentLivingTarget);
                currentLivingTarget = null;
                currentTarget = class_2338.field_10980;
            }
        } else {
            loadNextLivingTarget();
        }
        
    }
    
    public void setRedstonePowered(boolean redstonePowered) {
        this.redstonePowered = redstonePowered;
    }
    
    public void addBlockBreakProgress(int progress) {
        this.progress += progress;
    }
    
    public int getBlockBreakProgress() {
        return this.progress;
    }
    
    public int getTargetBlockEnergyNeeded() {
        return targetBlockEnergyNeeded;
    }
    
    public void finishBlockBreaking(class_2338 targetPos, class_2680 targetBlockState) {
        progress -= targetBlockEnergyNeeded;
        
        var targetEntity = field_11863.method_8321(targetPos);
        List<class_1799> dropped;
        // added getLaserPlayerEntity() to make ae2 certus quartz drop from certus
        // quartz clusters because it's expecting an entity in
        // LootContextParameters.THIS_ENTITY
        if (hasSilkTouchAddon) {
            dropped = DestroyerBlockEntity.getSilkTouchDrops(targetBlockState, (class_3218) field_11863, targetPos, targetEntity, getLaserPlayerEntity());
        } else if (yieldAddons > 0) {
            dropped = DestroyerBlockEntity.getLootDrops(targetBlockState, (class_3218) field_11863, targetPos, targetEntity, yieldAddons, getLaserPlayerEntity());
        } else {
            dropped = class_2248.method_9609(targetBlockState, (class_3218) field_11863, targetPos, targetEntity, getLaserPlayerEntity(), class_1799.field_8037);
        }
        
        var blockRecipe = tryGetRecipeOfBlock(targetBlockState, field_11863);
        if (blockRecipe != null) {
            var recipe = blockRecipe.comp_1933();
            var farmedCount = 1 + yieldAddons;
            dropped = List.of(new class_1799(recipe.getResults().get(0).method_7909(), farmedCount));
            ParticleContent.CHARGING.spawn(field_11863, class_243.method_24954(targetPos), 1);
        }
        
        // yes, this will discard items that wont fit anymore
        for (var stack : dropped) {
            this.inventory.insert(stack, false);
        }
        
        try {
            targetBlockState.method_26204().method_9576(field_11863, targetPos, targetBlockState, getLaserPlayerEntity());
        } catch (Exception exception) {
            Oritech.LOGGER.warn("Laser arm block break event failure when breaking " + targetBlockState + " at " + targetPos + ": " + exception.getLocalizedMessage());
        }
        field_11863.method_31595(targetPos, field_11863.method_8320(targetPos));
        field_11863.method_8396(null, targetPos, targetBlockState.method_26231().method_10595(), class_3419.field_15245, 1f, 1f);
        field_11863.method_22352(targetPos, false);
        
        findNextBlockBreakTarget();
    }
    
    public static class_8786<OritechRecipe> tryGetRecipeOfBlock(class_2680 destroyed, class_1937 world) {
        var inputItem = destroyed.method_26204().method_8389();
        var inputInv = new SimpleCraftingInventory(new class_1799(inputItem));
        var candidate = world.method_8433().method_8132(RecipeContent.LASER, inputInv, world);
        return candidate.orElse(null);
    }
    
    public class_1657 getLaserPlayerEntity() {
        if (!(field_11863 instanceof class_3218))
            return null;
        
        if (laserPlayerEntity == null) {
            laserPlayerEntity = FakeMachinePlayer.create((class_3218) field_11863, new GameProfile(UUID.randomUUID(), LASER_PLAYER_NAME), inventory);
            laserPlayerEntity.method_33574(class_243.method_24954(method_11016()));
        }
        
        if (hunterAddons > 0 && yieldAddons > 0) {
            var lootingSword = new class_1799(class_1802.field_22022);
            lootingSword.method_57379(class_9334.field_49630, new class_9300(false));
            var lootingEntry = field_11863.method_30349().method_30530(class_7924.field_41265).method_40264(class_1893.field_9110).get();
            lootingSword.method_7978(lootingEntry, Math.min(yieldAddons, 3));
            laserPlayerEntity.method_31548().field_7547.set(laserPlayerEntity.method_31548().field_7545, lootingSword);
        }
        
        return laserPlayerEntity;
    }
    
    private void findNextBlockBreakTarget() {
        
        while (pendingArea != null && !pendingArea.isEmpty()) {
            if (trySetNewTarget(pendingArea.pop(), false)) {
                if (pendingArea.isEmpty()) pendingArea = null;
                return;
            }
        }
        
        var direction = class_243.method_24954(targetDirection.method_10059(getLaserHeadPosition())).method_1029();
        var from = laserHead.method_1019(direction.method_1021(1.5));
        
        var nextBlock = basicRaycast(from, direction, range, 0.45F);
        if (nextBlock == null) {
            currentTarget = class_2338.field_10980;
            return;
        }
        
        var maxSize = (int) from.method_1022(nextBlock.method_46558()) - 1;
        var scanDist = Math.min(areaSize, maxSize);
        if (scanDist > 1)
            pendingArea = findNextAreaBlockTarget(nextBlock, scanDist);
        
        
        if (!trySetNewTarget(nextBlock, false)) {
            currentTarget = class_2338.field_10980;   // out of range or invalid for another reason
        }
        
    }
    
    private double hunterRange() {
        // hunter range is 2^hunterAddons, with max 3 hunterAddons
        // range should be calculated near the center of the laser head's cube, so add 0.5 to start counting range from side of cube
        return Math.pow(4, Math.min(hunterAddons, 3)) + 0.5;
    }
    
    private boolean canSee(class_1309 entity) {
        if (entity.method_37908() != this.method_10997() || entity.method_5767()) {
            return false;
        } else {
            var target = entity.method_33571();
            var direction = target.method_1020(laserHead).method_1029();
            if (laserHead.method_1022(target) > 128.0) {
                return false;
            } else {
                // can see if basicRaycast() doesn't find anything it can't pass through between laser and target
                return basicRaycast(laserHead.method_1019(direction.method_1021(1.5)), direction, (int) (laserHead.method_1022(target) - 1), 0.2f) == null;
            }
        }
    }
    
    private boolean validTarget(class_1309 entity) {
        return entity.method_5805() && canSee(entity) && huntedTarget(entity) && entity.method_19538().method_24802(getLaserHeadPosition().method_46558(), hunterRange());
    }
    
    private boolean huntedTarget(class_1309 entity) {
        // Regardless of mode, laser will always target player to charge energy storing chestplate
        if (entity instanceof class_1657) return true;
        // Not including Allay, Villagers, Trader, Iron Golem, Snow Golem
        // Also not including pets
        return switch (hunterTargetMode) {
            case HunterTargetMode.HOSTILE_ONLY -> entity instanceof class_1569;
            case HunterTargetMode.HOSTILE_NEUTRAL -> {
                if ((entity instanceof class_1429 animal && animal.method_6478() == null) || entity instanceof class_1480)
                    yield true;
                yield entity instanceof class_1569;
            }
            case HunterTargetMode.ALL -> true;
        };
    }
    
    // this only gets called if we don't have a target (e.g. null or not valid)
    private void loadNextLivingTarget() {
        
        // load targets if we don't have any (only every 10 ticks to save performance
        if (pendingLivingTargets.isEmpty() && (field_11863.method_8510() + field_11867.method_10063()) % 10 == 0) {
            updateEntityTargets();
        }
        
        // assign first target from cached, distance sorted target list
        while (!pendingLivingTargets.isEmpty()) {
            var candidate = pendingLivingTargets.pop();
            if (validTarget(candidate)) {
                currentLivingTarget = candidate;
                currentTarget = candidate.method_24515();
                return;
            }
        }
    }
    
    private void updateEntityTargets() {
        var entityRange = hunterRange();
        // Only sort the list when getting a new list of entities in range.
        // The entities can move around so the sort order isn't guaranteed to be correct, but it should be good enough.
        // There's no need to spend the time re-sorting the list every time the laser needs to pick a new target from the cached list.
        var targets = field_11863.method_8390(class_1309.class, new class_238(laserHead.field_1352 - entityRange, laserHead.field_1351 - entityRange, laserHead.field_1350 - entityRange, laserHead.field_1352 + entityRange, laserHead.field_1351 + entityRange, laserHead.field_1350 + entityRange), class_1301.field_6157.and(class_1301.field_6156));
        targets.sort(Comparator.comparingDouble((entity) -> entity.method_5707(laserHead)));
        pendingLivingTargets.addAll(targets);
    }
    
    // returns the first block in an X*X*X cube, from the outside in
    private ArrayDeque<class_2338> findNextAreaBlockTarget(class_2338 center, int scanDist) {
        
        var targets = new ArrayList<class_2338>();
        
        for (int x = -scanDist; x < scanDist; x++) {
            for (int y = -scanDist; y < scanDist; y++) {
                for (int z = -scanDist; z < scanDist; z++) {
                    var pos = center.method_10069(x, y, z);
                    if (!canPassThrough(field_11863.method_8320(pos), pos) && !center.equals(pos))
                        targets.add(pos);
                }
            }
        }
        
        targets.sort(Comparator.comparingInt(field_11867::method_19455));
        return new ArrayDeque<>(targets);
    }
    
    private class_2338 basicRaycast(class_243 from, class_243 direction, int range, float searchOffset) {
        
        for (float i = 0; i < range; i += 0.3f) {
            var to = from.method_1019(direction.method_1021(i));
            var targetBlockPos = class_2338.method_49638(to.method_1031(0, searchOffset, 0));
            var targetState = field_11863.method_8320(targetBlockPos);
            if (isSearchTerminatorBlock(targetState)) return null;
            if (!canPassThrough(targetState, targetBlockPos)) return targetBlockPos;
            
            
            if (searchOffset == 0.0F)
                return null;
            
            var offsetTop = to.method_1031(0, -searchOffset, 0);
            targetBlockPos = class_2338.method_49638(offsetTop);
            targetState = field_11863.method_8320(targetBlockPos);
            if (isSearchTerminatorBlock(targetState)) return null;
            if (!canPassThrough(targetState, targetBlockPos)) return targetBlockPos;
            
            var offsetLeft = to.method_1031(-searchOffset, 0, 0);
            targetBlockPos = class_2338.method_49638(offsetLeft);
            targetState = field_11863.method_8320(targetBlockPos);
            if (isSearchTerminatorBlock(targetState)) return null;
            if (!canPassThrough(targetState, targetBlockPos)) return targetBlockPos;
            
            var offsetRight = to.method_1031(searchOffset, 0, 0);
            targetBlockPos = class_2338.method_49638(offsetRight);
            targetState = field_11863.method_8320(targetBlockPos);
            if (isSearchTerminatorBlock(targetState)) return null;
            if (!canPassThrough(targetState, targetBlockPos)) return targetBlockPos;
            
            var offsetFront = to.method_1031(0, 0, searchOffset);
            targetBlockPos = class_2338.method_49638(offsetFront);
            targetState = field_11863.method_8320(targetBlockPos);
            if (isSearchTerminatorBlock(targetState)) return null;
            if (!canPassThrough(targetState, targetBlockPos)) return targetBlockPos;
            
            var offsetBack = to.method_1031(0, 0, -searchOffset);
            targetBlockPos = class_2338.method_49638(offsetBack);
            targetState = field_11863.method_8320(targetBlockPos);
            if (isSearchTerminatorBlock(targetState)) return null;
            if (!canPassThrough(targetState, targetBlockPos)) return targetBlockPos;
        }
        
        return null;
    }
    
    private boolean isSearchTerminatorBlock(class_2680 state) {
        return state.method_26204().equals(class_2246.field_22422);
    }
    
    public boolean canPassThrough(class_2680 state, class_2338 blockPos) {
        // When targetting entities, don't let grass, vines, small mushrooms, pressure plates, etc. get in the way of the laser
        return state.method_26215() || !state.method_26227().method_15769() || state.method_26164(TagContent.LASER_PASSTHROUGH) || (hunterAddons > 0 && !state.method_26212(field_11863, blockPos));
    }
    
    @Override
    public void gatherAddonStats(List<AddonBlock> addons) {
        
        areaSize = 1;
        yieldAddons = 0;
        hunterAddons = 0;
        hasCropFilterAddon = false;
        hasSilkTouchAddon = false;
        
        MachineAddonController.super.gatherAddonStats(addons);
        
        yieldAddons = Math.min(yieldAddons, 3);
    }
    
    @Override
    public void getAdditionalStatFromAddon(AddonBlock addonBlock) {
        MachineAddonController.super.getAdditionalStatFromAddon(addonBlock);
        
        if (addonBlock.state().method_26204().equals(BlockContent.QUARRY_ADDON))
            areaSize++;
        if (addonBlock.state().method_26204().equals(BlockContent.MACHINE_HUNTER_ADDON))
            hunterAddons++;
        if (addonBlock.state().method_26204().equals(BlockContent.MACHINE_YIELD_ADDON))
            yieldAddons++;
        if (addonBlock.state().method_26204().equals(BlockContent.CROP_FILTER_ADDON))
            hasCropFilterAddon = true;
        if (addonBlock.state().method_26204().equals(BlockContent.MACHINE_SILK_TOUCH_ADDON))
            hasSilkTouchAddon = true;
        
        if (addonBlock.addonEntity() instanceof CombiAddonEntity combi) {
            areaSize = combi.getQuarryCount();
            yieldAddons =  combi.getYieldCount();
            hasCropFilterAddon = combi.hasCropFilter();
            hasSilkTouchAddon = combi.hasSilk();
        }
        
    }
    
    public int energyRequiredToFire() {
        return (int) (Oritech.CONFIG.laserArmConfig.energyPerTick() * (1 / addonData.speed()));
    }
    
    public float getDamageTick() {
        return (Oritech.CONFIG.laserArmConfig.damageTickBase() * (1 / addonData.speed()));
    }
    
    public boolean setTargetFromDesignator(class_2338 targetPos) {
        var success = trySetNewTarget(targetPos, true);
        findNextBlockBreakTarget();
        
        return success;
    }
    
    public void cycleHunterTargetMode() {
        hunterTargetMode = hunterTargetMode.next();
    }
    
    private boolean trySetNewTarget(class_2338 targetPos, boolean alsoSetDirection) {
        
        // if target is coreblock, adjust it to point to controller if connected
        var targetState = Objects.requireNonNull(field_11863).method_8320(targetPos);
        if (targetState.method_26204() instanceof MachineCoreBlock && targetState.method_11654(MachineCoreBlock.USED)) {
            var coreEntity = (MachineCoreEntity) field_11863.method_8321(targetPos);
            var controllerPos = Objects.requireNonNull(coreEntity).getControllerPos();
            if (controllerPos != null) targetPos = controllerPos;
        }
        
        var distance = targetPos.method_19455(field_11867);
        var blockHardness = targetState.method_26204().method_36555();
        if (distance > range || blockHardness < 0.0 || targetState.method_26204().equals(class_2246.field_10124)) {
            return false;
        }
        
        this.targetBlockEnergyNeeded = (int) (BLOCK_BREAK_ENERGY * Math.pow(blockHardness, Oritech.CONFIG.blockBreakHardnessExponentialFactor()) * addonData.efficiency());
        
        if (targetState.method_26164(TagContent.LASER_FAST_BREAKING))
            targetBlockEnergyNeeded /= 8;
        
        this.currentTarget = targetPos;
        
        if (alsoSetDirection) {
            this.targetDirection = targetPos;
            pendingArea = null;
            method_5431();
        }
        this.method_5431();
        
        return true;
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        class_1262.method_5427(nbt, inventory.heldStacks, false, registryLookup);
        addMultiblockToNbt(nbt);
        writeAddonToNbt(nbt);
        addColorToNbt(nbt);
        nbt.method_10544("energy_stored", energyStorage.amount);
        nbt.method_10556("redstone", redstonePowered);
        nbt.method_10569("areaSize", areaSize);
        nbt.method_10569("yieldAddons", yieldAddons);
        nbt.method_10569("hunterAddons", hunterAddons);
        nbt.method_10556("cropAddon", hasCropFilterAddon);
        nbt.method_10556("silkAddon", hasSilkTouchAddon);
        nbt.method_10569("hunterTargetMode", hunterTargetMode.value);
        
        if (targetDirection != null && currentTarget != null) {
            nbt.method_10544("target_position", currentTarget.method_10063());
            nbt.method_10544("target_direction", targetDirection.method_10063());
        }
        
        if (pendingArea != null && !pendingArea.isEmpty()) {
            var positions = pendingArea.stream().mapToLong(class_2338::method_10063).toArray();
            nbt.method_10564("pendingPositions", positions);
        } else {
            nbt.method_10551("pendingPositions");
        }
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        class_1262.method_5429(nbt, inventory.heldStacks, registryLookup);
        loadMultiblockNbtData(nbt);
        loadAddonNbtData(nbt);
        loadColorFromNbt(nbt);
        
        updateEnergyContainer();
        
        redstonePowered = nbt.method_10577("redstone");
        energyStorage.amount = nbt.method_10537("energy_stored");
        targetDirection = class_2338.method_10092(nbt.method_10537("target_direction"));
        currentTarget = class_2338.method_10092(nbt.method_10537("target_position"));
        areaSize = nbt.method_10550("areaSize");
        yieldAddons = nbt.method_10550("yieldAddons");
        hunterAddons = nbt.method_10550("hunterAddons");
        hunterTargetMode = HunterTargetMode.fromValue(nbt.method_10550("hunterTargetMode"));
        hasCropFilterAddon = nbt.method_10577("cropAddon");
        hasSilkTouchAddon = nbt.method_10577("silkAddon");
        
        if (nbt.method_10545("pendingPositions")) {
            pendingArea = Arrays.stream(nbt.method_10565("pendingPositions")).mapToObj(class_2338::method_10092).collect(Collectors.toCollection(ArrayDeque::new));
        }
    }
    
    //region multiblock
    @Override
    public ArrayList<class_2338> getConnectedCores() {
        return coreBlocksConnected;
    }
    
    @Override
    public class_2350 getFacingForMultiblock() {
        var state = method_11010();
        return state.method_11654(class_2741.field_12525).method_10153();
    }
    
    @Override
    public class_2338 getPosForMultiblock() {
        return field_11867;
    }
    
    @Override
    public class_1937 getWorldForMultiblock() {
        return field_11863;
    }
    
    @Override
    public float getCoreQuality() {
        return this.coreQuality;
    }
    
    @Override
    public void setCoreQuality(float quality) {
        this.coreQuality = quality;
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryForMultiblock() {
        return inventory;
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorageForMultiblock(class_2350 direction) {
        return energyStorage;
    }
    
    @Override
    public List<class_2382> getCorePositions() {
        return List.of(
          new class_2382(1, 0, 0)
        );
    }
    //endregion
    
    // energyprovider
    @Override
    public EnergyApi.EnergyStorage getEnergyStorage(class_2350 direction) {
        return energyStorage;
    }
    
    //region addons
    @Override
    public List<class_2338> getConnectedAddons() {
        return connectedAddons;
    }
    
    @Override
    public List<class_2338> getOpenAddonSlots() {
        return openSlots;
    }
    
    @Override
    public class_2350 getFacingForAddon() {
        var state = method_11010();
        return state.method_11654(class_2741.field_12525).method_10153();
    }
    
    @Override
    public DynamicEnergyStorage getStorageForAddon() {
        return energyStorage;
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryForAddon() {
        return inventory;
    }
    
    @Override
    public ScreenProvider getScreenProvider() {
        return null;
    }
    
    @Override
    public List<class_2382> getAddonSlots() {
        return List.of(
          new class_2382(-1, 0, 0)
        );
    }
    
    @Override
    public BaseAddonData getBaseAddonData() {
        return addonData;
    }
    
    @Override
    public void setBaseAddonData(BaseAddonData data) {
        this.addonData = data;
    }
    
    @Override
    public long getDefaultCapacity() {
        return Oritech.CONFIG.laserArmConfig.energyCapacity();
    }
    
    @Override
    public long getDefaultInsertRate() {
        return Oritech.CONFIG.laserArmConfig.maxEnergyInsertion();
    }
    //endregion
    
    // region animation
    @Override
    public void registerControllers(AnimatableManager.ControllerRegistrar controllers) {
        controllers.add(animationController);
    }
    
    @Override
    public AnimatableInstanceCache getAnimatableInstanceCache() {
        return animatableInstanceCache;
    }
    
    private AnimationController<LaserArmBlockEntity> getAnimationController() {
        return new AnimationController<>(this, state -> {
            
            if (state.isCurrentAnimation(MachineBlockEntity.SETUP)) {
                if (state.getController().hasAnimationFinished()) {
                    state.setAndContinue(MachineBlockEntity.IDLE);
                } else {
                    return state.setAndContinue(MachineBlockEntity.SETUP);
                }
            }
            
            if (isActive(method_11010())) {
                if (isFiring()) {
                    return state.setAndContinue(MachineBlockEntity.WORKING);
                } else {
                    return state.setAndContinue(MachineBlockEntity.IDLE);
                }
            } else {
                return state.setAndContinue(MachineBlockEntity.PACKAGED);
            }
        }).triggerableAnim("setup", MachineBlockEntity.SETUP).setSoundKeyframeHandler(new AutoPlayingSoundKeyframeHandler<>());
    }
    
    @Override
    public void triggerSetupAnimation() {
        triggerAnim("base_controller", "setup");
    }
    
    @Override
    public void sendUpdate(SyncType type, class_3222 player) {
        
        if (type.equals(SyncType.GUI_TICK)) {
            super.sendUpdate(SyncType.GUI_OPEN, player);
            return;
        }
        
        super.sendUpdate(type, player);
    }
    
    public boolean isActive(class_2680 state) {
        return state.method_11654(ASSEMBLED);
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryStorage(class_2350 direction) {
        return inventory;
    }
    
    //endregion
    
    
    public class_2338 getCurrentTarget() {
        return currentTarget;
    }
    
    public class_243 getVisualTarget() {
        if (hunterAddons > 0 && currentLivingTarget != null) {
            return currentLivingTarget.method_33571().method_1023(0.5f, 0, 0.5f);
        } else {
            return getCurrentTarget().method_46558();
        }
    }
    
    
    @Override
    public class_2338 getPosForAddon() {
        return method_11016();
    }
    
    @Override
    public class_1937 getWorldForAddon() {
        return method_10997();
    }
    
    public boolean isFiring() {
        var idleTime = field_11863.method_8510() - lastFiredAt;
        return idleTime < 5;
    }
    
    @Override
    public int getTickUpdateInterval() {
        return 2;
    }
    
    public boolean isTargetingAtomicForge(class_2248 block) {
        return block.equals(BlockContent.ATOMIC_FORGE_BLOCK);
    }
    
    public boolean isTargetingDeepdrill(class_2248 block) {
        return block.equals(BlockContent.DEEP_DRILL_BLOCK);
    }
    
    public boolean isTargetingCatalyst(class_2248 block) {
        return block.equals(BlockContent.ENCHANTMENT_CATALYST_BLOCK);
    }
    
    public boolean isTargetingUnstableContainer(class_2248 block) {
        return block.equals(BlockContent.UNSTABLE_CONTAINER);
    }
    
    public boolean isTargetingEnergyContainer() {
        var storageCandidate = EnergyApi.BLOCK.find(field_11863, currentTarget, null);
        var block = field_11863.method_8320(currentTarget).method_26204();
        return storageCandidate != null || isTargetingAtomicForge(block) || isTargetingDeepdrill(block) || isTargetingCatalyst(block) || isTargetingUnstableContainer(block);
    }
    
    public boolean isTargetingBuddingAmethyst() {
        return field_11863.method_8320(currentTarget).method_26164(TagContent.LASER_ACCELERATED);
    }
    
    @Override
    public List<class_3545<class_2561, class_2561>> getExtraExtensionLabels() {
        if (areaSize == 1 && yieldAddons == 0 && hunterAddons == 0 && !hasSilkTouchAddon)
            return ScreenProvider.super.getExtraExtensionLabels();
        if (hunterAddons > 0)
            return List.of(
              new class_3545<>(class_2561.method_43469("title.oritech.machine.addon_range", (int) hunterRange()), class_2561.method_43471("tooltip.oritech.laser_arm.addon_hunter_range")),
              new class_3545<>(class_2561.method_43469("title.oritech.laser_arm.addon_hunter_damage", String.format("%.2f", getDamageTick())), class_2561.method_43471("tooltip.oritech.laser_arm.addon_hunter_damage")),
              new class_3545<>(class_2561.method_43469("title.oritech.machine.addon_looting", yieldAddons), class_2561.method_43471("tooltip.oritech.machine.addon_looting")));
        
        if (hasSilkTouchAddon) {
            return List.of(
              new class_3545<>(class_2561.method_43469("title.oritech.machine.addon_range", areaSize), class_2561.method_43471("tooltip.oritech.laser_arm.addon_range")),
              new class_3545<>(class_2561.method_43471("enchantment.minecraft.silk_touch"), class_2561.method_43471("tooltip.oritech.machine.addon_silk_touch")));
        }
        
        return List.of(
          new class_3545<>(class_2561.method_43469("title.oritech.machine.addon_range", areaSize), class_2561.method_43471("tooltip.oritech.laser_arm.addon_range")),
          new class_3545<>(class_2561.method_43469("title.oritech.machine.addon_fortune", yieldAddons), class_2561.method_43471("tooltip.oritech.machine.addon_fortune")));
    }
    
    @Override
    public List<GuiSlot> getGuiSlots() {
        return List.of(
          new GuiSlot(0, 117, 20, true),
          new GuiSlot(1, 117, 38, true),
          new GuiSlot(2, 117, 56, true));
    }
    
    @Override
    public float getDisplayedEnergyUsage() {
        return energyRequiredToFire();
    }
    
    @Override
    public float getProgress() {
        return 0;
    }
    
    @Override
    public boolean showProgress() {
        return false;
    }
    
    @Override
    public InventoryInputMode getInventoryInputMode() {
        return InventoryInputMode.FILL_LEFT_TO_RIGHT;
    }
    
    @Override
    public boolean inputOptionsEnabled() {
        return false;
    }
    
    @Override
    public class_1263 getDisplayedInventory() {
        return inventory;
    }
    
    @Override
    public float getDisplayedEnergyTransfer() {
        return energyStorage.maxInsert;
    }
    
    @Override
    public class_3917<?> getScreenHandlerType() {
        return ModScreens.LASER_SCREEN;
    }
    
    @Override
    public class_2769<class_2350> getBlockFacingProperty() {
        return class_2741.field_12525;
    }
    
    @Nullable
    @Override
    public class_1703 createMenu(int syncId, class_1661 playerInventory, class_1657 player) {
        return new UpgradableMachineScreenHandler(syncId, playerInventory, this);
    }
    
    @Override
    public void saveExtraData(class_2540 buf) {
        this.sendUpdate(SyncType.GUI_OPEN);
        buf.method_10807(field_11867);
        
    }
    
    @Override
    public class_2561 method_5476() {
        return class_2561.method_43470("");
    }
    
    @Override
    public int getComparatorEnergyAmount() {
        return (int) ((energyStorage.amount / (float) energyStorage.capacity) * 15);
    }
    
    @Override
    public int getComparatorSlotAmount(int slot) {
        if (inventory.heldStacks.size() <= slot) return 0;
        
        var stack = inventory.method_5438(slot);
        if (stack.method_7960()) return 0;
        
        return (int) ((stack.method_7947() / (float) stack.method_7914()) * 15);
    }
    
    @Override
    public int getComparatorProgress() {
        if (currentTarget == null || currentTarget.equals(class_2338.field_10980)) return 0;
        
        return (int) (currentTarget.method_10262(field_11867) / range) * 15;
    }
    
    @Override
    public int getComparatorActiveState() {
        var idleTicks = field_11863.method_8510() - lastFiredAt;
        return idleTicks > 3 ? 15 : 0;
    }
    
    @Override
    public boolean hasRedstoneControlAvailable() {
        return true;
    }
    
    @Override
    public int receivedRedstoneSignal() {
        if (redstonePowered) return 15;
        return field_11863.method_49804(field_11867);
    }
    
    @Override
    public String currentRedstoneEffect() {
        if (redstonePowered) return "tooltip.oritech.redstone_disabled";
        return "tooltip.oritech.redstone_enabled_direct";
    }
    
    public class_2338 getLaserHeadPosition() {
        var state = method_11010();
        var facing = state.method_11654(class_2741.field_12525);
        var offset = new class_2382(-1, 0, 0);
        return new class_2338(Geometry.offsetToWorldPosition(facing, offset, field_11867));
    }
    
    @Override
    public void onRedstoneEvent(boolean isPowered) {
        this.redstonePowered = isPowered;
    }
    
    @Override
    public ColorVariant getCurrentColor() {
        return currentColor;
    }
    
    @Override
    public void assignColor(ColorVariant color) {
        this.currentColor = color;
        
        if (this.field_11863 != null && !this.field_11863.method_8608()) {
            this.setChanged(false);
            this.sendUpdate(SyncType.SPARSE_TICK);
        }
    }
    
    public enum HunterTargetMode {
        HOSTILE_ONLY(1, "message.oritech.target_designator.hunter_hostile"),
        HOSTILE_NEUTRAL(2, "message.oritech.target_designator.hunter_neutral"),
        ALL(3, "message.oritech.target_designator.hunter_all");
        
        public final int value;
        public final String message;
        
        HunterTargetMode(int value, String message) {
            this.value = value;
            this.message = message;
        }
        
        private static final Map<Integer, HunterTargetMode> map = new HashMap<Integer, HunterTargetMode>();
        
        static {
            for (HunterTargetMode targetMode : HunterTargetMode.values())
                map.put(targetMode.value, targetMode);
        }
        
        public static HunterTargetMode fromValue(int i) {
            return map.getOrDefault(i, HOSTILE_ONLY);
        }
        
        public HunterTargetMode next() {
            return values()[(ordinal() + 1) % values().length];
        }
    }
    
    public static WorldPacketCodec<class_9129, class_1309> LASER_TARGET_PACKET_CODEC = new WorldPacketCodec<>() {
        @Override
        public class_1309 decode(class_9129 buf, @Nullable class_1937 world) {
            
            var id = buf.readInt();
            if (world != null && id >= 0) {
                var candidate = world.method_8469(id);
                if (candidate instanceof class_1309 livingEntity) {
                    return livingEntity;
                }
            }
            
            return null;
        }
        
        @Override
        public void encode(class_9129 buf, class_1309 value, @Nullable class_1937 world) {
            var id = value != null ? value.method_5628() : -1;
            buf.method_53002(id);
        }
    };
}
