package rearth.oritech.block.entity.pipes;

import net.minecraft.class_18;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2503;
import net.minecraft.class_2519;
import net.minecraft.class_2520;
import net.minecraft.class_2586;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_3545;
import net.minecraft.class_5558;
import net.minecraft.class_7225;
import net.minecraft.nbt.*;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.block.blocks.pipes.AbstractPipeBlock;
import rearth.oritech.block.blocks.pipes.GenericPipeBlock;
import rearth.oritech.block.entity.interaction.PipeBoosterBlockEntity;

import java.util.*;
import java.util.stream.Collectors;

import static rearth.oritech.block.blocks.pipes.GenericPipeBlock.NO_CONNECTION;


public abstract class GenericPipeInterfaceEntity extends class_2586 implements class_5558<GenericPipeInterfaceEntity> {
    
    public static final int MAX_SEARCH_COUNT = 2048;
    
    public class_2338 connectedBooster = class_2338.field_10980;

    private PipeBoosterBlockEntity cachedBooster;

    public GenericPipeInterfaceEntity(class_2591<?> type, class_2338 pos, class_2680 state) {
        super(type, pos, state);
    }

	public boolean isBoostAvailable() {
		var booster = tryGetCachedBooster();
		return booster != null && booster.canUseBoost();
	}

	public void onBoostUsed() {
		var booster = tryGetCachedBooster();
		if (booster != null) booster.useBoost();
	}

	@Nullable
	private PipeBoosterBlockEntity tryGetCachedBooster() {

		// booster was removed
		if (cachedBooster != null && cachedBooster.method_11015()) {
			cachedBooster = null;
			connectedBooster = class_2338.field_10980;
			return null;
		}

		if (connectedBooster == class_2338.field_10980) {  // no booster set
			if (cachedBooster != null) cachedBooster = null;
			return null;
		} else if (cachedBooster == null) { // booster freshly set
			var candidate = Objects.requireNonNull(field_11863).method_8321(connectedBooster);
			if (candidate instanceof PipeBoosterBlockEntity booster) {
				cachedBooster = booster;
				return cachedBooster;
			} else {
				connectedBooster = class_2338.field_10980;
				return null;
			}
		} else {    // no change
			return cachedBooster;
		}
	}

    public static void addNode(class_1937 world, class_2338 pos, boolean isInterface, class_2680 newState, PipeNetworkData data) {
        Oritech.LOGGER.debug("registering/updating node: " + pos);

        data.pipes.add(pos);
        var connectedMachines = new HashSet<class_2338>(6);
        var block = (AbstractPipeBlock) newState.method_26204();
        for (var neighbor : class_2350.values()) {
            var neighborPos = pos.method_10093(neighbor);
            var neighborMap = data.machinePipeNeighbors.getOrDefault(neighborPos, new HashSet<>());
            if (block.hasMachineInDirection(neighbor, world, pos, block.apiValidationFunction())) {
                if (block.isConnectingInDirection(newState, neighbor, pos, world, false))
                    connectedMachines.add(pos.method_10093(neighbor));

                neighborMap.add(neighbor.method_10153());
            } else {
                neighborMap.remove(neighbor.method_10153());
            }

            if (!neighborMap.isEmpty()) data.machinePipeNeighbors.put(neighborPos, neighborMap);
            else data.machinePipeNeighbors.remove(neighborPos);
        }

        if (isInterface) {
            data.machineInterfaces.put(pos, connectedMachines);
        } else {
            data.machineInterfaces.remove(pos);
        }

        updateFromNode(world, pos, data);
    }

    public static void removeNode(class_1937 world, class_2338 pos, boolean wasInterface, class_2680 oldState, PipeNetworkData data) {
        Oritech.LOGGER.debug("removing node: " + pos + " | " + wasInterface);

        var oldNetwork = data.pipeNetworkLinks.getOrDefault(pos, -1);

        data.pipes.remove(pos);
        if (wasInterface) data.machineInterfaces.remove(pos);

        removeStaleMachinePipeNeighbors(pos, data);

        data.pipeNetworks.remove(oldNetwork);
        data.pipeNetworkInterfaces.remove(oldNetwork);
        data.pipeNetworkLinks.remove(pos);

        // re-calculate old network, is either shorter or split into multiple ones (starting from ones this block was connected to)
        if (oldNetwork != -1) {
            var block = oldState.method_26204();
            for (var direction : class_2350.values()) {
                if (block instanceof GenericPipeBlock pipeBlock && oldState.method_11654(pipeBlock.directionToProperty(direction)) == NO_CONNECTION) {
                    continue;
                }

                updateFromNode(world, pos.method_10093(direction), data);
            }
        }

        data.method_80();
    }

    private static void updateFromNode(class_1937 world, class_2338 pos, PipeNetworkData data) {

        var searchInstance = new FloodFillSearch(pos, data.pipes, world);
        var foundNetwork = new HashSet<>(searchInstance.complete());
        var foundMachines = findConnectedMachines(foundNetwork, data);

        Oritech.LOGGER.debug("Nodes:    " + foundNetwork.size() + " | " + foundNetwork);
        Oritech.LOGGER.debug("Machines: " + foundMachines.size() + " | " + foundMachines.stream().map(elem -> elem.method_15442() + ":" + elem.method_15441()).toList());

        var netID = foundNetwork.hashCode();
        data.pipeNetworks.put(netID, foundNetwork);
        data.pipeNetworkInterfaces.put(netID, foundMachines);

        // these networks will be replaced, since these nodes now belong to the new network
        var networksToRemove = new HashSet<Integer>();

        for (var node : foundNetwork) {
            networksToRemove.add(data.pipeNetworkLinks.getOrDefault(node, -1));
            data.pipeNetworkLinks.put(node, netID);
        }

        networksToRemove.stream().filter(i -> i != -1 && i != netID).forEach(i -> {
            data.pipeNetworks.remove(i);
            data.pipeNetworkInterfaces.remove(i);
        });

        data.method_80();
    }

    private static Set<class_3545<class_2338, class_2350>> findConnectedMachines(Set<class_2338> network, PipeNetworkData data) {

        var res = new HashSet<class_3545<class_2338, class_2350>>();

        for (var node : network) {
            if (data.machineInterfaces.containsKey(node)) {
                for (var machinePos : data.machineInterfaces.get(node)) {
                    var offset = machinePos.method_10059(node);
                    var direction = class_2350.method_50026(offset.method_10263(), offset.method_10264(), offset.method_10260()).method_10153();
                    res.add(new class_3545<>(machinePos, direction));
                }
            }
        }

        return res;
    }

    public static Set<class_3545<class_2338, class_2350>> findNetworkTargets(class_2338 from, PipeNetworkData data) {
        var connectedNetwork = data.pipeNetworkLinks.getOrDefault(from, -1);
        if (connectedNetwork == -1) return new HashSet<>();

        return data.pipeNetworkInterfaces.get(connectedNetwork);
    }

    /**
     * Removes any stale machine -> neighboring pipes mappings
     * Used when a pipe node is destroyed
     *
     * @param pos  position of the destroyed node
     * @param data network data
     */
    public static void removeStaleMachinePipeNeighbors(class_2338 pos, PipeNetworkData data) {
        for (var neighbor : class_2350.values()) {
            var machine = pos.method_10093(neighbor);
            var machineNeighbors = data.machinePipeNeighbors.get(machine);
            if (machineNeighbors == null) continue;

            machineNeighbors.remove(class_2350.method_58251(class_243.method_24954(pos.method_10059(machine))));
            if (machineNeighbors.isEmpty())
                data.machinePipeNeighbors.remove(machine);
            else
                data.machinePipeNeighbors.put(machine, machineNeighbors);
        }
    }

    private static class FloodFillSearch {

        final HashSet<class_2338> checkedPositions = new HashSet<>();
        final HashSet<class_2338> nextTargets = new HashSet<>();
        final Deque<class_2338> foundTargets = new ArrayDeque<>();
        final HashSet<class_2338> pipes;
        final class_1937 world;

        public FloodFillSearch(class_2338 startPosition, HashSet<class_2338> pipes, class_1937 world) {
            this.pipes = pipes;
            this.world = world;
            nextTargets.add(startPosition);
        }

        public Deque<class_2338> complete() {
            var active = true;
            while (active) {
                active = !nextGeneration();
            }

            return foundTargets;
        }

        // returns true when done
        @SuppressWarnings("unchecked")
        public boolean nextGeneration() {

            var currentGeneration = (HashSet<class_2338>) nextTargets.clone();

            for (var target : currentGeneration) {
                if (isValidTarget(target)) {
                    foundTargets.addLast(target);
                    addNeighborsToQueue(target);
                }

                checkedPositions.add(target);
                nextTargets.remove(target);
            }

            if (cutoffSearch()) nextTargets.clear();

            return nextTargets.isEmpty();
        }

        private boolean cutoffSearch() {
            return foundTargets.size() >= MAX_SEARCH_COUNT;
        }

        private boolean isValidTarget(class_2338 target) {
            return pipes.contains(target);
        }

        private void addNeighborsToQueue(class_2338 self) {
            var targetState = world.method_8320(self);

            if (!(targetState.method_26204() instanceof AbstractPipeBlock targetBlock)) return;
            for (var direction : class_2350.values()) {
                var neighbor = self.method_10093(direction);
                if (checkedPositions.contains(neighbor)) continue;
                if (!isValidTarget(neighbor)) {
                    checkedPositions.add(neighbor);
                    continue;
                }

                // check if the target can connect to the neighbor
                if (!targetBlock.isConnectingInDirection(targetState, direction, self, world, false)) continue;

                nextTargets.add(neighbor);
            }
        }
    }

    public static final class PipeNetworkData extends class_18 {
        public final HashMap<class_2338, Integer> pipeNetworkLinks = new HashMap<>(); // which blockpos belongs to which network (ID)
        public final HashSet<class_2338> pipes = new HashSet<>();
        public final HashMap<class_2338, Set<class_2338>> machineInterfaces = new HashMap<>(); // list of machines per interface/connection block
        public final HashMap<Integer, Set<class_2338>> pipeNetworks = new HashMap<>();   // networks are never updated, and instead always replaced by new ones with different ids
        public final HashMap<Integer, Set<class_3545<class_2338, class_2350>>> pipeNetworkInterfaces = new HashMap<>(); // list of machines that are connected to the network

        public final HashMap<class_2338, Set<class_2350>> machinePipeNeighbors = new HashMap<>(); // List of neighboring pipes per machine, and the direction they are in. Missing direction means no connection

        @Override
        public int hashCode() {
            int result = pipeNetworkLinks.hashCode();
            result = 31 * result + pipes.hashCode();
            result = 31 * result + machineInterfaces.hashCode();
            result = 31 * result + pipeNetworks.hashCode();
            result = 31 * result + pipeNetworkInterfaces.hashCode();
            return result;
        }

        public static class_8645<PipeNetworkData> TYPE = new class_8645<>(PipeNetworkData::new, PipeNetworkData::fromNbt, null);

        public static PipeNetworkData fromNbt(class_2487 nbt, class_7225.class_7874 registryLookup) {

            var result = new PipeNetworkData();

            if (nbt.method_10573("pipeNetworkLinks", class_2520.field_33259)) {
                var pipeNetworkLinksList = nbt.method_10554("pipeNetworkLinks", class_2520.field_33260);
                for (var element : pipeNetworkLinksList) {
                    var entry = (class_2487) element;
                    var pos = class_2338.method_10092(entry.method_10537("pos"));
                    var id = entry.method_10550("id");
                    result.pipeNetworkLinks.put(pos, id);
                }
            }

            // Deserialize pipes
            if (nbt.method_10573("pipes", class_2520.field_33259)) {
                var pipesList = nbt.method_10554("pipes", class_2520.field_33254);
                pipesList.stream().map(element -> class_2338.method_10092(((class_2503) element).method_10699())).forEach(result.pipes::add);
            }

            // Deserialize machineInterfaces
            if (nbt.method_10573("machineInterfaces", class_2520.field_33260)) {
                var machineInterfacesNbt = nbt.method_10562("machineInterfaces");
                for (var key : machineInterfacesNbt.method_10541()) {
                    var interfacePos = class_2338.method_10092(Long.parseLong(key));
                    var machinesArray = machineInterfacesNbt.method_10565(key);
                    var machines = Arrays.stream(machinesArray)
                                     .mapToObj(class_2338::method_10092)
                                     .collect(Collectors.toSet());
                    result.machineInterfaces.put(interfacePos, machines);
                }
            }

            // Deserialize pipeNetworks
            if (nbt.method_10573("pipeNetworks", class_2520.field_33260)) {
                var pipeNetworksNbt = nbt.method_10562("pipeNetworks");
                for (var key : pipeNetworksNbt.method_10541()) {
                    var id = Integer.parseInt(key);
                    var networkArray = pipeNetworksNbt.method_10565(key);
                    var network = Arrays.stream(networkArray)
                                    .mapToObj(class_2338::method_10092)
                                    .collect(Collectors.toSet());
                    result.pipeNetworks.put(id, network);
                }
            }

            // Deserialize pipeNetworkInterfaces
            if (nbt.method_10573("pipeNetworkInterfaces", class_2520.field_33260)) {
                var pipeNetworkInterfacesNbt = nbt.method_10562("pipeNetworkInterfaces");
                for (var key : pipeNetworkInterfacesNbt.method_10541()) {
                    var id = Integer.parseInt(key);
                    var interfacesList = pipeNetworkInterfacesNbt.method_10554(key, class_2520.field_33260);
                    var interfaces = new HashSet<class_3545<class_2338, class_2350>>();
                    for (var interfaceElement : interfacesList) {
                        var pairNbt = (class_2487) interfaceElement;
                        var pos = class_2338.method_10092(pairNbt.method_10537("pos"));
                        var direction = class_2350.method_10168(pairNbt.method_10558("direction"));
                        interfaces.add(new class_3545<>(pos, direction));
                    }
                    result.pipeNetworkInterfaces.put(id, interfaces);
                }
            }

            // Deserialize machinePipeNeighbors
            if (nbt.method_10573("machinePipeNeighbors", class_2520.field_33260)) {
                var connectionPipeNeighborsNbt = nbt.method_10562("machinePipeNeighbors");
                for (var key : connectionPipeNeighborsNbt.method_10541()) {
                    var pos = class_2338.method_10092(Long.parseLong(key));
                    var neighborsList = connectionPipeNeighborsNbt.method_10554(key, class_2520.field_33258);
                    var neighbors = new HashSet<class_2350>();
                    for (var neighborElement : neighborsList) {
                        var direction = class_2350.method_10168(neighborElement.method_10714());
                        neighbors.add(direction);
                    }
                    result.machinePipeNeighbors.put(pos, neighbors);
                }
            }

            result.method_80();

            return result;
        }

        @Override
        public class_2487 method_75(class_2487 nbt, class_7225.class_7874 registryLookup) {

            // Serialize pipeNetworkLinks
            var pipeNetworkLinksList = new class_2499();
            pipeNetworkLinks.forEach((pos, id) -> {
                var entry = new class_2487();
                entry.method_10544("pos", pos.method_10063());
                entry.method_10569("id", id);
                pipeNetworkLinksList.add(entry);
            });
            nbt.method_10566("pipeNetworkLinks", pipeNetworkLinksList);

            // Serialize pipes
            var pipesList = new class_2499();
            pipes.forEach(pos -> pipesList.add(class_2503.method_23251(pos.method_10063())));
            nbt.method_10566("pipes", pipesList);

            // Serialize machineInterfaces
            var machineInterfacesNbt = new class_2487();
            machineInterfaces.forEach((interfacePos, machines) -> {
                machineInterfacesNbt.method_10538(Long.toString(interfacePos.method_10063()), machines.stream().map(class_2338::method_10063).collect(Collectors.toList()));
            });
            nbt.method_10566("machineInterfaces", machineInterfacesNbt);

            // Serialize pipeNetworks
            var pipeNetworksNbt = new class_2487();
            pipeNetworks.forEach((id, network) -> {
                pipeNetworksNbt.method_10538(id.toString(), network.stream().map(class_2338::method_10063).collect(Collectors.toList()));
            });
            nbt.method_10566("pipeNetworks", pipeNetworksNbt);

            // Serialize pipeNetworkInterfaces
            var pipeNetworkInterfacesNbt = new class_2487();
            pipeNetworkInterfaces.forEach((id, interfaces) -> {
                var interfacesList = new class_2499();
                interfaces.forEach(pair -> {
                    var pairNbt = new class_2487();
                    pairNbt.method_10544("pos", pair.method_15442().method_10063());
                    pairNbt.method_10582("direction", pair.method_15441().method_10151());
                    interfacesList.add(pairNbt);
                });
                pipeNetworkInterfacesNbt.method_10566(id.toString(), interfacesList);
            });
            nbt.method_10566("pipeNetworkInterfaces", pipeNetworkInterfacesNbt);

            // Serialize machinePipeNeighbors
            var connectionPipeNeighborsNbt = new class_2487();
            machinePipeNeighbors.forEach((pos, neighbors) -> {
                var neighborsList = new class_2499();
                neighbors.forEach(direction -> {
                    var nbtElement = class_2519.method_23256(direction.method_10151());
                    neighborsList.add(nbtElement);
                });
                connectionPipeNeighborsNbt.method_10566(Long.toString(pos.method_10063()), neighborsList);
            });
            nbt.method_10566("machinePipeNeighbors", connectionPipeNeighborsNbt);

            return nbt;
        }
    }

}
