package rearth.oritech.block.base.entity;

import dev.architectury.fluid.FluidStack;
import dev.architectury.hooks.fluid.FluidStackHooks;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.EnergyApi;
import rearth.oritech.api.fluid.containers.SimpleInOutFluidStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.entity.generators.SteamEngineEntity;
import rearth.oritech.init.BlockContent;
import rearth.oritech.init.recipes.OritechRecipe;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import net.minecraft.class_1262;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_3545;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_7225;

public abstract class UpgradableGeneratorBlockEntity extends UpgradableMachineBlockEntity {
    
    @SyncField
    public int currentMaxBurnTime; // needed only for progress display and animation speed
    private List<class_1799> pendingOutputs = new ArrayList<>(); // used if a recipe produces a byproduct at the end
    
    // this is used just for steam
    @SyncField(SyncType.GUI_OPEN)
    public boolean isProducingSteam = false;
    @SyncField(SyncType.GUI_TICK)
    public final SimpleInOutFluidStorage boilerStorage = new SimpleInOutFluidStorage((long) (Oritech.CONFIG.generators.steamEngineData.steamBoilerCapacityBuckets() * FluidStackHooks.bucketAmount()), this::method_5431) {
        @Override
        public long insert(FluidStack toInsert, boolean simulate) {
            if (!boilerAcceptsInput(toInsert.getFluid())) return 0L;
            return super.insert(toInsert, simulate);
        }
    };
    
    // speed multiplier increases output rate and reduces burn time by same percentage
    // efficiency multiplier only increases burn time
    public UpgradableGeneratorBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state, int energyPerTick) {
        super(type, pos, state, energyPerTick);
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        
        // check remaining burn time
        // if burn time is zero, try to consume item thus adding burn time
        // if burn time is remaining, use up one tick of it
        
        if (world.field_9236 || !isActive(state) || disabledViaRedstone) return;
        
        if (progress == 0 && canFitEnergy())
            tryConsumeInput();
        
        // progress var is used as remaining burn time
        if (progress > 0) {
            if (canFitEnergy()) {
                
                progress--;
                produceEnergy();
                lastWorkedAt = world.method_8510();
                
                if (progress == 0) {
                    burningFinished();
                }
                method_5431();
            }
        }
        
        outputEnergy();
    }
    
    protected void tryConsumeInput() {
        
        if (isProducingSteam && (boilerStorage.getInStack().getAmount() == 0 || boilerStorage.getOutStack().getAmount() >= boilerStorage.getCapacity())) return;
        
        var recipeCandidate = getRecipe();
        if (recipeCandidate.isEmpty())
            currentRecipe = OritechRecipe.DUMMY;     // reset recipe when invalid or no input is given
        
        
        if (recipeCandidate.isPresent()) {
            // this is separate so that progress is not reset when out of energy
            var activeRecipe = recipeCandidate.get().comp_1933();
            currentRecipe = activeRecipe;
            
            // speed -> lower = faster, efficiency -> lower = better
            var recipeTime = (int) (currentRecipe.getTime() * getSpeedMultiplier() * (1 / getEfficiencyMultiplier()));
            progress = recipeTime;
            currentMaxBurnTime = recipeTime;
            
            // remove inputs
            for (int i = 0; i < activeRecipe.getInputs().size(); i++) {
                var taken = class_1262.method_5430(getInputView(), i, 1);  // amount is not configurable, because ingredient doesn't parse amount in recipe
            }
            pendingOutputs = activeRecipe.getResults();
            
            method_5431();
            
        }
    }
    
    protected void burningFinished() {
        produceResultItems();
    }
    
    protected void produceResultItems() {
        if (!pendingOutputs.isEmpty()) {
            for (var stack : pendingOutputs) {
                this.inventory.insert(stack, false);
            }
        }
        
        pendingOutputs.clear();
    }
    
    @Override
    public void gatherAddonStats(List<AddonBlock> addons) {
        isProducingSteam = false;
        super.gatherAddonStats(addons);
    }
    
    @Override
    public void getAdditionalStatFromAddon(AddonBlock addonBlock) {
        super.getAdditionalStatFromAddon(addonBlock);
        if (addonBlock.state().method_26204() == BlockContent.STEAM_BOILER_ADDON) {
            isProducingSteam = true;
            field_11863.method_8452(addonBlock.pos(), addonBlock.state().method_26204());
        }
    }
    
    // ensure that insertion is disabled, and instead upgrade extraction rates
    @Override
    public void updateEnergyContainer() {
        super.updateEnergyContainer();
        
        var insert = energyStorage.maxInsert;
        energyStorage.maxExtract = getDefaultExtractionRate() + insert;
        energyStorage.maxInsert = 0;
        
    }
    
    // check if the energy can fit
    protected boolean canFitEnergy() {
        if (isProducingSteam) return true;
        var produced = calculateEnergyUsage();
        return energyStorage.capacity >= energyStorage.amount + produced;
    }
    
    // gives energy in this case
    @SuppressWarnings("lossy-conversions")
    protected void produceEnergy() {
        var produced = calculateEnergyUsage();
        if (isProducingSteam) {
            // yes this will void excess steam. Generators will only stop producing when the RF storage is full, not the steam storage
            // this is by design and supposed to be one of the negatives of steam production
            produced *= Oritech.CONFIG.generators.steamEngineData.rfToSteamRatio();
            produced *= SteamEngineEntity.STEAM_AMOUNT_MULTIPLIER;
            
            var extracted = boilerStorage.getInputContainer().extract(FluidStack.create(class_3612.field_15910.method_15751(), Math.round(produced)), false);
            boilerStorage.getOutputContainer().insert(FluidStack.create(SteamEngineEntity.getUsedSteamFluid(), extracted), false);
        } else {
            energyStorage.amount += produced;
        }
    }
    
    // returns energy production in this case
    @Override
    protected float calculateEnergyUsage() {
        return energyPerTick * (1 / getSpeedMultiplier());
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        nbt.method_10569("storedBurn", currentMaxBurnTime);
        boilerStorage.writeNbt(nbt, "");
        nbt.method_10556("steamAddon", isProducingSteam);
        
        var resList = new class_2499();
        for (var stack : pendingOutputs) {
            var data = stack.method_57358(registryLookup);
            resList.add(data);
        }
        nbt.method_10566("pendingResults", resList);
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        currentMaxBurnTime = nbt.method_10550("storedBurn");
        boilerStorage.readNbt(nbt, "");
        isProducingSteam = nbt.method_10577("steamAddon");
        
        var storedResults = nbt.method_10554("pendingResults", class_2520.field_33260);
        for (var elem : storedResults) {
            var compound = (class_2487) elem;
            var stack = class_1799.method_57360(registryLookup, compound).get();
            pendingOutputs.add(stack);
        }
    }
    
    protected abstract Set<class_3545<class_2338, class_2350>> getOutputTargets(class_2338 pos, class_1937 world);
    
    protected void outputEnergy() {
        if (energyStorage.getAmount() <= 0) return;
        
        var moved = 0L;
        
        // todo caching for targets? Used to be BlockApiCache.create()
        for (var target : getOutputTargets(field_11867, field_11863)) {
            var candidate = EnergyApi.BLOCK.find(field_11863, target.method_15442(), target.method_15441());
            if (candidate != null)
                moved += EnergyApi.transfer(energyStorage, candidate, Long.MAX_VALUE, false);
        }
        
        if (moved > 0)
            this.method_5431();
        
    }
    
    public boolean boilerAcceptsInput(class_3611 fluid ){
        return fluid.equals(class_3612.field_15910);
    }
    
    @Override
    public float getProgress() {
        return 1 - ((float) progress / currentMaxBurnTime);
    }
    
    public int getCurrentMaxBurnTime() {
        return currentMaxBurnTime;
    }
    
    public void setCurrentMaxBurnTime(int currentMaxBurnTime) {
        this.currentMaxBurnTime = currentMaxBurnTime;
    }
    
    @Override
    public long getDefaultInsertRate() {
        return 0;
    }
    
    @Override
    public float getDisplayedEnergyTransfer() {
        return energyStorage.maxExtract;
    }
    
    @Override
    public boolean showEnergy() {
        if (this.energyStorage.maxExtract <= 0 && !isProducingSteam) return false;
        return super.showEnergy();
    }
    
    @Override
    protected float getAnimationSpeed() {
        
        if (currentMaxBurnTime <= 0) return 1;
        var recipeTicks = currentMaxBurnTime;
        var animationTicks = 60f;    // 3s, length which all animations are defined as
        return animationTicks / recipeTicks * Oritech.CONFIG.generators.animationSpeedMultiplier();
    }
}
