package rearth.oritech.block.entity.pipes;

import org.apache.commons.lang3.time.StopWatch;
import rearth.oritech.Oritech;
import rearth.oritech.api.item.ItemApi;
import rearth.oritech.api.item.ItemApi.InventoryStorage;
import rearth.oritech.api.networking.NetworkManager;
import rearth.oritech.block.blocks.pipes.AbstractPipeBlock;
import rearth.oritech.block.blocks.pipes.ExtractablePipeConnectionBlock;
import rearth.oritech.block.blocks.pipes.item.ItemPipeBlock;
import rearth.oritech.block.blocks.pipes.item.ItemPipeConnectionBlock;
import rearth.oritech.init.BlockContent;
import rearth.oritech.init.BlockEntitiesContent;
import java.util.*;
import java.util.stream.IntStream;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_3545;
import net.minecraft.class_5455;
import net.minecraft.class_8710;

public class ItemPipeInterfaceEntity extends ExtractablePipeInterfaceEntity {
    
    private static final int TRANSFER_AMOUNT = Oritech.CONFIG.itemPipeTransferAmount();
    private static final int TRANSFER_PERIOD = Oritech.CONFIG.itemPipeIntervalDuration();
    
    private List<class_3545<ItemApi.InventoryStorage, class_2338>> filteredTargetItemStorages;
    
    // item path cache (invalidated on network update)
    private final HashMap<class_2338, class_3545<ArrayList<class_2338>, Integer>> cachedTransferPaths = new HashMap<>();
    private final boolean renderItems;
    
    private static final HashMap<class_2338, Long> blockedUntil = new HashMap<>();   // used to fake item movement in transparent pipes
    
    // client only
    public Set<RenderStackData> activeStacks = new HashSet<>();
    
    public ItemPipeInterfaceEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.ITEM_PIPE_ENTITY, pos, state);
        this.renderItems = state.method_26204().equals(BlockContent.TRANSPARENT_ITEM_PIPE_CONNECTION);
        
    }
    
    @SuppressWarnings("DataFlowIssue")
    @Override
    public void tick(class_1937 world, class_2338 pos, class_2680 state, GenericPipeInterfaceEntity blockEntity) {
        var block = (ExtractablePipeConnectionBlock) state.method_26204();
        if (world.field_9236 || !block.isExtractable(state))
            return;
        
        // boosted pipe works every tick, otherwise only every N tick
        if ((world.method_8510() + this.field_11867.method_10063()) % TRANSFER_PERIOD != 0 && !isBoostAvailable())
            return;
        
        // find first itemstack from connected invs (that can be extracted)
        // try to move it to one of the destinations
        
        var data = ItemPipeBlock.ITEM_PIPE_DATA.getOrDefault(world.method_27983().method_29177(), new PipeNetworkData());
        
        var sources = data.machineInterfaces.getOrDefault(pos, new HashSet<>());
        var stackToMove = class_1799.field_8037;
        ItemApi.InventoryStorage moveFromInventory = null;
        class_2338 takenFrom = null;
        var moveCapacity = isBoostAvailable() ? 64 : TRANSFER_AMOUNT;
        
        var hasMotor = state.method_11654(ItemPipeConnectionBlock.HAS_MOTOR);
        
        for (var sourcePos : sources) {
            var blockedTimer = blockedUntil.getOrDefault(sourcePos, 0L);
            if (world.method_8510() < blockedTimer) continue;
            
            if (blockedTimer > 0)   // if timer has expired but was set
                blockedUntil.remove(sourcePos);
            
            var offset = pos.method_10059(sourcePos);
            var direction = class_2350.method_50026(offset.method_10263(), offset.method_10264(), offset.method_10260());
            if (!block.isSideExtractable(state, direction.method_10153())) continue;
            var inventory = ItemApi.BLOCK.find(world, sourcePos, direction);
            if (inventory == null || !inventory.supportsExtraction()) continue;
            
            for (int i = 0; i < inventory.getSlotCount(); i++) {
                var slotStack = inventory.getStackInSlot(i);
                if (slotStack.method_7960()) continue;
                var canTake = inventory.extractFromSlot(slotStack.method_46651(moveCapacity), i, true);
                if (canTake > 0) {
                    stackToMove = slotStack.method_46651(canTake);
                    moveFromInventory = inventory;
                    takenFrom = sourcePos;
                } else {
                    stackToMove = class_1799.field_8037;
                }
                
                if (stackToMove.method_7960()) continue;
                
                var targets = findNetworkTargets(pos, data);
                if (targets == null) {
                    System.err.println("Yeah your pipe network likely is too long. At: " + this.method_11016());
                    return;
                }
                
                var netHash = targets.hashCode();
                
                if (netHash != filteredTargetsNetHash || filteredTargetItemStorages == null) {
                    filteredTargetItemStorages = targets.stream()
                                                   .filter(target -> {
                                                       var targetDir = target.method_15441();
                                                       var pipePos = target.method_15442().method_10081(targetDir.method_10163());
                                                       var pipeState = world.method_8320(pipePos);
                                                       if (!(pipeState.method_26204() instanceof ItemPipeConnectionBlock itemBlock))
                                                           return true;   // edge case, this should never happen
                                                       var extracting = itemBlock.isSideExtractable(pipeState, targetDir.method_10153());
                                                       return !extracting;
                                                   })
                                                   .map(target -> new class_3545<>(ItemApi.BLOCK.find(world, target.method_15442(), target.method_15441()), target.method_15442()))
                                                   .filter(obj -> Objects.nonNull(obj.method_15442()) && obj.method_15442().supportsInsertion())
                                                   .sorted(Comparator.comparingInt(a -> a.method_15441().method_19455(pos)))
                                                   .toList();
                    
                    filteredTargetsNetHash = netHash;
                    cachedTransferPaths.clear();
                }
                
                var toMove = stackToMove.method_7947();
                var moved = 0;
                
                for (var storagePair : filteredTargetItemStorages) {
                    if (storagePair.method_15442().equals(moveFromInventory))
                        continue;    // skip when targeting same machine
                    
                    var targetStorage = storagePair.method_15442();
                    var wasEmptyStorage = IntStream.range(0, targetStorage.getSlotCount()).allMatch(slot -> targetStorage.getStackInSlot(slot).method_7960());
                    
                    var inserted = targetStorage.insert(stackToMove, false);
                    toMove -= inserted;
                    moved += inserted;
                    
                    if (inserted > 0) {
                        onItemMoved(this.field_11867, takenFrom, storagePair.method_15441(), data.pipeNetworks.getOrDefault(data.pipeNetworkLinks.getOrDefault(this.field_11867, 0), new HashSet<>()), world, stackToMove.method_7909(), inserted, wasEmptyStorage);
                    }
                    
                    if (toMove <= 0) break;  // target has been found for all items
                }
                var extracted = moveFromInventory.extract(stackToMove.method_46651(moved), false);
                
                if (extracted != moved) {
                    Oritech.LOGGER.warn("Invalid state while transferring inventory. Caused at position {}", pos);
                }
                
                // only move one slot content
                if (moved > 0)
                    break;
                
                // only try to move the first non-empty stack without motors
                if (!hasMotor)
                    break;
            }
        }
        
        if (moveCapacity > TRANSFER_AMOUNT) onBoostUsed();
        
    }
    
    private void onItemMoved(class_2338 startPos, class_2338 from, class_2338 to, Set<class_2338> network, class_1937 world, class_1792 moved, int movedCount, boolean wasEmpty) {
        if (!renderItems) return;
        var path = cachedTransferPaths.computeIfAbsent(to, ignored -> calculatePath(startPos, from, to, network, world));
        if (path == null) return;
        
        var codedPath = path.method_15442();
        var pathLength = 0;
        for (int i = 0; i < codedPath.size() - 1; i++) {
            var pathPos = codedPath.get(i);
            var nextPathPos = codedPath.get(i + 1);
            pathLength += nextPathPos.method_19455(pathPos);
        }
        var packet = new RenderStackData(field_11867, new class_1799(moved, movedCount), codedPath, world.method_8510(), pathLength);
        NetworkManager.sendBlockHandle(this, packet);
        
        if (wasEmpty) {
            var arrivalTime = world.method_8510() + (int) calculatePathLength(path.method_15441());
            blockedUntil.putIfAbsent(to, arrivalTime);
        }
        
    }
    
    public static double calculatePathLength(int pathBlocksCount) {
        return Math.pow(pathBlocksCount * 32, 0.6);
    }
    
    // return pair is optimized path and total path length
    private static class_3545<ArrayList<class_2338>, Integer> calculatePath(class_2338 startPos, class_2338 from, class_2338 to, Set<class_2338> network, class_1937 world) {
        
        if (network.isEmpty() || !network.contains(startPos)) {
            Oritech.LOGGER.warn("tried to calculate invalid item pipe from: {} to {} with network size: {}", startPos, to, network.size());
            return null;
        }
        
        var length = 1;
        
        var path = new LinkedList<class_2338>();
        path.add(startPos);
        
        var visited = new HashSet<class_2338>();
        
        var watch = new StopWatch();
        watch.start();
        
        for (int i = 0; i < network.size() * 3; i++) {
            
            var currentPos = path.peekLast();
            
            if (currentPos == null || currentPos.method_19455(to) == 1) { // target reached (or invalid)
                break;
            }
            
            visited.add(currentPos);
            
            var currentPosState = world.method_8320(currentPos);
            if (!(currentPosState.method_26204() instanceof AbstractPipeBlock pipeBlock)) break;
            
            // collect potential edges in graph, ordered by basic cost heuristic (manhattan dist to target)
            var openEdges = getNeighbors(currentPos).stream()
                              .filter(network::contains)
                              .filter(candidate -> !visited.contains(candidate))
                              .filter(candidate -> pipeBlock.isConnectingInDirection(currentPosState, getDirectionFromOffset(currentPos, candidate), currentPos, world, false))
                              .sorted(Comparator.comparingInt(a -> a.method_19455(to)))
                              .toArray(class_2338[]::new);
            
            if (openEdges.length == 0) {    // dead end, go back
                path.pollLast();
            } else {
                path.add(openEdges[0]);
                length++;
            }
            
        }
        
        path.addFirst(from);
        path.add(to);
        
        // compact path (by removing straight segments)
        var result = optimizePath(path);
        
        watch.stop();
        
        Oritech.LOGGER.debug("pathsize: {} success: {} time ms: {}", result.size(), path.size() > 2, watch.getNanoTime() / 1_000_000f);
        return new class_3545<>(result, path.size());
    }
    
    private static ArrayList<class_2338> optimizePath(LinkedList<class_2338> path) {
        var result = new ArrayList<class_2338>();
        if (path.isEmpty()) {
            return result;
        }
        
        var iterator = path.iterator();
        var first = iterator.next();
        result.add(first);
        
        if (!iterator.hasNext()) {
            return result;
        }
        
        var current = iterator.next();
        var currentDirection = current.method_10059(first);
        
        while (iterator.hasNext()) {
            var next = iterator.next();
            var nextDirection = next.method_10059(current);
            
            if (!nextDirection.equals(currentDirection)) {
                result.add(current);
                currentDirection = nextDirection;
            }
            
            current = next;
        }
        
        result.add(current);
        return result;
    }
    
    // returns all neighboring positions except up
    private static List<class_2338> getNeighbors(class_2338 pos) {
        return Arrays.asList(pos.method_10074(), pos.method_10084(), pos.method_10095(), pos.method_10078(), pos.method_10072(), pos.method_10067());
    }
    
    private static class_2350 getDirectionFromOffset(class_2338 self, class_2338 target) {
        var offset = target.method_10059(self);
        return class_2350.method_50026(offset.method_10263(), offset.method_10264(), offset.method_10260());
    }
    
    public static void receiveVisualItemsPacket(RenderStackData message, class_1937 world, class_5455 registryAccess) {
        var blockEntity = world.method_35230(message.self, BlockEntitiesContent.ITEM_PIPE_ENTITY);
        if (blockEntity.isPresent()) {
            var pipeEntity = blockEntity.get();
            // use local time for moved item to avoid rendering issues caused by lag
            pipeEntity.activeStacks.add(new RenderStackData(pipeEntity.field_11867, message.rendered, message.path, world.method_8510(), message.pathLength));
        }
    }
    
    @Override
    public void method_5431() {
        if (this.field_11863 != null)
            field_11863.method_8524(field_11867);
    }
    
    public record RenderStackData(class_2338 self, class_1799 rendered, List<class_2338> path, Long startedAt, int pathLength) implements class_8710 {
        
        public static final class_8710.class_9154<RenderStackData> PIPE_ITEMS_ID = new class_8710.class_9154<>(Oritech.id("pipe_items"));
        
        @Override
        public class_9154<? extends class_8710> method_56479() {
            return PIPE_ITEMS_ID;
        }
    }
}
