package rearth.oritech.block.entity.accelerator;

import org.jetbrains.annotations.Nullable;
import oshi.util.tuples.Pair;
import rearth.oritech.Oritech;
import rearth.oritech.block.blocks.accelerator.AcceleratorPassthroughBlock;
import rearth.oritech.block.blocks.accelerator.AcceleratorRingBlock;
import rearth.oritech.init.BlockContent;
import rearth.oritech.util.Geometry;

import java.util.*;
import net.minecraft.class_1309;
import net.minecraft.class_2338;
import net.minecraft.class_238;
import net.minecraft.class_2382;
import net.minecraft.class_243;
import net.minecraft.class_2741;
import net.minecraft.class_3218;
import net.minecraft.class_3545;

// move this into a second class to keep the entity class smaller and focus on recipe handling, work interaction, etc.
public class AcceleratorParticleLogic {
    private final class_2338 pos;
    private final class_3218 world;
    private final AcceleratorControllerBlockEntity entity;
    
    private static final Map<CompPair<class_2338, class_2382>, class_2338> cachedGates = new HashMap<>();    // stores the next gate for a combo of source gate and direction
    private static final Map<class_2338, class_2338> activeParticles = new HashMap<>(); // stores relations between position of particle -> position of controller
    
    public AcceleratorParticleLogic(class_2338 pos, class_3218 world, AcceleratorControllerBlockEntity entity) {
        this.pos = pos;
        this.world = world;
        this.entity = entity;
    }
    
    
    @SuppressWarnings("lossy-conversions")
    public void update(ActiveParticle particle) {
        
        var timePassed = 1 / 20f;
        
        var renderedTrail = new ArrayList<class_243>();
        renderedTrail.add(particle.position);
        
        // list of positions this frame checked for entities
        var checkedPositions = new HashSet<class_2338>();
        
        var foundCollisions = new ArrayList<class_2338>();
        Pair<ActiveParticle, AcceleratorControllerBlockEntity> collidedWith = null;
        
        var availableDistance = particle.velocity * timePassed;
        while (availableDistance > 0.001) {
            
            if (particle.nextGate == null) {
                exitParticle(particle, new class_243(0, 0, 0), AcceleratorControllerBlockEntity.ParticleEvent.ERROR);
                return;
            }
            
            var path = particle.nextGate.method_46558().method_1020(particle.position);
            var pathLength = path.method_1033();
            var moveDist = Math.min(pathLength, availableDistance);
            availableDistance -= moveDist;
            var movedBy = path.method_1029().method_1021(moveDist);
            
            // check if position intersects with another particle
            if (particle.lastGate != null) {
                var candidate = updateParticleCollision(particle.lastGate, foundCollisions);
                if (candidate.isPresent())
                    collidedWith = candidate.get();
            }
            var candidate = updateParticleCollision(particle.nextGate, foundCollisions);
            if (candidate.isPresent())
                collidedWith = candidate.get();
            
            // update position
            particle.position = particle.position.method_1019(movedBy);
            
            renderedTrail.add(particle.position);
            particle.lastBendDistance += moveDist;
            
            checkParticleEntityCollision(particle.position, particle, checkedPositions);
            
            if (moveDist >= pathLength - 0.1f) {
                // gate reached
                // calculate next gate direction
                var reachedGate = particle.nextGate;
                var nextDirection = getGateExitDirection(particle.lastGate, particle.nextGate);
                // try find next valid gate
                var nextGate = findNextGateCached(reachedGate, nextDirection, particle.velocity);
                
                // no gate built / too slow
                if (nextGate == null) {
                    if (collidedWith != null) {
                        calculateCollision(particle, foundCollisions, collidedWith);
                    } else {
                        exitParticle(particle, class_243.method_24954(nextDirection), AcceleratorControllerBlockEntity.ParticleEvent.EXITED_NO_GATE);
                    }
                    return;
                }
                
                // check if curve is too strong (based on reached gate)
                var gateOffset = particle.nextGate.method_10059(particle.lastGate);
                var lastDirection = new class_2382(Math.clamp(gateOffset.method_10263(), -1, 1), 0, Math.clamp(gateOffset.method_10260(), -1, 1));
                var wasBend = !lastDirection.equals(nextDirection);
                if (wasBend) {
                    
                    var combinedDist = getParticleBendDist(particle.lastBendDistance, particle.lastBendDistance2);
                    var requiredDist = getRequiredBendDist(particle.velocity);
                    
                    if (combinedDist <= requiredDist) {
                        if (collidedWith != null) {
                            calculateCollision(particle, foundCollisions, collidedWith);
                        } else {
                            exitParticle(particle, class_243.method_24954(particle.nextGate.method_10059(particle.lastGate)), AcceleratorControllerBlockEntity.ParticleEvent.EXITED_FAST);
                        }
                        return;
                    }
                    
                    particle.lastBendDistance2 = particle.lastBendDistance;
                    particle.lastBendDistance = 0;
                }
                
                // handle gate interaction (e.g. motor or sensor)
                var gateBlock = world.method_8320(reachedGate).method_26204();
                if (gateBlock.equals(BlockContent.ACCELERATOR_MOTOR)) {
                    entity.handleParticleMotorInteraction(reachedGate);
                } else if (gateBlock.equals(BlockContent.ACCELERATOR_SENSOR) && world.method_8321(reachedGate) instanceof AcceleratorSensorBlockEntity sensorEntity) {
                    sensorEntity.measureParticle(particle);
                }
                
                particle.nextGate = nextGate;
                particle.lastGate = reachedGate;
            }
        }
        
        if (collidedWith != null) {
            calculateCollision(particle, foundCollisions, collidedWith);
            return;
        }
        
        entity.onParticleMoved(renderedTrail);
    }
    
    private void calculateCollision(ActiveParticle particle, ArrayList<class_2338> foundCollisions, Pair<ActiveParticle, AcceleratorControllerBlockEntity> collidedWith) {
        var even = foundCollisions.size() % 2 == 0;
        class_243 collisionPoint;
        if (!even) {
            // get middle position
            collisionPoint = foundCollisions.get(foundCollisions.size() / 2).method_46558();
        } else {
            var a = foundCollisions.get(foundCollisions.size() / 2 - 1).method_46558();
            var b = foundCollisions.get(foundCollisions.size() / 2).method_46558();
            collisionPoint = a.method_1019(b).method_18805(0.5f, 0.5f, 0.5f);
        }
        
        var impactSpeed = particle.velocity + collidedWith.getA().velocity;
        entity.onParticleCollided(impactSpeed, collisionPoint, collidedWith.getB());
    }
    
    private void checkParticleEntityCollision(class_243 position, ActiveParticle particle, Set<class_2338> alreadyChecked) {
        
        var blockPos = class_2338.method_49638(position);
        if (alreadyChecked.contains(blockPos)) return;
        alreadyChecked.add(blockPos);
        
        var targets = world.method_8390(class_1309.class, new class_238(blockPos), elem -> elem.method_5805() && elem.method_5732() && !elem.method_7325());
        var remainingMomentum = particle.velocity;
        for (var mob : targets) {
            var usedMomentum = entity.handleParticleEntityCollision(blockPos, particle, remainingMomentum, mob);
            remainingMomentum -= usedMomentum;
            
            if (remainingMomentum <= 0.1f) return;
        }
        
        particle.velocity = remainingMomentum;
    }
    
    private void exitParticle(ActiveParticle particle, class_243 direction, AcceleratorControllerBlockEntity.ParticleEvent reason) {
        
        var exitFrom = particle.position;
        
        var distance = Math.max(Math.sqrt(particle.velocity), 0.4) * 0.9;
        var exitTo = exitFrom.method_1019(direction.method_1029().method_1021(distance));
        
        entity.onParticleExited(exitFrom, exitTo, particle.lastGate, direction, reason);
        
        var searchDist = (int) distance;
        var searchDirection = new class_2382((int) Math.round(direction.field_1352), 0, (int) Math.round(direction.field_1350));
        var searchStart = particle.nextGate;
        if (searchStart == null) searchStart = particle.lastGate;
        
        var remainingMomentum = particle.velocity;
        
        for (int i = 1; i <= searchDist; i++) {
            var checkPos = searchStart.method_10081(searchDirection.method_35862(i));
            
            var targets = world.method_8390(class_1309.class, new class_238(checkPos), elem -> elem.method_5805() && elem.method_5732() && !elem.method_7325());
            
            for (var mob : targets) {
                var usedMomentum = entity.handleParticleEntityCollision(checkPos, particle, remainingMomentum, mob);
                remainingMomentum -= usedMomentum;
                
                if (remainingMomentum <= 0.1f) return;
            }
            
            var block = world.method_8320(checkPos);
            var targetableBlock = !block.method_26215() && !(block.method_26204() instanceof AcceleratorPassthroughBlock);
            if (targetableBlock) {
                var usedMomentum = entity.handleParticleBlockCollision(checkPos, particle, remainingMomentum, block);
                remainingMomentum -= usedMomentum;
                
                if (remainingMomentum <= 0.1f) return;
            }
            
        }
        
    }
    
    private Optional<Pair<ActiveParticle, AcceleratorControllerBlockEntity>> updateParticleCollision(class_2338 blockPos, List<class_2338> collisionPositions) {
        
        var duplicate = !collisionPositions.isEmpty() && collisionPositions.getLast().equals(blockPos);
        
        if (activeParticles.containsKey(blockPos) && !activeParticles.get(blockPos).equals(this.pos) && !duplicate) {
            // found collision
            var secondControllerPos = activeParticles.get(blockPos);
            
            if (!(world.method_8321(secondControllerPos) instanceof AcceleratorControllerBlockEntity secondAccelerator))
                return Optional.empty();
            
            var secondParticle = secondAccelerator.getParticle();
            if (secondParticle == null)
                return Optional.empty();
            
            // entity.onParticleCollided(impactSpeed, particle.position, secondControllerPos, secondAccelerator);
            collisionPositions.add(blockPos);
            return Optional.of(new Pair<>(secondParticle, secondAccelerator));
        }
        
        activeParticles.put(blockPos, this.pos);
        return Optional.empty();
        
    }
    
    // this assumes the next gate is a valid target for a particle coming from lastGate.
    // Returns a neighboring or diagonal direction
    private class_2382 getGateExitDirection(class_2338 lastGate, class_2338 nextGate) {
        
        var incomingPath = nextGate.method_10059(lastGate);
        var incomingStraight = incomingPath.method_10263() == 0 || incomingPath.method_10260() == 0;
        var incomingDir = new class_2382(Math.clamp(incomingPath.method_10263(), -1, 1), 0, Math.clamp(incomingPath.method_10260(), -1, 1));
        
        var targetState = world.method_8320(nextGate);
        var targetBlock = targetState.method_26204();
        
        // go straight through motors and sensors
        if (targetBlock.equals(BlockContent.ACCELERATOR_MOTOR) || targetBlock.equals(BlockContent.ACCELERATOR_SENSOR))
            return incomingDir;
        
        // if the target gate has just been destroyed
        if (!targetBlock.equals(BlockContent.ACCELERATOR_RING)) return incomingDir;
        
        var targetFacing = targetState.method_11654(class_2741.field_12481);
        var targetBent = targetState.method_11654(AcceleratorRingBlock.BENT);
        var targetRedstone = targetState.method_11654(AcceleratorRingBlock.REDSTONE_STATE);
        
        // if we come straight, the exit can be either curved or bent
        // if we come bent, the exit has to be straight
        
        // if we come in straight, redstone is 0, and we come in from the front (straight), we exit straight (weird edge case) (e.g. we arrive at the entrance enabled with redstone)
        if (targetRedstone == 0 && incomingStraight && Geometry.getBackward(targetFacing).equals(incomingDir)) {
            return Geometry.getBackward(targetFacing);
        }
        
        if (!incomingStraight) {
            // if we come in from the bent side, we always exit at the back of nextGate
            return Geometry.getBackward(targetFacing);
        } else {
            // if we come in straight, we either exit straight or bent
            if (targetBent == 0) {  // straight, keep direction. We don't know whether we enter from forward or behind
                return incomingDir;
            } else if (targetBent == 1) {   // bent left
                return Geometry.getForward(targetFacing).method_35853(Geometry.getLeft(targetFacing));
            } else {   // bent right
                return Geometry.getForward(targetFacing).method_35853(Geometry.getRight(targetFacing));
            }
        }
        
    }
    
    public static float getMaxGateDist(float speed) {
        return (float) Math.clamp(Math.sqrt(speed) / 2, 2, Oritech.CONFIG.maxGateDist());
    }
    
    public static float getRequiredBendDist(float speed) {
        return (float) (Math.sqrt(speed) / Oritech.CONFIG.bendFactor());
    }
    
    public static float getParticleBendDist(float distA, float distB) {
        return distA + distB;
    }
    
    @Nullable
    private class_2338 findNextGateCached(class_2338 from, class_2382 direction, float speed) {
        
        var maxDist = getMaxGateDist(speed);
        var key = new CompPair<>(from, direction);
        
        if (cachedGates.containsKey(key)) {
            var result = cachedGates.get(key);
            var dist = (int) result.method_46558().method_1022(from.method_46558());
            if (dist <= maxDist) return result;
        }
        
        var candidate = findNextGate(from, direction, speed);
        if (candidate != null) {
            cachedGates.put(key, candidate);
        }
        
        return candidate;
        
    }
    
    // tries to find the next gate candidate, based on the starting gate
    // direction can be either straight or diagonal
    @Nullable
    public class_2338 findNextGate(class_2338 from, class_2382 direction, float speed) {
        
        // longer empty areas only work at higher speeds
        var maxDist = getMaxGateDist(speed);
        
        for (int i = 1; i <= maxDist; i++) {
            var candidatePos = from.method_10081(direction.method_35862(i));
            var candidateState = world.method_8320(candidatePos);
            if (candidateState.method_26215()) continue;
            
            if (candidateState.method_26204().equals(BlockContent.ACCELERATOR_MOTOR) || candidateState.method_26204().equals(BlockContent.ACCELERATOR_SENSOR))
                return candidatePos;
            
            if (!candidateState.method_26204().equals(BlockContent.ACCELERATOR_RING)) return null;
            
            // check if ring is facing source pos (from)
            var candidateBent = candidateState.method_11654(AcceleratorRingBlock.BENT);
            var candidateFacing = candidateState.method_11654(class_2741.field_12481);
            var candidateRedstone = candidateState.method_11654(AcceleratorRingBlock.REDSTONE_STATE);
            
            var candidateBack = candidatePos.method_10081(Geometry.getBackward(candidateFacing).method_35862(i));
            var candidateFront = candidatePos.method_10081(Geometry.getForward(candidateFacing).method_35862(i));
            
            // front can be bent
            if (candidateBent == 1)
                candidateFront = candidateFront.method_10081(Geometry.getLeft(candidateFacing).method_35862(i));
            if (candidateBent == 2)
                candidateFront = candidateFront.method_10081(Geometry.getRight(candidateFacing).method_35862(i));
            
            var isValid = candidateBack.equals(from) || candidateFront.equals(from);
            
            // check if redstone input is valid
            if (!isValid && candidateRedstone != 3) {
                candidateFront = candidatePos.method_10081(Geometry.getForward(candidateFacing).method_35862(i));    // reset front
                if (candidateRedstone == 1) {
                    candidateFront = candidateFront.method_10081(Geometry.getLeft(candidateFacing).method_35862(i));
                } else if (candidateRedstone == 2) {
                    candidateFront = candidateFront.method_10081(Geometry.getRight(candidateFacing).method_35862(i));
                }
                
                isValid = candidateFront.equals(from);
            }
            
            if (isValid) return candidatePos;
            
        }
        
        return null;
        
    }
    
    // called on server tick end. Used for collision detection
    public static void onTickEnd() {
        activeParticles.clear();
    }
    
    // remove caches that have either source or target as pos. Called from gate blocks
    public static void resetCachedGate(class_2338 pos) {
        var toRemove = cachedGates.entrySet().stream().filter(elem -> elem.getKey().method_15442().equals(pos) || elem.getValue().equals(pos)).map(Map.Entry::getKey).toList();
        toRemove.forEach(cachedGates::remove);
    }
    
    public static void resetNearbyCache(class_2338 pos) {
        var toRemove = cachedGates.keySet().stream().filter(blockPos -> blockPos.method_15442().method_19455(pos) < Oritech.CONFIG.maxGateDist() + 1).toList();
        toRemove.forEach(cachedGates::remove);
    }
    
    public static final class CompPair<A, B> extends class_3545<A, B> {
        
        public CompPair(A left, B right) {
            super(left, right);
        }
        
        @Override
        public int hashCode() {
            return (method_15442() == null ? 0 : method_15442().hashCode()) ^ (method_15441() == null ? 0 : method_15441().hashCode());
        }
        
        @Override
        public boolean equals(Object o) {
            if (!(o instanceof CompPair<?, ?> p)) {
                return false;
            }
            
            return Objects.equals(p.method_15442(), method_15442()) && Objects.equals(p.method_15441(), method_15441());
        }
    }
    
    public static final class ActiveParticle {
        public class_243 position;
        public float velocity;
        public class_2338 nextGate;
        public class_2338 lastGate;
        public float lastBendDistance = 15000;
        public float lastBendDistance2 = 15000;
        
        public ActiveParticle(class_243 position, float velocity, class_2338 nextGate, class_2338 lastGate) {
            this.position = position;
            this.velocity = velocity;
            this.nextGate = nextGate;
            this.lastGate = lastGate;
        }
    }
    
}
