package rearth.oritech.block.base.entity;

import dev.architectury.registry.menu.ExtendedMenuProvider;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.EnergyApi;
import rearth.oritech.api.energy.containers.DynamicEnergyStorage;
import rearth.oritech.api.item.ItemApi;
import rearth.oritech.api.item.containers.DelegatingInventoryStorage;
import rearth.oritech.api.item.containers.InOutInventoryStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.entity.addons.RedstoneAddonBlockEntity;
import rearth.oritech.client.ui.BasicMachineScreenHandler;
import rearth.oritech.init.recipes.OritechRecipe;
import rearth.oritech.init.recipes.OritechRecipeType;
import rearth.oritech.util.*;
import software.bernie.geckolib.animatable.GeoBlockEntity;
import software.bernie.geckolib.animatable.SingletonGeoAnimatable;
import software.bernie.geckolib.animatable.instance.AnimatableInstanceCache;
import software.bernie.geckolib.animation.*;
import software.bernie.geckolib.util.GeckoLibUtil;

import java.util.*;
import net.minecraft.class_1262;
import net.minecraft.class_1263;
import net.minecraft.class_1277;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_5455;
import net.minecraft.class_7225;
import net.minecraft.class_8710;
import net.minecraft.class_8786;
import net.minecraft.class_9695;

public abstract class MachineBlockEntity extends NetworkedBlockEntity
  implements ExtendedMenuProvider, GeoBlockEntity, EnergyApi.BlockProvider, ScreenProvider, ItemApi.BlockProvider, RedstoneAddonBlockEntity.RedstoneControllable {
    
    // animations
    public static final RawAnimation PACKAGED = RawAnimation.begin().thenPlayAndHold("packaged");
    public static final RawAnimation SETUP = RawAnimation.begin().thenPlay("deploy");
    public static final RawAnimation IDLE = RawAnimation.begin().thenPlayAndHold("idle");
    public static final RawAnimation WORKING = RawAnimation.begin().thenPlay("working");
    
    protected final AnimatableInstanceCache animatableInstanceCache = GeckoLibUtil.createInstanceCache(this);
    
    // synced data
    @SyncField({SyncType.GUI_TICK, SyncType.SPARSE_TICK})
    public int progress;
    @SyncField({SyncType.GUI_TICK})
    protected OritechRecipe currentRecipe = OritechRecipe.DUMMY;
    @SyncField({SyncType.GUI_TICK})
    protected InventoryInputMode inventoryInputMode = InventoryInputMode.FILL_LEFT_TO_RIGHT;
    @SyncField({SyncType.GUI_TICK})
    protected boolean disabledViaRedstone = false;
    @SyncField({SyncType.TICK})
    public long lastWorkedAt;
    
    // static data
    protected int energyPerTick;
    
    // own storages
    public final FilteringInventory inventory = new FilteringInventory(getInventorySize(), this::method_5431, getSlotAssignments());
    private final Map<class_2350, ItemApi.InventoryStorage> sidedInventories = new HashMap<>(); // only for sided input mode
    @SyncField({SyncType.GUI_TICK, SyncType.GUI_OPEN})
    public final DynamicEnergyStorage energyStorage = new DynamicEnergyStorage(getDefaultCapacity(), getDefaultInsertRate(), getDefaultExtractionRate(), this::method_5431, this.canEnergyStorageChangeWhileGUIOpen());
    
    public MachineBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state, int energyPerTick) {
        super(type, pos, state);
        this.energyPerTick = energyPerTick;
        
        if (field_11863 != null)
            lastWorkedAt = field_11863.method_8510();
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        
        if (!isActive(state) || disabledViaRedstone) return;
        
        // if a recipe is found, this means the input items are all available
        var recipeCandidate = getRecipe();
        if (recipeCandidate.isEmpty())
            currentRecipe = OritechRecipe.DUMMY;     // reset recipe when invalid or no input is given
        
        if (recipeCandidate.isPresent() && canOutputRecipe(recipeCandidate.get().comp_1933()) && canProceed(recipeCandidate.get().comp_1933())) {
            
            // reset when recipe was switched while running
            if (currentRecipe != recipeCandidate.get().comp_1933()) resetProgress();
            
            // this is separate so that progress is not reset when out of energy
            if (hasEnoughEnergy()) {
                var activeRecipe = recipeCandidate.get().comp_1933();
                currentRecipe = activeRecipe;
                lastWorkedAt = world.method_8510();
                
                useEnergy();
                
                // increase progress
                progress++;
                
                if (checkCraftingFinished(activeRecipe)) {
                    craftItem(activeRecipe, getOutputView(), getInputView());
                    resetProgress();
                }
                
                method_5431();
            }
            
        } else {
            // this happens if either the input slot is empty, or the output slot is blocked
            if (progress > 0) resetProgress();
        }
    }
    
    // used to do additional checks, if the recipe match is not enough
    protected boolean canProceed(OritechRecipe value) {
        return true;
    }
    
    protected boolean hasEnoughEnergy() {
        return energyStorage.amount >= calculateEnergyUsage();
    }
    
    @SuppressWarnings("lossy-conversions")
    protected void useEnergy() {
        energyStorage.amount -= calculateEnergyUsage();
    }
    
    protected float calculateEnergyUsage() {
        return energyPerTick * getEfficiencyMultiplier() * (1 / getSpeedMultiplier());
    }
    
    public List<class_1799> getCraftingResults(OritechRecipe activeRecipe) {
        return activeRecipe.getResults();
    }
    
    protected void craftItem(OritechRecipe activeRecipe, List<class_1799> outputInventory, List<class_1799> inputInventory) {
        
        var results = getCraftingResults(activeRecipe);
        var inputs = activeRecipe.getInputs();
        
        // create outputs
        for (int i = 0; i < results.size(); i++) {
            var result = results.get(i);
            var slot = outputInventory.get(i);
            
            var newCount = slot.method_7947() + result.method_7947();
            if (slot.method_7960()) {
                outputInventory.set(i, result.method_7972());
            } else {
                slot.method_7939(newCount);
            }
        }
        
        // remove inputs. Each input is 1 ingredient.
        var startOffset = 0;    // used so when multiple matching stacks are available, they're drained somewhat evenly
        for (var removedIng : inputs) {
            // try to find current ingredient
            for (int i = 0; i < inputInventory.size(); i++) {
                var inputStack = inputInventory.get((i + startOffset) % inputInventory.size());
                if (removedIng.method_8093(inputStack)) {
                    inputStack.method_7934(1);
                    startOffset++;
                    break;
                }
            }
            
            
        }
        
    }
    
    protected boolean checkCraftingFinished(OritechRecipe activeRecipe) {
        return progress >= activeRecipe.getTime() * getSpeedMultiplier();
    }
    
    protected void resetProgress() {
        progress = 0;
    }
    
    // check if output slots are valid, meaning: each slot is either empty, or of the same type and can add the target amount without overfilling
    public boolean canOutputRecipe(OritechRecipe recipe) {
        
        var outInv = getOutputInventory();
        
        if (outInv.method_5442()) return true;
        
        List<class_1799> results = recipe.getResults();
        for (int i = 0; i < results.size(); i++) {
            var result = results.get(i);
            var outSlot = outInv.method_5438(i);
            
            if (outSlot.method_7960()) continue;
            
            if (!canAddToSlot(result, outSlot)) return false;
            
        }
        
        return true;
    }
    
    protected boolean canAddToSlot(class_1799 input, class_1799 slot) {
        if (slot.method_7960()) return true;
        if (!slot.method_7909().equals(input.method_7909())) return false;  // type mismatch
        return slot.method_7947() + input.method_7947() <= slot.method_7914();  // count too high
    }
    
    protected Optional<class_8786<OritechRecipe>> getRecipe() {
        
        // check if old recipe fits
        if (currentRecipe != null && currentRecipe != OritechRecipe.DUMMY) {
            if (currentRecipe.method_8115(getInputInventory(), field_11863)) return Optional.of(new class_8786<>(currentRecipe.getOriType().getIdentifier(), currentRecipe));
        }
        
        return field_11863.method_8433().method_8132(getOwnRecipeType(), getInputInventory(), field_11863);
    }
    
    protected abstract OritechRecipeType getOwnRecipeType();
    
    public abstract InventorySlotAssignment getSlotAssignments();
    
    protected List<class_1799> getInputView() {
        var slots = getSlotAssignments();
        return this.inventory.heldStacks.subList(slots.inputStart(), slots.inputStart() + slots.inputCount());
    }
    
    protected List<class_1799> getOutputView() {
        var slots = getSlotAssignments();
        return this.inventory.heldStacks.subList(slots.outputStart(), slots.outputStart() + slots.outputCount());
    }
    
    protected class_9695 getInputInventory() {
        return new SimpleCraftingInventory(getInputView().toArray(class_1799[]::new));
    }
    
    protected class_1263 getOutputInventory() {
        return new class_1277(getOutputView().toArray(class_1799[]::new));
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        
        class_1262.method_5427(nbt, inventory.heldStacks, false, registryLookup);
        nbt.method_10569("oritech.machine_progress", progress);
        nbt.method_10544("oritech.machine_energy", energyStorage.amount);
        nbt.method_10575("oritech.machine_input_mode", (short) inventoryInputMode.ordinal());
        nbt.method_10556("oritech.redstone", disabledViaRedstone);
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        class_1262.method_5429(nbt, inventory.heldStacks, registryLookup);
        progress = nbt.method_10550("oritech.machine_progress");
        energyStorage.amount = nbt.method_10537("oritech.machine_energy");
        inventoryInputMode = InventoryInputMode.values()[nbt.method_10568("oritech.machine_input_mode")];
        disabledViaRedstone = nbt.method_10577("oritech.redstone");
    }
    
    private int findLowestMatchingSlot(class_1799 stack, List<class_1799> inv, boolean allowEmpty) {
        
        var lowestMatchingIndex = -1;
        var lowestMatchingCount = 64;
        
        for (int i = 0; i < inv.size(); i++) {
            var invSlot = inv.get(i);
            
            // if a slot is empty, is it automatically the lowest
            if (invSlot.method_7960() && allowEmpty) return i;
            
            if (invSlot.method_7909().equals(stack.method_7909()) && invSlot.method_7947() < lowestMatchingCount) {
                lowestMatchingIndex = i;
                lowestMatchingCount = invSlot.method_7947();
            }
        }
        
        return lowestMatchingIndex;
    }
    
    @Override
    public void registerControllers(AnimatableManager.ControllerRegistrar controllers) {
        controllers.add(new AnimationController<>(this, this::onAnimationUpdate)
                          .triggerableAnim("setup", SETUP)
                          .setAnimationSpeedHandler(animatable -> (double) getAnimationSpeed())
                          .setSoundKeyframeHandler(new AutoPlayingSoundKeyframeHandler<>(this::getAnimationSpeed)));
    }
    
    public PlayState onAnimationUpdate(final AnimationState<MachineBlockEntity> state) {
        
        if (state.getController().isPlayingTriggeredAnimation()) return PlayState.CONTINUE;
        
        if (isActive(method_11010())) {
            if (isActivelyWorking()) {
                return state.setAndContinue(WORKING);
            } else {
                return state.setAndContinue(IDLE);
            }
        }
        
        return state.setAndContinue(PACKAGED);
    }
    
    public boolean isActivelyWorking() {
        return field_11863.method_8510() - lastWorkedAt < 15;
    }
    
    protected float getAnimationSpeed() {
        if (getRecipeDuration() < 0) return 1;
        var recipeTicks = getRecipeDuration() * getSpeedMultiplier();
        return (getAnimationDuration() / recipeTicks) * 0.99f;
    }
    
    public int getAnimationDuration() {
        return 60;  // 3s
    }
    
    protected int getRecipeDuration() {
        return getCurrentRecipe().getTime();
    }
    
    @Override
    public AnimatableInstanceCache getAnimatableInstanceCache() {
        return animatableInstanceCache;
    }
    
    @Override
    public void saveExtraData(class_2540 buf) {
        this.sendUpdate(SyncType.GUI_OPEN);
        buf.method_10807(field_11867);
        
    }
    
    protected class_2350 getFacing() {
        return Objects.requireNonNull(field_11863).method_8320(method_11016()).method_11654(class_2741.field_12481);
    }
    
    @Override
    public class_2561 method_5476() {
        return class_2561.method_43470("");
    }
    
    @Nullable
    @Override
    public class_1703 createMenu(int syncId, class_1661 playerInventory, class_1657 player) {
        return new BasicMachineScreenHandler(syncId, playerInventory, this);
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorage(class_2350 direction) {
        return energyStorage;
    }
    
    @Override
    public abstract List<GuiSlot> getGuiSlots();
    
    @Override
    public float getProgress() {
        return (float) progress / (currentRecipe.getTime() * getSpeedMultiplier());
    }
    
    public void setProgress(int progress) {
        this.progress = progress;
    }
    
    public DynamicEnergyStorage getEnergyStorage() {
        return energyStorage;
    }
    
    public OritechRecipe getCurrentRecipe() {
        return currentRecipe;
    }
    
    public void setCurrentRecipe(OritechRecipe currentRecipe) {
        this.currentRecipe = currentRecipe;
    }
    
    // lower = better for both (speed and efficiency)
    public float getSpeedMultiplier() {
        return 1;
    }
    
    public float getEfficiencyMultiplier() {
        return 1;
    }
    
    public void cycleInputMode() {
        switch (inventoryInputMode) {
            case FILL_LEFT_TO_RIGHT:
                inventoryInputMode = InventoryInputMode.FILL_EVENLY;
                break;
            case FILL_EVENLY:
                inventoryInputMode = InventoryInputMode.SIDED;
                break;
            case SIDED:
                inventoryInputMode = InventoryInputMode.FILL_LEFT_TO_RIGHT;
                break;
        }
        
        method_5431();
    }
    
    @Override
    public InventoryInputMode getInventoryInputMode() {
        return inventoryInputMode;
    }
    
    public abstract int getInventorySize();
    
    public boolean isActive(class_2680 state) {
        return true;
    }
    
    public void setEnergyStored(long amount) {
        energyStorage.amount = amount;
    }
    
    @Override
    public float getDisplayedEnergyUsage() {
        return calculateEnergyUsage();
    }
    
    public long getDefaultCapacity() {
        return 5000;
    }
    
    public long getDefaultInsertRate() {
        return 1024;
    }
    
    @Override
    public float getDisplayedEnergyTransfer() {
        return energyStorage.maxInsert;
    }
    
    public long getDefaultExtractionRate() {
        return 0;
    }
    
    public int getEnergyPerTick() {
        return energyPerTick;
    }
    
    @Override
    public class_1263 getDisplayedInventory() {
        return inventory;
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryStorage(class_2350 direction) {
        if (inventoryInputMode.equals(InventoryInputMode.SIDED)) {
            return sidedInventories.computeIfAbsent(direction, this::getDirectedStorage);
        }
        return inventory;
    }
    
    @Override
    public int getComparatorEnergyAmount() {
        return (int) ((energyStorage.amount / (float) energyStorage.capacity) * 15);
    }
    
    @Override
    public int getComparatorSlotAmount(int slot) {
        if (inventory.heldStacks.size() <= slot) return 0;
        
        var stack = inventory.method_5438(slot);
        if (stack.method_7960()) return 0;
        
        return (int) ((stack.method_7947() / (float) stack.method_7914()) * 15);
    }
    
    @Override
    public int getComparatorProgress() {
        if (currentRecipe.getTime() <= 0) return 0;
        return (int) ((progress / (float) currentRecipe.getTime() * getSpeedMultiplier()) * 15);
    }
    
    @Override
    public int getComparatorActiveState() {
        return isActivelyWorking() ? 15 : 0;
    }
    
    @Override
    public void onRedstoneEvent(boolean isPowered) {
        this.disabledViaRedstone = isPowered;
    }
    
    // whether the energy storage should only send the current amount on network updates, or the full data
    public boolean canEnergyStorageChangeWhileGUIOpen() {
        return false;
    }
    
    public static void receiveCycleModePacket(InventoryInputModeSelectorPacket packet, class_1657 player, class_5455 dynamicRegistryManager) {
        if (player.method_37908().method_8321(packet.position()) instanceof MachineBlockEntity machineBlock)
            machineBlock.cycleInputMode();
    }
    
    public ItemApi.InventoryStorage getDirectedStorage(class_2350 direction) {
        
        var slots = getSlotAssignments();
        if (slots.inputCount() <= 1) return inventory;
        
        if (direction == null) return inventory;
        
        // input only, disable output
        if (direction.equals(class_2350.field_11036)) {
            return new DelegatingInventoryStorage(inventory, () -> true) {
                @Override
                public int extract(class_1799 extracted, boolean simulate) {
                    return 0;
                }
                
                @Override
                public int extractFromSlot(class_1799 extracted, int slot, boolean simulate) {
                    return 0;
                }
                
                @Override
                public boolean supportsExtraction() {
                    return false;
                }
            };
        } else if (direction.equals(class_2350.field_11033)) {
            return new DelegatingInventoryStorage(inventory, () -> true) {
                @Override
                public int insert(class_1799 inserted, boolean simulate) {
                    return 0;
                }
                
                @Override
                public int insertToSlot(class_1799 inserted, int slot, boolean simulate) {
                    return 0;
                }
                
                @Override
                public boolean supportsInsertion() {
                    return false;
                }
            };
        } else {
            // north = 0, east = 1, ...
            var horizontalOrdinal = 0;
            if (direction.equals(class_2350.field_11034)) horizontalOrdinal = 1;
            if (direction.equals(class_2350.field_11035)) horizontalOrdinal = 2;
            if (direction.equals(class_2350.field_11039)) horizontalOrdinal = 3;
            var inputSlotIndex = slots.inputStart() + horizontalOrdinal % slots.inputCount();
            
            return new DelegatingInventoryStorage(inventory, () -> true) {
                @Override
                public int insertToSlot(class_1799 inserted, int slot, boolean simulate) {
                    if (slot != inputSlotIndex) return 0;
                    return super.insertToSlot(inserted, slot, simulate);
                }
                
                @Override
                public int insert(class_1799 inserted, boolean simulate) {
                    return insertToSlot(inserted, inputSlotIndex, simulate);
                }
            };
        }
    }
    
    public class FilteringInventory extends InOutInventoryStorage {
        
        public FilteringInventory(int size, Runnable onUpdate, InventorySlotAssignment slotAssignment) {
            super(size, onUpdate, slotAssignment);
        }
        
        @Override
        public int insert(class_1799 toInsert, boolean simulate) {
            
            if (inventoryInputMode.equals(InventoryInputMode.FILL_EVENLY)) {
                var remaining = toInsert.method_7947();
                var slotCountTarget = toInsert.method_7947() / getSlotAssignments().inputCount();
                slotCountTarget = Math.clamp(slotCountTarget, 1, remaining);
                
                // start at slot with fewest items
                var lowestSlot = 0;
                var lowestSlotCount = Integer.MAX_VALUE;
                for (int i = getSlotAssignments().inputStart(); i < getSlotAssignments().inputStart() + getSlotAssignments().inputCount(); i++) {
                    var content = this.method_5438(i);
                    if (!content.method_7960() && !content.method_7909().equals(toInsert.method_7909()))
                        continue;    // skip slots containing other items
                    if (content.method_7947() < lowestSlotCount) {
                        lowestSlotCount = content.method_7947();
                        lowestSlot = i;
                    }
                }
                
                for (var slot = 0; slot < method_5439() && remaining > 0; slot++) {
                    remaining -= customSlotInsert(toInsert.method_46651(slotCountTarget), (slot + lowestSlot) % method_5439(), simulate);
                }
                
                return toInsert.method_7947() - remaining;
            }
            
            
            return super.insert(toInsert, simulate);
        }
        
        @Override
        public int insertToSlot(class_1799 addedStack, int slot, boolean simulate) {
            
            if (inventoryInputMode.equals(InventoryInputMode.FILL_EVENLY)) {
                return insert(addedStack, simulate);
            }
            
            return customSlotInsert(addedStack, slot, simulate);
        }
        
        private int customSlotInsert(class_1799 toInsert, int slot, boolean simulate) {
            return super.insertToSlot(toInsert, slot, simulate);
        }
    }
    
    // Client -> Server (e.g. from UI interactions
    public record InventoryInputModeSelectorPacket(class_2338 position) implements class_8710 {
        
        public static final class_8710.class_9154<InventoryInputModeSelectorPacket> PACKET_ID = new class_8710.class_9154<>(Oritech.id("input_mode"));
        
        @Override
        public class_9154<? extends class_8710> method_56479() {
            return PACKET_ID;
        }
    }
}
