package rearth.oritech.util;

import com.mojang.datafixers.util.Either;
import dev.architectury.fluid.FluidStack;
import io.netty.buffer.ByteBuf;
import io.wispforest.endec.Endec;
import io.wispforest.endec.impl.StructEndecBuilder;
import io.wispforest.owo.serialization.CodecUtils;
import io.wispforest.owo.serialization.endec.MinecraftEndecs;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_5321;
import net.minecraft.class_6862;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;

// Inspired by Immersive Engineering https://github.com/BluSunrize/ImmersiveEngineering/blob/1.21.1/src/api/java/blusunrize/immersiveengineering/api/crafting/FluidTagInput.java

/**
 * A FluidIngredient can be either a fluid Identifier or a fluid TagKey
 * <p>
 * Used for input to recipes. The amount defaults to 1 bucket.
 */
public record FluidIngredient(Either<class_6862<class_3611>, class_2960> fluidContent,
                              long amount) implements Predicate<FluidStack> {
    
    // A FluidIngredient should have a "fluid" which can be an identifier with a namespace or a tag beginning in #
    // A FluidIngredient can have an "amount" should be a long integer and will default to 1 bucket
    public static final Endec<FluidIngredient> FLUID_INGREDIENT_ENDEC = StructEndecBuilder.of(
      CodecUtils.eitherEndec(
        Endec.STRING.xmap(
          s -> {
              if (s.charAt(0) != '#') throw new IllegalStateException("tag must start with #");
              return class_6862.method_40092(class_7924.field_41270, class_2960.method_60654(s.substring(1)));
          },
          tag -> "#" + tag.comp_327()
        ),
        MinecraftEndecs.IDENTIFIER
      ).fieldOf("fluid", FluidIngredient::fluidContent),
      Endec.LONG.optionalFieldOf("amount", FluidIngredient::amount, FluidStack.bucketAmount()),
      FluidIngredient::new
    );
    
    public static final class_9139<ByteBuf, class_6862<class_3611>> FLUID_TAG_KEY_CODEC = class_2960.field_48267.method_56432(id -> class_6862.method_40092(class_7924.field_41270, id), class_6862::comp_327);
    
    public static final class_9139<class_9129, Either<class_6862<class_3611>, class_2960>> FLUID_CONTENT_CODEC =
      class_9135.method_57995(FLUID_TAG_KEY_CODEC, class_2960.field_48267);
    
    public static final class_9139<class_9129, FluidIngredient> PACKET_CODEC =
      class_9139.method_56435(
        FLUID_CONTENT_CODEC, FluidIngredient::fluidContent,
        class_9135.field_48551, FluidIngredient::amount,
        FluidIngredient::new
      );
    
    public static final FluidIngredient EMPTY = new FluidIngredient();
    
    public FluidIngredient(Either<class_6862<class_3611>, class_2960> fluidContent, long amount) {
        this.fluidContent = fluidContent;
        this.amount = amount;
    }
    
    // Construct EMPTY FluidIngredient
    public FluidIngredient() {
        this(Either.right(class_7923.field_41173.method_10221(class_3612.field_15906)), 0L);
    }
    
    // All with* methods will return a copy of the record with updated fluid content
    // If the fluid amount is zero when a non-empty fluid is set, it will default the amount to 1 bucket
    public FluidIngredient withContent(class_2960 fluidId) {
        return new FluidIngredient(
          Either.right(fluidId),
          (amount == 0 && class_7923.field_41173.method_10223(fluidId) != class_3612.field_15906) ? FluidStack.bucketAmount() : amount);
    }
    
    public FluidIngredient withContent(class_5321<class_3611> fluidKey) {
        return withContent(fluidKey.method_29177());
    }
    
    public FluidIngredient withContent(class_3611 fluid) {
        return withContent(class_7923.field_41173.method_10221(fluid));
    }
    
    public FluidIngredient withContent(class_6862<class_3611> fluidTag) {
        // even if the tag is empty now, it might not be later.
        // don't test for empty tag at this point. If a tag might be empty, it would be better
        // to add conditional loading for the recipes that use it.
        return new FluidIngredient(Either.left(fluidTag), amount == 0 ? FluidStack.bucketAmount() : amount);
    }
    
    public FluidIngredient withAmount(long withAmount) {
        return new FluidIngredient(fluidContent, withAmount);
    }
    
    public FluidIngredient withAmount(float withAmountInBuckets) {
        return new FluidIngredient(fluidContent, (long) (withAmountInBuckets * FluidStack.bucketAmount()));
    }
    
    public FluidIngredient withSpecificAmount(long amountInMillis) {
        return new FluidIngredient(fluidContent, amountInMillis);
    }
    
    public static FluidIngredient ofStack(FluidStack fluidStack) {
        return new FluidIngredient(Either.right(class_7923.field_41173.method_10221(fluidStack.getFluid())), fluidStack.getAmount());
    }
    
    public class_2561 name() {
        class_2960 fluidId = fluidContent.map(tag -> tag.comp_327(), id -> id);
        return hasTag()
                 ? class_2561.method_30163("#" + fluidId.method_12836() + ":" + fluidId.method_12832())
                 : class_2561.method_43471("fluid." + fluidContent.map(tag -> tag.comp_327(), id -> id).method_42094());
    }
    
    @Override
    public boolean test(@Nullable FluidStack fluidStack) {
        return matchesFluid(fluidStack) && fluidStack.getAmount() >= this.amount;
    }
    
    public boolean matchesFluid(class_3611 fluid) {
        class_7923.field_41173.method_10223(class_2960.method_60654("")).method_15780(fluid);
        return fluidContent.map(tag -> class_7923.field_41173.method_47983(fluid).method_40220(tag), id -> class_7923.field_41173.method_10223(id).method_15780(fluid));
    }
    
    public boolean matchesFluid(@Nullable FluidStack fluidStack) {
        return fluidStack != null && matchesFluid(fluidStack.getFluid());
    }
    
    // Intended for recipe viewer plugins
    public List<FluidStack> getFluidStacks() {
        return (List<FluidStack>) fluidContent.map(
          tag -> class_7923.field_41173.method_40270().filter(fluidEntry -> fluidEntry.method_40220(tag)).map(fluidEntry -> FluidStack.create(fluidEntry.comp_349(), amount)).collect(Collectors.toList()),
          id -> List.of(FluidStack.create(class_7923.field_41173.method_10223(fluidContent.right().get()), amount))
        );
    }
    
    public boolean isEmpty() {
        return this == EMPTY;
    }
    
    public boolean hasTag() {
        return fluidContent.left().isPresent();
    }
    
    // mostly for convenience in recipe viewers
    // make sure this is a tag calling, otherwise it could throw an exception
    public class_6862<class_3611> getTag() {
        return fluidContent.left().get();
    }
    
    // mostly for convenience in recipe viewers
    // make sure this isn't a tag before calling, otherwise it could throw an exception
    public class_3611 getFluid() {
        return class_7923.field_41173.method_10223(fluidContent.right().get());
    }
    
    @Override
    public String toString() {
        return "FluidIngredient{" + "fluidContent={" + "tag=" + fluidContent.left() + ", id=" + fluidContent.right() + "}" + ", amount=" + amount + '}';
    }
}
