package rearth.oritech.block.entity.reactor;

import dev.architectury.registry.menu.ExtendedMenuProvider;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector2i;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.EnergyApi;
import rearth.oritech.api.energy.EnergyApi.EnergyStorage;
import rearth.oritech.api.energy.containers.SimpleEnergyStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.blocks.reactor.*;
import rearth.oritech.client.init.ParticleContent;
import rearth.oritech.client.ui.ReactorScreenHandler;
import rearth.oritech.init.BlockContent;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.init.SoundContent;

import rearth.oritech.util.Geometry;

import java.util.*;
import java.util.Map.Entry;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_3419;
import net.minecraft.class_3545;
import net.minecraft.class_7225;

public class ReactorControllerBlockEntity extends NetworkedBlockEntity implements EnergyApi.BlockProvider, ExtendedMenuProvider {
    
    public static final int MAX_SIZE = Oritech.CONFIG.maxSize();
    public static final int RF_PER_PULSE = Oritech.CONFIG.rfPerPulse();
    public static final int ABSORBER_RATE = Oritech.CONFIG.absorberRate();
    public static final int VENT_BASE_RATE = Oritech.CONFIG.ventBaseRate();
    public static final int VENT_RELATIVE_RATE = Oritech.CONFIG.ventRelativeRate();
    public static final int MAX_HEAT = Oritech.CONFIG.maxHeat();
    public static final int MAX_UNSTABLE_TICKS = Oritech.CONFIG.maxUnstableTicks();
    
    private final HashMap<Vector2i, BaseReactorBlock> activeComponents = new HashMap<>();   // 2d local position on the first layer containing the reactor blocks
    private final HashMap<Vector2i, ReactorFuelPortEntity> fuelPorts = new HashMap<>();     // same grid, but contains a reference to the port at the ceiling
    private final HashMap<Vector2i, ReactorAbsorberPortEntity> absorberPorts = new HashMap<>(); // same
    private final HashMap<Vector2i, Integer> componentHeats = new HashMap<>();              // same grid, contains the current heat of the component
    private final HashSet<class_3545<class_2338, class_2350>> energyPorts = new HashSet<>();   // list of all energy port outputs (e.g. the targets to output to)
    private final HashSet<class_2338> redstonePorts = new HashSet<>();   // list of all redstone ports
    
    @SyncField(SyncType.GUI_TICK)
    public final HashMap<Vector2i, ComponentStatistics> componentStats = new HashMap<>(); // mainly for client displays, same grid
    
    @SyncField(SyncType.GUI_TICK)
    public SimpleEnergyStorage energyStorage = new SimpleEnergyStorage(0, Oritech.CONFIG.reactorMaxEnergyStored(), Oritech.CONFIG.reactorMaxEnergyStored(), this::method_5431);
    
    public boolean active = false;
    
    @SyncField(SyncType.GUI_OPEN)
    public class_2338 areaMin;
    @SyncField(SyncType.GUI_OPEN)
    public class_2338 areaMax;
    
    private int reactorStackHeight;
    private boolean disabledViaRedstone = false;
    private int unstableTicks = 0;
    public long disabledUntil = 0;
    
    private boolean doAutoInit = false; // used to auto-init when save is being loaded
    
    public ReactorControllerBlockEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.REACTOR_CONTROLLER_BLOCK_ENTITY, pos, state);
    }
    
    // heat is only used for reactor rods and heat pipes
    // rods generate heat. Multi-cores and reflectors (expensive) change this
    // heat pipes move heat to themselves
    // vents remove heat from the hottest neighbor component
    // absorbers remove fixed heat amount from all neighboring blocks
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        
        if (!active && doAutoInit) {
            doAutoInit = false;
            init(null);
        }
        
        if (!active || activeComponents.isEmpty()) return;
        
        var activeRods = 0;
        var hottestHeat = 0;
        
        for (var entry : activeComponents.entrySet()) {
            var localPos = entry.getKey();
            var component = entry.getValue();
            var componentHeat = componentHeats.get(localPos);
            
            if (component instanceof ReactorRodBlock rodBlock) {
                
                var ownRodCount = rodBlock.getRodCount();
                var receivedPulses = rodBlock.getInternalPulseCount();
                
                var portEntity = fuelPorts.get(localPos);
                if (portEntity == null || portEntity.method_11015()) {
                    continue;
                }
                
                var hasFuel = portEntity.tryConsumeFuel(ownRodCount * reactorStackHeight, isDisabled() || disabledViaRedstone);
                var heatCreated = 0;
                
                setRodBlockState(localPos, hasFuel);
                
                if (hasFuel) {
                    // check how many pulses are received from neighbors / reflectors
                    for (var neighborPos : getNeighborsInBounds(localPos, activeComponents.keySet())) {
                        
                        var neighbor = activeComponents.get(neighborPos);
                        if (neighbor instanceof ReactorRodBlock neighborRod) {
                            receivedPulses += neighborRod.getRodCount();
                        } else if (neighbor instanceof ReactorReflectorBlock reflectorBlock) {
                            receivedPulses += rodBlock.getRodCount();
                        }
                    }
                    
                    if (!isDisabled()) {
                        activeRods++;
                        energyStorage.insertIgnoringLimit(RF_PER_PULSE * receivedPulses * reactorStackHeight, false);
                    }
                    
                    // generate heat per pulse
                    heatCreated = (receivedPulses / 2 * receivedPulses + 4);
                    componentHeat += heatCreated;
                    
                    if (componentHeat > MAX_HEAT * 0.85) {
                        playMeltdownAnimation(portEntity.method_11016());
                    }
                    
                } else {
                    receivedPulses = 0;
                }
                
                componentStats.put(localPos, new ComponentStatistics((short) receivedPulses, componentHeat, (short) heatCreated));
                
            } else if (component instanceof ReactorHeatPipeBlock heatPipeBlock) {
                
                var sumGainedHeat = 0;
                
                // take heat in from neighbors
                for (var neighbor : getNeighborsInBounds(localPos, activeComponents.keySet())) {
                    var neighborHeat = componentHeats.get(neighbor);
                    if (neighborHeat <= componentHeat) continue;
                    var diff = neighborHeat - componentHeat;
                    var gainedHeat = Math.min(diff / 4 + 10, diff);
                    neighborHeat -= gainedHeat;
                    componentHeats.put(neighbor, neighborHeat);
                    componentHeat += gainedHeat;
                    sumGainedHeat += gainedHeat;
                }
                
                componentStats.put(localPos, new ComponentStatistics((short) 0, componentHeat, (short) sumGainedHeat));
                
            } else if (component instanceof ReactorAbsorberBlock absorberBlock) {
                
                var sumRemovedHeat = 0;
                var portEntity = absorberPorts.get(localPos);
                if (portEntity == null || portEntity.method_11015()) {
                    continue;
                }
                var fuelAvailable = portEntity.getAvailableFuel();
                
                if (fuelAvailable >= reactorStackHeight) {
                    // take heat in from neighbors and remove it
                    for (var neighbor : getNeighborsInBounds(localPos, activeComponents.keySet())) {
                        var neighborHeat = componentHeats.get(neighbor);
                        if (neighborHeat <= 0) continue;
                        neighborHeat -= ABSORBER_RATE;
                        sumRemovedHeat += ABSORBER_RATE;
                        componentHeats.put(neighbor, neighborHeat);
                    }
                } else if (fuelAvailable > 0) {
                    // remove last small unusable part
                    portEntity.consumeFuel(fuelAvailable);
                }
                
                if (sumRemovedHeat > 0) {
                    portEntity.consumeFuel(reactorStackHeight);
                }
                
                componentStats.put(localPos, new ComponentStatistics((short) 0, 0, (short) sumRemovedHeat));
            } else if (component instanceof ReactorHeatVentBlock ventBlock) {
                
                // remove heat from hottest neighbor
                
                var hottestPos = localPos;
                var max = 0;
                for (var neighbor : getNeighborsInBounds(localPos, activeComponents.keySet())) {
                    var neighborHeat = componentHeats.get(neighbor);
                    if (neighborHeat <= max) continue;
                    hottestPos = neighbor;
                    max = neighborHeat;
                }
                
                var removed = 0;
                if (max != 0) {
                    var neighborHeat = max;
                    removed = Math.min(neighborHeat / VENT_RELATIVE_RATE + VENT_BASE_RATE, neighborHeat);
                    neighborHeat -= removed;
                    componentHeats.put(hottestPos, neighborHeat);
                }
                
                componentStats.put(localPos, new ComponentStatistics((short) 0, 0, (short) removed));
                
            }
            
            componentHeats.put(localPos, componentHeat);
            
            if (componentHeat > hottestHeat)
                hottestHeat = componentHeat;
            
        }
        
        outputEnergy();
        updateRedstonePorts(hottestHeat, activeRods);
        
        if (activeRods > 0)
            playAmbientSound();
        
        if (activeRods > 0 && hottestHeat > MAX_HEAT * 0.8f) {
            playWarningSound();
        }
        
        if (hottestHeat > MAX_HEAT && activeRods > 0) {
            unstableTicks++;
            if (unstableTicks > MAX_UNSTABLE_TICKS)
                doReactorExplosion(activeRods * reactorStackHeight);
        } else {
            unstableTicks = 0;
        }
        
    }
    
    @Override
    public void preNetworkUpdate(SyncType type) {
        super.preNetworkUpdate(type);
        
        if (type != SyncType.GUI_TICK) return;
        for (var port : fuelPorts.values()) port.updateNetwork();
        for (var port : absorberPorts.values()) port.updateNetwork();
    }
    
    private boolean isDisabled() {
        return field_11863.method_8510() < disabledUntil;
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        
        nbt.method_10544("energy_stored", energyStorage.getAmount());
        nbt.method_10556("was_active", active);
        nbt.method_10556("redstone_disabled", disabledViaRedstone);
        
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        
        energyStorage.setAmount(nbt.method_10537("energy_stored"));
        doAutoInit = nbt.method_10577("was_active");
        disabledViaRedstone = nbt.method_10577("redstone_disabled");
    }
    
    private void playMeltdownAnimation(class_2338 port) {
        ParticleContent.MELTDOWN_IMMINENT.spawn(field_11863, port.method_46558().method_1031(0, 0.3, 0), 5);
    }
    
    private void playAmbientSound() {
        var soundDuration = 250;
        
        if (field_11863.method_8510() % soundDuration == 0)
            field_11863.method_8396(null, field_11867, SoundContent.REACTOR, class_3419.field_15245, 0.7f, 0.8f);
    }
    
    
    private void playWarningSound() {
        var soundDuration = 50;
        
        if (field_11863.method_8510() % soundDuration == 0)
            field_11863.method_8396(null, field_11867, SoundContent.REACTOR_WARNING, class_3419.field_15245, 4f, 0.8f);
    }
    
    // strength is the amount of total active rods (e.g. activeRods * stackHeight)
    private void doReactorExplosion(int strength) {
        
        if (Oritech.CONFIG.safeMode()) {
            disableReactor();
            return;
        }
        
        var spawnedBlock = BlockContent.REACTOR_EXPLOSION_SMALL;
        if (strength > 8 && strength <= 25) {
            spawnedBlock = BlockContent.REACTOR_EXPLOSION_MEDIUM;
        } else if (strength > 25) {
            spawnedBlock = BlockContent.REACTOR_EXPLOSION_LARGE;
        }
        
        field_11863.method_8501(field_11867, spawnedBlock.method_9564());
    }
    
    private void disableReactor() {
        this.disabledUntil = field_11863.method_8510() + Oritech.CONFIG.safeModeCooldown();
    }
    
    public void init(@Nullable class_1657 player) {
        
        active = false;
        
        // find low and high corners of reactor
        var cornerA = field_11867;
        cornerA = expandWall(cornerA, new class_2382(0, -1, 0), true);   // first go down through other wall blocks
        cornerA = expandWall(cornerA, new class_2382(0, 0, -1));
        cornerA = expandWall(cornerA, new class_2382(-1, 0, 0));
        cornerA = expandWall(cornerA, new class_2382(0, 0, -1)); // expand z again to support all rotations
        
        var cornerB = cornerA;
        cornerB = expandWall(cornerB, new class_2382(0, 1, 0));
        cornerB = expandWall(cornerB, new class_2382(0, 0, 1));
        cornerB = expandWall(cornerB, new class_2382(1, 0, 0));
        
        if (cornerA == field_11867 || cornerB == field_11867 || cornerA == cornerB || onSameAxis(cornerA, cornerB)) {
            if (player != null)
                player.method_43496(class_2561.method_43471("message.oritech.reactor_edge_invalid"));
            return;
        }
        
        // verify and load all blocks in reactor area
        var finalCornerA = cornerA;
        var finalCornerB = cornerB;
        
        // these get loaded in the next step
        energyPorts.clear();
        redstonePorts.clear();
        
        // verify edges
        var wallsValid = class_2338.method_20437(cornerA, cornerB).allMatch(pos -> {
            if (isAtEdgeOfBox(pos, finalCornerA, finalCornerB)) {
                var block = field_11863.method_8320(pos).method_26204();
                return block instanceof ReactorWallBlock;
            } else if (isOnWall(pos, finalCornerA, finalCornerB)) {
                var state = field_11863.method_8320(pos);
                var block = state.method_26204();
                
                // load wall energy ports
                if (block instanceof ReactorEnergyPortBlock) {
                    var facing = state.method_11654(class_2741.field_12525);
                    var blockInFront = pos.method_10081(Geometry.getForward(facing));
                    energyPorts.add(new class_3545<>(blockInFront, class_2350.method_50026(Geometry.getBackward(facing).method_10263(), Geometry.getBackward(facing).method_10264(), Geometry.getBackward(facing).method_10260())));
                } else if (block instanceof ReactorRedstonePortBlock) {
                    redstonePorts.add(pos.method_10062());
                }
                
                return !(block instanceof BaseReactorBlock reactorBlock) || reactorBlock.validForWalls();
            }
            
            return true;
        });
        
        if (!wallsValid) {
            if (player != null)
                player.method_43496(class_2561.method_43471("message.oritech.reactor_wall_invalid"));
            return;
        }
        
        // verify interior is identical in all layers
        var interiorHeight = cornerB.method_10264() - cornerA.method_10264() - 1;
        var cornerAFlat = cornerA.method_10069(1, 1, 1);
        var cornerBFlat = new class_2338(cornerB.method_10263() - 1, cornerA.method_10264() + 1, cornerB.method_10260() - 1);
        
        // these get loaded in the next step
        fuelPorts.clear();
        absorberPorts.clear();
        reactorStackHeight = interiorHeight;
        
        var interiorStackedRight = class_2338.method_20437(cornerAFlat, cornerBFlat).allMatch(pos -> {
            
            var offset = pos.method_10059(cornerAFlat);
            var localPos = new Vector2i(offset.method_10263(), offset.method_10260());
            
            var block = field_11863.method_8320(pos).method_26204();
            if (!(block instanceof BaseReactorBlock reactorBlock)) return true;
            
            for (int i = 1; i < interiorHeight; i++) {
                var candidatePos = pos.method_10069(0, i, 0);
                var candidate = field_11863.method_8320(candidatePos);
                if (!candidate.method_26204().equals(block))
                    return false;
            }
            
            var requiredCeiling = reactorBlock.requiredStackCeiling();
            if (requiredCeiling != class_2246.field_10124) {
                var ceilingPos = pos.method_10069(0, interiorHeight, 0);
                var ceilingBlock = field_11863.method_8320(ceilingPos).method_26204();
                if (!requiredCeiling.equals(ceilingBlock)) return false;
                
                if (block instanceof ReactorRodBlock) {
                    fuelPorts.put(localPos, (ReactorFuelPortEntity) field_11863.method_8321(ceilingPos));
                } else if (block instanceof ReactorAbsorberBlock) {
                    absorberPorts.put(localPos, (ReactorAbsorberPortEntity) field_11863.method_8321(ceilingPos));
                }
                
            }
            activeComponents.put(localPos, reactorBlock);
            componentHeats.putIfAbsent(localPos, 0);
            
            return true;
        });
        
        if (!interiorStackedRight) {
            if (player != null)
                player.method_43496(class_2561.method_43471("message.oritech.reactor_interior_issues"));
            return;
        }
        
        areaMin = finalCornerA;
        areaMax = finalCornerB;
        active = true;
        
    }
    
    private void setRodBlockState(Vector2i localPos, boolean on) {
        if (field_11863.method_8510() % 10 != 0) return;
        var stackTop = fuelPorts.get(localPos).method_11016();
        
        for (int i = 1; i <= reactorStackHeight; i++) {
            var candidatePos = stackTop.method_10087(i);
            var candidateState = field_11863.method_8320(candidatePos);
            if (!(candidateState.method_26204() instanceof ReactorRodBlock)) continue;
            var oldLit = candidateState.method_11654(class_2741.field_12548);
            if (oldLit != on) {
                // update only when changed
                field_11863.method_30092(candidatePos, candidateState.method_11657(class_2741.field_12548, on), class_2248.field_31028, 0);
            }
        }
    }
    
    private static Set<Vector2i> getNeighborsInBounds(Vector2i pos, Set<Vector2i> keys) {
        
        var res = new HashSet<Vector2i>(4);
        
        var a = new Vector2i(pos).add(-1, 0);
        if (keys.contains(a)) res.add(a);
        var b = new Vector2i(pos).add(0, 1);
        if (keys.contains(b)) res.add(b);
        var c = new Vector2i(pos).add(1, 0);
        if (keys.contains(c)) res.add(c);
        var d = new Vector2i(pos).add(0, -1);
        if (keys.contains(d)) res.add(d);
        
        return res;
    }
    
    private static boolean onSameAxis(class_2338 A, class_2338 B) {
        return A.method_10263() == B.method_10263() || A.method_10264() == B.method_10264() || A.method_10260() == B.method_10260();
    }
    
    private static boolean isOnWall(class_2338 pos, class_2338 min, class_2338 max) {
        return onSameAxis(pos, min) || onSameAxis(pos, max);
    }
    
    private static boolean isAtEdgeOfBox(class_2338 pos, class_2338 min, class_2338 max) {
        int planesAligned = 0;
        
        if (pos.method_10263() == min.method_10263() || pos.method_10263() == max.method_10263()) planesAligned++;
        if (pos.method_10264() == min.method_10264() || pos.method_10264() == max.method_10264()) planesAligned++;
        if (pos.method_10260() == min.method_10260() || pos.method_10260() == max.method_10260()) planesAligned++;
        
        return planesAligned >= 2;
    }
    
    private class_2338 expandWall(class_2338 from, class_2382 direction) {
        return expandWall(from, direction, false);
    }
    
    private class_2338 expandWall(class_2338 from, class_2382 direction, boolean allReactorBlocks) {
        
        var result = from;
        for (int i = 1; i < MAX_SIZE; i++) {
            var candidate = from.method_10081(direction.method_35862(i));
            var candidateBlock = field_11863.method_8320(candidate).method_26204();
            
            if (!allReactorBlocks && !(candidateBlock instanceof ReactorWallBlock)) return result;
            if (allReactorBlocks && !(candidateBlock instanceof BaseReactorBlock)) return result;
            
            result = candidate;
        }
        
        return result;
        
    }
    
    private void updateRedstonePorts(int hottestTemp, int filledRods) {
        
        disabledViaRedstone = false;
        
        for (var pos : redstonePorts) {
            var state = field_11863.method_8320(pos);
            if (!state.method_26204().equals(BlockContent.REACTOR_REDSTONE_PORT)) continue;
            
            var resOutput = 0;
            
            var mode = state.method_11654(ReactorRedstonePortBlock.PORT_MODE);
            if (mode == 0 && hottestTemp > 0) {    // temp of hottest component
                resOutput = (int) ((hottestTemp / (float) MAX_HEAT) * 15);
                resOutput = Math.max(resOutput, 1);  // ensure at least level 1 if any component has heat
            } else if (mode == 1) { // amount of rods with fuel
                resOutput = Math.min(filledRods, 15);
            } else if (mode == 2 && energyStorage.getAmount() > 0) { // amount of energy stored
                var fillPercentage = energyStorage.getAmount() / (float) energyStorage.getCapacity();
                resOutput = (int) (1 + fillPercentage * 14);
            }
            
            resOutput = Math.min(resOutput, 15);
            
            var lastLevel = state.method_11654(class_2741.field_12511);
            if (lastLevel != resOutput) {
                field_11863.method_8501(pos, state.method_11657(class_2741.field_12511, resOutput));
                field_11863.method_8524(pos);
            }
            
            if (field_11863.method_49803(pos)) {
                disabledViaRedstone = true;
            }
            
        }
        
    }
    
    private void outputEnergy() {
        
        var totalMoved = 0;
        var maxRatePerSlot = Oritech.CONFIG.reactorMaxEnergyOutput();
        
        var randomOrderedList = new ArrayList<>(energyPorts);
        Collections.shuffle(randomOrderedList);
        
        for (var candidateData : randomOrderedList) {
            var candidate = EnergyApi.BLOCK.find(field_11863, candidateData.method_15442(), candidateData.method_15441());
            if (candidate == null) continue;
            var moved = EnergyApi.transfer(energyStorage, candidate, maxRatePerSlot, false);
            
            if (moved > 0)
                candidate.update();
            
            totalMoved += moved;
        }
        
        if (totalMoved > 0)
            energyStorage.update();
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorage(class_2350 direction) {
        return energyStorage;
    }
    
    @Override
    public class_2561 method_5476() {
        return class_2561.method_30163("");
    }
    
    @Nullable
    @Override
    public class_1703 createMenu(int syncId, class_1661 playerInventory, class_1657 player) {
        return new ReactorScreenHandler(syncId, playerInventory, this);
    }
    
    @Override
    public void saveExtraData(class_2540 buf) {
        sendUpdate(SyncType.GUI_OPEN);
        buf.method_10807(field_11867);
    }
    
    public record ComponentStatistics(short receivedPulses, int storedHeat, short heatChanged) {
        public static final ComponentStatistics EMPTY = new ComponentStatistics((short) 0, -1, (short) 0);
    }
}
