package rearth.oritech.block.entity.processing;

import dev.architectury.fluid.FluidStack;
import dev.architectury.hooks.fluid.FluidStackHooks;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.OritechPlatform;
import rearth.oritech.api.fluid.FluidApi;
import rearth.oritech.api.fluid.containers.SimpleInOutFluidStorage;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.base.entity.MultiblockMachineEntity;
import rearth.oritech.block.entity.MachineCoreEntity;
import rearth.oritech.client.init.ModScreens;
import rearth.oritech.client.ui.CentrifugeScreenHandler;
import rearth.oritech.init.BlockContent;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.init.recipes.OritechRecipe;
import rearth.oritech.init.recipes.OritechRecipeType;
import rearth.oritech.init.recipes.RecipeContent;
import rearth.oritech.util.FluidIngredient;
import rearth.oritech.util.InventorySlotAssignment;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_2487;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3917;
import net.minecraft.class_7225;
import net.minecraft.class_8786;

public class CentrifugeBlockEntity extends MultiblockMachineEntity implements FluidApi.BlockProvider {
    
    @SyncField(SyncType.GUI_TICK)
    public final SimpleInOutFluidStorage fluidContainer = new SimpleInOutFluidStorage(Oritech.CONFIG.processingMachines.centrifugeData.tankSizeInBuckets() * FluidStackHooks.bucketAmount(), this::method_5431);
    
    @SyncField(SyncType.GUI_OPEN)
    public boolean hasFluidAddon = false;
    
    public CentrifugeBlockEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.CENTRIFUGE_ENTITY, pos, state, Oritech.CONFIG.processingMachines.centrifugeData.energyPerTick());
    }
    
    @Override
    public long getDefaultCapacity() {
        return Oritech.CONFIG.processingMachines.centrifugeData.energyCapacity();
    }
    
    @Override
    public long getDefaultInsertRate() {
        return Oritech.CONFIG.processingMachines.centrifugeData.maxEnergyInsertion();
    }
    
    @Override
    protected boolean canProceed(OritechRecipe recipe) {
        
        if (!hasFluidAddon) return super.canProceed(recipe);
        
        if (!recipeInputMatchesTank(fluidContainer.getInStack(), recipe)) return false;
        
        // check if output fluid would fit
        var output = recipe.getFluidOutputs().isEmpty() ? null : recipe.getFluidOutputs().getFirst();
        if (output != null && !output.isEmpty()) { // only verify fluid output if fluid output exists
            
            if (fluidContainer.getOutStack().getAmount() + output.getAmount() > fluidContainer.getCapacity())
                return false; // output too full
            
            if (!fluidContainer.getOutStack().isEmpty() && !output.isFluidEqual(fluidContainer.getOutStack()))
                return false;   // output type mismatch
        }
        
        return true;
        
    }
    
    @Override
    protected Optional<class_8786<OritechRecipe>> getRecipe() {
        
        if (!hasFluidAddon)
            return super.getRecipe();
        
        // get recipes matching input items
        var candidates = Objects.requireNonNull(field_11863).method_8433().method_17877(getOwnRecipeType(), getInputInventory(), field_11863);
        // filter out recipes based on input tank
        var fluidRecipe = candidates.stream().filter(candidate -> recipeInputMatchesTank(fluidContainer.getInStack(), candidate.comp_1933())).findAny();
        if (fluidRecipe.isPresent()) {
            return fluidRecipe;
        }
        
        return getNormalRecipe();
    }
    
    // this is provided as fallback for fluid centrifuges that may still process normal stuff
    private Optional<class_8786<OritechRecipe>> getNormalRecipe() {
        return field_11863.method_8433().method_8132(RecipeContent.CENTRIFUGE, getInputInventory(), field_11863);
    }
    
    public static boolean recipeInputMatchesTank(FluidStack available, OritechRecipe recipe) {
        
        var recipeNeedsFluid = recipe.getFluidInput() != null && recipe.getFluidInput().amount() > 0;
        if (!recipeNeedsFluid) return true;
        
        var isTankEmpty = available.isEmpty();
        if (isTankEmpty) return false;
        
        var recipeFluid = recipe.getFluidInput();
        return recipeFluid.matchesFluid(available) && available.getAmount() >= recipe.getFluidInput().amount();
    }
    
    @Override
    protected void craftItem(OritechRecipe activeRecipe, List<class_1799> outputInventory, List<class_1799> inputInventory) {
        
        var chamberCount = getBaseAddonData().extraChambers() + 1;
        
        for (int i = 0; i < chamberCount; i++) {
            var newRecipe = getRecipe();
            if (newRecipe.isEmpty() || !newRecipe.get().comp_1933().equals(currentRecipe) || !canOutputRecipe(activeRecipe) || !canProceed(activeRecipe)) break;
            super.craftItem(activeRecipe, outputInventory, inputInventory);
            
            if (hasFluidAddon) {
                craftFluids(activeRecipe);
            }
        }
    }
    
    @Override
    public boolean supportExtraChambersAuto() {
        return false;
    }
    
    private void craftFluids(OritechRecipe activeRecipe) {
        
        var input = activeRecipe.getFluidInput();
        var output = activeRecipe.getFluidOutputs().isEmpty() ? null : activeRecipe.getFluidOutputs().getFirst();
        
        if (input != null && input.amount() > 0)
            fluidContainer.getInputContainer().extract(fluidContainer.getInStack().copyWithAmount(input.amount()), false);
        if (output != null && output.getAmount() > 0)
            fluidContainer.getOutputContainer().insert(output, false);
        
    }
    
    @Override
    public void getAdditionalStatFromAddon(AddonBlock addonBlock) {
        if (addonBlock.state().method_26204().equals(BlockContent.MACHINE_FLUID_ADDON)) {
            hasFluidAddon = true;
        }
    }
    
    @Override
    public void resetAddons() {
        super.resetAddons();
        hasFluidAddon = false;
    }
    
    @Override
    public void initAddons(class_2338 brokenAddon) {
        
        var hadAddon = hasFluidAddon;
        hasFluidAddon = false;
        super.initAddons(brokenAddon);
        
        if (hasFluidAddon != hadAddon && field_11863 instanceof class_3218 serverLevel) {
            
            // reset cache of core above
            var coreCandidate = field_11863.method_35230(field_11867.method_10084(), BlockEntitiesContent.MACHINE_CORE_ENTITY);
            if (coreCandidate.isPresent()) {
                var core = coreCandidate.get();
                core.resetCaches();
            }
            
            OritechPlatform.INSTANCE.resetCapabilities(serverLevel, field_11867);
            OritechPlatform.INSTANCE.resetCapabilities(serverLevel, field_11867.method_10084());
            
            // trigger block update to allow pipes to connect/disconnect
            field_11863.method_8408(field_11867, method_11010().method_26204());
            field_11863.method_8408(field_11867.method_10084(), field_11863.method_8320(field_11867.method_10084()).method_26204());
        }
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        nbt.method_10556("fluidAddon", hasFluidAddon);
        fluidContainer.writeNbt(nbt, "");
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        
        hasFluidAddon = nbt.method_10577("fluidAddon");
        fluidContainer.readNbt(nbt, "");
    }
    
    @Override
    protected OritechRecipeType getOwnRecipeType() {
        if (hasFluidAddon) return RecipeContent.CENTRIFUGE_FLUID;
        return RecipeContent.CENTRIFUGE;
    }
    
    @Override
    public InventorySlotAssignment getSlotAssignments() {
        return new InventorySlotAssignment(0, 1, 1, 2);
    }
    
    @Override
    public List<GuiSlot> getGuiSlots() {
        return List.of(
          new GuiSlot(0, 56, 38),
          new GuiSlot(1, 113, 38, true),
          new GuiSlot(2, 113, 56, true));
    }
    
    @Override
    public class_3917<?> getScreenHandlerType() {
        return ModScreens.CENTRIFUGE_SCREEN;
    }
    
    @Override
    public int getInventorySize() {
        return 3;
    }
    
    @Override
    public List<class_2382> getCorePositions() {
        return List.of(
          new class_2382(0, 1, 0)
        );
    }
    
    @Override
    public boolean inputOptionsEnabled() {
        return false;
    }
    
    @Override
    public List<class_2382> getAddonSlots() {
        
        return List.of(
          new class_2382(0, 0, -1),
          new class_2382(0, 0, 1)
        );
    }
    
    @Nullable
    @Override
    public class_1703 createMenu(int syncId, class_1661 playerInventory, class_1657 player) {
        return new CentrifugeScreenHandler(syncId, playerInventory, this);
    }
    
    @Override
    public int getAnimationDuration() {
        return 20 * 9;
    }
    
    @Override
    public FluidApi.FluidStorage getFluidStorage(@Nullable class_2350 direction) {
        if (!hasFluidAddon) return null;
        return fluidContainer;
    }
}
