package rearth.oritech.api.networking;

import com.mojang.serialization.Codec;
import dev.architectury.fluid.FluidStack;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.apache.logging.log4j.util.TriConsumer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector2i;
import rearth.oritech.Oritech;
import rearth.oritech.OritechPlatform;
import rearth.oritech.block.base.entity.MachineBlockEntity;
import rearth.oritech.block.entity.accelerator.AcceleratorControllerBlockEntity;
import rearth.oritech.block.entity.addons.InventoryProxyAddonBlockEntity;
import rearth.oritech.block.entity.addons.RedstoneAddonBlockEntity;
import rearth.oritech.block.entity.arcane.EnchanterBlockEntity;
import rearth.oritech.block.entity.arcane.EnchantmentCatalystBlockEntity;
import rearth.oritech.block.entity.arcane.SpawnerControllerBlockEntity;
import rearth.oritech.block.entity.augmenter.AugmentApplicationEntity;
import rearth.oritech.block.entity.augmenter.PlayerAugments;
import rearth.oritech.block.entity.interaction.LaserArmBlockEntity;
import rearth.oritech.block.entity.pipes.ItemFilterBlockEntity;
import rearth.oritech.block.entity.pipes.ItemPipeInterfaceEntity;
import rearth.oritech.init.recipes.OritechRecipe;
import rearth.oritech.init.recipes.OritechRecipeType;
import rearth.oritech.item.tools.PortableLaserItem;
import rearth.oritech.item.tools.armor.JetpackItem;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_3545;
import net.minecraft.class_5455;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import net.minecraft.class_8710;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;

public class NetworkManager {
    
    private static final Map<Type, class_9139<? extends ByteBuf, ?>> AUTO_CODECS = new HashMap<>();
    private static final Map<Integer, List<Field>> CACHED_FIELDS = new HashMap<Integer, List<Field>>();
    
    // these two are basically copies of the architectury built-in fluid stack codecs, but using the OPTIONAL_STREAM_CODEC to allow for empty fluid stacks
    public static Codec<FluidStack> FLUID_STACK_CODEC;
    public static class_9139<class_9129, FluidStack> FLUID_STACK_STREAM_CODEC;
    
    public static void sendBlockHandle(class_2586 blockEntity, class_8710 message) {
        OritechPlatform.INSTANCE.sendBlockHandle(blockEntity, message);
    }
    
    public static void sendPlayerHandle(class_8710 message, class_3222 player) {
        OritechPlatform.INSTANCE.sendPlayerHandle(message, player);
    }
    
    public static void sendToServer(class_8710 message) {
        OritechPlatform.INSTANCE.sendToServer(message);
    }
    
    public static <T extends class_8710> void registerToClient(class_8710.class_9154<T> id, class_9139<class_9129, T> packetCodec, TriConsumer<T, class_1937, class_5455> consumer) {
        OritechPlatform.INSTANCE.registerToClient(id, packetCodec, consumer);
    }
    
    public static <T extends class_8710> void registerToServer(class_8710.class_9154<T> id, class_9139<class_9129, T> packetCodec, TriConsumer<T, class_1657, class_5455> consumer) {
        OritechPlatform.INSTANCE.registerToServer(id, packetCodec, consumer);
    }
    
    public static void registerDefaultCodecs() {
        
        registerCodec(class_9135.field_49675, Integer.class, int.class);
        registerCodec(class_9135.field_48551, Long.class, long.class);
        registerCodec(class_9135.field_48552, Float.class, float.class);
        registerCodec(class_9135.field_48547, Boolean.class, boolean.class);
        registerCodec(class_9135.field_48553, Double.class, double.class);
        registerCodec(class_9135.field_48548, Byte.class, byte.class);
        registerCodec(class_9135.field_48549, Short.class, short.class);
        registerCodec(class_9135.field_48554, String.class);
        registerCodec(class_2960.field_48267, class_2960.class);
        registerCodec(class_2338.field_48404, class_2338.class);
        registerCodec(class_1799.field_49268, class_1799.class);
        registerCodec(VEC2I_PACKED_CODEC, Vector2i.class);
        registerCodec(VEC3D_PACKET_CODEC, class_243.class);
        registerCodec(SIMPLE_BLOCK_STATE_PACKET_CODEC, class_2680.class);
        registerCodec(FLUID_STACK_STREAM_CODEC, FluidStack.class);
        registerCodec(ItemFilterBlockEntity.FilterData.PACKET_CODEC, ItemFilterBlockEntity.FilterData.class);
        registerCodec(OritechRecipeType.PACKET_CODEC, OritechRecipe.class);
        registerCodec(LaserArmBlockEntity.LASER_TARGET_PACKET_CODEC, class_1309.class);
        registerCodec(AugmentApplicationEntity.ResearchState.PACKET_CODEC, AugmentApplicationEntity.ResearchState.class);
        
    }
    
    public static <T> void registerCodec(class_9139<? extends ByteBuf, T> codec, Type... classes) {
        for (var clazz : classes)
            AUTO_CODECS.put(clazz, codec);
    }
    
    @SuppressWarnings("unchecked")
    public static void init() {
        registerDefaultCodecs();
        
        registerToServer(ItemFilterBlockEntity.ItemFilterPayload.FILTER_PACKET_ID, ItemFilterBlockEntity.ItemFilterPayload.PACKET_CODEC, ItemFilterBlockEntity::handleClientUpdate);
        registerToServer(EnchanterBlockEntity.SelectEnchantingPacket.PACKET_ID, getAutoCodec(EnchanterBlockEntity.SelectEnchantingPacket.class), EnchanterBlockEntity::receiveEnchantmentSelection);
        registerToServer(RedstoneAddonBlockEntity.RedstoneAddonServerUpdate.PACKET_ID, getAutoCodec(RedstoneAddonBlockEntity.RedstoneAddonServerUpdate.class), RedstoneAddonBlockEntity::receiveOnServer);
        registerToServer(PortableLaserItem.LaserPlayerUsePacket.PACKET_ID, getAutoCodec(PortableLaserItem.LaserPlayerUsePacket.class), PortableLaserItem::receiveUsePacket);
        registerToServer(MachineBlockEntity.InventoryInputModeSelectorPacket.PACKET_ID, getAutoCodec(MachineBlockEntity.InventoryInputModeSelectorPacket.class), MachineBlockEntity::receiveCycleModePacket);
        registerToServer(InventoryProxyAddonBlockEntity.InventoryProxySlotSelectorPacket.PACKET_ID, getAutoCodec(InventoryProxyAddonBlockEntity.InventoryProxySlotSelectorPacket.class), InventoryProxyAddonBlockEntity::receiveSlotSelection);
        registerToServer(JetpackItem.JetpackUsageUpdatePacket.PACKET_ID, getAutoCodec(JetpackItem.JetpackUsageUpdatePacket.class), JetpackItem::receiveUsagePacket);
        registerToServer(PlayerAugments.AugmentInstallTriggerPacket.PACKET_ID, getAutoCodec(PlayerAugments.AugmentInstallTriggerPacket.class), PlayerAugments::receiveInstallTrigger);
        registerToServer(PlayerAugments.LoadPlayerAugmentsToMachinePacket.PACKET_ID, getAutoCodec(PlayerAugments.LoadPlayerAugmentsToMachinePacket.class), PlayerAugments::receivePlayerLoadMachine);
        registerToServer(PlayerAugments.OpenAugmentScreenPacket.PACKET_ID, getAutoCodec(PlayerAugments.OpenAugmentScreenPacket.class), PlayerAugments::receiveOpenAugmentScreen);
        registerToServer(PlayerAugments.AugmentPlayerTogglePacket.PACKET_ID, getAutoCodec(PlayerAugments.AugmentPlayerTogglePacket.class), PlayerAugments::receiveToggleAugment);
        
        
        registerToClient(MessagePayload.GENERIC_PACKET_ID, MessagePayload.PACKET_CODEC, NetworkManager::receiveMessage);
        registerToClient(ItemPipeInterfaceEntity.RenderStackData.PIPE_ITEMS_ID, getAutoCodec(ItemPipeInterfaceEntity.RenderStackData.class), ItemPipeInterfaceEntity::receiveVisualItemsPacket);
        registerToClient(EnchantmentCatalystBlockEntity.CatalystSyncPacket.PACKET_ID, getAutoCodec(EnchantmentCatalystBlockEntity.CatalystSyncPacket.class), EnchantmentCatalystBlockEntity::receiveUpdatePacket);
        registerToClient(SpawnerControllerBlockEntity.SpawnerSyncPacket.PACKET_ID, getAutoCodec(SpawnerControllerBlockEntity.SpawnerSyncPacket.class), SpawnerControllerBlockEntity::receiveUpdatePacket);
        registerToClient(RedstoneAddonBlockEntity.RedstoneAddonClientUpdate.PACKET_ID, getAutoCodec(RedstoneAddonBlockEntity.RedstoneAddonClientUpdate.class), RedstoneAddonBlockEntity::receiveOnClient);
        registerToClient(AcceleratorControllerBlockEntity.ParticleRenderTrail.PACKET_ID, getAutoCodec(AcceleratorControllerBlockEntity.ParticleRenderTrail.class), AcceleratorControllerBlockEntity::receiveTrail);
        registerToClient(AcceleratorControllerBlockEntity.LastEventPacket.PACKET_ID, getAutoCodec(AcceleratorControllerBlockEntity.LastEventPacket.class), AcceleratorControllerBlockEntity::receiveEvent);
    }
    
    public static void receiveMessage(MessagePayload message, class_1937 world, class_5455 registryAccess) {
        var receivedBuf = new class_9129(Unpooled.wrappedBuffer(message.message), registryAccess);
        var receiverEntity = world.method_8321(message.pos);
        var receiverType = registryAccess.method_30530(class_7924.field_41255).method_10223(message.targetEntityType);
        if (receiverEntity != null && receiverType != null && receiverType.equals(receiverEntity.method_11017())) {
            decodeFields(receiverEntity, message.syncType, receivedBuf, world);
            if (receiverEntity instanceof NetworkedEventHandler networkedBlock) {
                networkedBlock.onNetworkUpdated();
            }
        } else {
            Oritech.LOGGER.debug("Unable to start decoding for block entity type {} at {}. Target Mismatch!", receiverType, message.pos);
        }
    }
    
    // returns the number of encoded fields
    @SuppressWarnings({"unchecked", "rawtypes"})
    public static int encodeFields(Object target, SyncType type, ByteBuf byteBuf, @Nullable class_1937 world) {
        
        var fields = getCachedFields(target, type);
        
        var encodedCount = 0;
        for (var field : fields) {
            try {
                if (UpdatableField.class.isAssignableFrom(field.getType())) {
                    var fieldInstance = ((UpdatableField) field.get(target));
                    var deltaOnly = fieldInstance.useDeltaOnly(type);
                    var dataToSend = deltaOnly ? fieldInstance.getDeltaData() : fieldInstance;
                    var codec = deltaOnly ? fieldInstance.getDeltaCodec() : fieldInstance.getFullCodec();
                    if (codec instanceof WorldPacketCodec worldPacketCodec) {
                        worldPacketCodec.encode(byteBuf, dataToSend, world);
                    } else {
                        codec.encode(byteBuf, dataToSend);
                    }
                } else {
                    var codec = getAutoCodec(field);
                    var value = field.get(target);
                    if (codec instanceof WorldPacketCodec worldPacketCodec) {
                        worldPacketCodec.encode(byteBuf, value, world);
                    } else {
                        codec.encode(byteBuf, value);
                    }
                }
                
                encodedCount++;
                
            } catch (Exception ex) {
                Oritech.LOGGER.warn("failed to encode field: {}", field.getName(), ex);
            }
        }
        
        return encodedCount;
    }
    
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static void decodeFields(Object target, SyncType type, ByteBuf byteBuf, class_1937 world) {
        
        var fields = getCachedFields(target, type);
        
        for (var field : fields) {
            try {
                // fields that implement UpdatableField either get a delta or full update. Otherwise, we just set the full value
                if (UpdatableField.class.isAssignableFrom(field.getType())) {
                    var fieldInstance = ((UpdatableField) field.get(target));
                    var deltaOnly = fieldInstance.useDeltaOnly(type);
                    var codec = deltaOnly ? fieldInstance.getDeltaCodec() : fieldInstance.getFullCodec();
                    Object value;
                    if (codec instanceof WorldPacketCodec worldPacketCodec) {
                        value = worldPacketCodec.decode(byteBuf, world);
                    } else {
                        value = codec.decode(byteBuf);
                    }
                    if (deltaOnly) {
                        fieldInstance.handleDeltaUpdate(value);
                    } else {
                        fieldInstance.handleFullUpdate(value);
                    }
                } else {
                    var codec = getAutoCodec(field);
                    Object value;
                    if (codec instanceof WorldPacketCodec worldPacketCodec) {
                        value = worldPacketCodec.decode(byteBuf, world);
                    } else {
                        value = codec.decode(byteBuf);
                    }
                    field.set(target, value);
                }
                
            } catch (Exception ex) {
                Oritech.LOGGER.warn("failed to decode field: {}", field.getName(), ex);
            }
        }
    }
    
    private static @NotNull List<Field> getCachedFields(Object target, SyncType type) {
        var key = target.getClass().hashCode() + type.hashCode();
        return CACHED_FIELDS.computeIfAbsent(key, elem -> getSyncFields(target, type));
    }
    
    private static @NotNull List<Field> getSyncFields(Object target, SyncType type) {
        var fields = new ArrayList<>(Arrays.asList(target.getClass().getDeclaredFields()));
        var superClass = target.getClass().getSuperclass();
        while (superClass != null) {
            fields.addAll(Arrays.asList(superClass.getDeclaredFields()));
            superClass = superClass.getSuperclass();
        }
        
        var filteredFields = new ArrayList<Field>();
        fields.stream().filter(field -> hasSyncType(field.getAnnotation(SyncField.class), type)).forEachOrdered(field -> {
            field.setAccessible(true);
            filteredFields.add(field);
        });
        
        if (target instanceof AdditionalNetworkingProvider additionalNetworkingProvider) {
            var addedFields = additionalNetworkingProvider.additionalSyncedFields(type);
            addedFields.forEach(field -> {
                field.setAccessible(true);
                filteredFields.add(field);
            });
        }
        
        return filteredFields;
    }
    
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static class_9139 getAutoCodec(Class<?> type) {
        
        // try to create codec for records
        if (!AUTO_CODECS.containsKey(type)) {
            if (type.isRecord()) {
                Oritech.LOGGER.debug("creating reflective codec for: " + type);
                var computedCodec = ReflectiveCodecBuilder.create((Class<? extends Record>) type);
                AUTO_CODECS.put(type, computedCodec);
                return computedCodec;
            } else if (type.isEnum()) {
                Oritech.LOGGER.debug("creating reflective enum codec for: " + type);
                var computedCodec = ReflectiveCodecBuilder.createForEnum((Class<? extends Enum>) type);
                AUTO_CODECS.put(type, computedCodec);
                return computedCodec;
            }
        }
        
        if (!AUTO_CODECS.containsKey(type)) {
            Oritech.LOGGER.error("No codec defined for: {}", type);
        }
        
        return AUTO_CODECS.get(type);
    }
    
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static class_9139 getAutoCodec(Field field) {
        var listType = getListType(field.getGenericType());
        if (listType.isPresent()) {
            var listTypeCodec = getAutoCodec((Class<?>) listType.get());
            return listTypeCodec.method_56433(class_9135.method_56363());
        }
        var setType = getSetType(field.getGenericType());
        if (setType.isPresent()) {
            var setTypeCodec = getAutoCodec((Class<?>) setType.get());
            return setTypeCodec.method_56433(toSet());
        }
        var mapType = getMapType(field.getGenericType());
        if (mapType.isPresent()) {
            var keyCodec = getAutoCodec((Class<?>) mapType.get().method_15442());
            var valueCodec = getAutoCodec((Class<?>) mapType.get().method_15441());
            
            if (keyCodec == null)
                Oritech.LOGGER.error("Unable to get codec for map key type: {}", field.getType());
            if (valueCodec == null)
                Oritech.LOGGER.error("Unable to get codec for map value type: {}", field.getType());
            
            return class_9135.method_56377(HashMap::new, keyCodec, valueCodec);
        }
        
        return getAutoCodec(field.getType());
    }
    
    // Method for checking if a given type is a List and for retrieving its type parameter
    public static Optional<Type> getListType(Type type) {
        if (type instanceof ParameterizedType pType) {
            var rawType = (Class<?>) pType.getRawType();
            if (rawType instanceof Class && List.class.isAssignableFrom(rawType)) {
                return Optional.of(pType.getActualTypeArguments()[0]);
            }
        }
        return Optional.empty();
    }
    
    // Method for checking if a given type is a Set and for retrieving its type parameter
    public static Optional<Type> getSetType(Type type) {
        if (type instanceof ParameterizedType pType) {
            var rawType = (Class<?>) pType.getRawType();
            if (rawType instanceof Class && Set.class.isAssignableFrom(rawType)) {
                return Optional.of(pType.getActualTypeArguments()[0]);
            }
        }
        return Optional.empty();
    }
    
    // Method for checking if a given type is a Map and for retrieving its type parameters
    public static Optional<class_3545<Type, Type>> getMapType(Type type) {
        if (type instanceof ParameterizedType pType) {
            var rawType = (Class<?>) pType.getRawType();
            if (rawType instanceof Class && Map.class.isAssignableFrom(rawType)) {
                var typeArgs = pType.getActualTypeArguments();
                return Optional.of(new class_3545<>(typeArgs[0], typeArgs[1]));
            }
        }
        return Optional.empty();
    }
    
    private static boolean hasSyncType(SyncField annotation, SyncType type) {
        if (annotation == null) return false;
        
        for (var value : annotation.value()) {
            if (value.equals(type)) return true;
        }
        return false;
    }
    
    public record MessagePayload(class_2338 pos, class_2960 targetEntityType, SyncType syncType,
                                 byte[] message) implements class_8710 {
        @Override
        public net.minecraft.class_8710.class_9154<? extends class_8710> method_56479() {
            return GENERIC_PACKET_ID;
        }
        
        public static final class_8710.class_9154<MessagePayload> GENERIC_PACKET_ID = new class_8710.class_9154<>(Oritech.id("generic"));
        
        public static final class_9139<class_9129, MessagePayload> PACKET_CODEC = new class_9139<>() {
            @Override
            public MessagePayload decode(class_9129 buf) {
                return new MessagePayload(class_2338.field_48404.decode(buf), class_2960.field_48267.decode(buf), SyncType.PACKET_CODEC.decode(buf), class_9135.field_48987.decode(buf));
            }
            
            @Override
            public void encode(class_9129 buf, MessagePayload value) {
                class_2338.field_48404.encode(buf, value.pos);
                class_2960.field_48267.encode(buf, value.targetEntityType);
                SyncType.PACKET_CODEC.encode(buf, value.syncType);
                class_9135.field_48987.encode(buf, value.message);
            }
        };
    }
    
    static <B extends ByteBuf, V> class_9139.class_9140<B, V, Set<V>> toSet() {
        return (codec) -> class_9135.method_56376(HashSet::new, codec);
    }
    
    // transmits only the block type, with the default block state. Custom properties are not sent.
    public static class_9139<class_9129, class_2680> SIMPLE_BLOCK_STATE_PACKET_CODEC = new class_9139<>() {
        @Override
        public class_2680 decode(class_9129 buf) {
            return class_7923.field_41175.method_10223(class_2960.field_48267.decode(buf)).method_9564();
        }
        
        @Override
        public void encode(class_9129 buf, class_2680 value) {
            class_2960.field_48267.encode(buf, class_7923.field_41175.method_10221(value.method_26204()));
        }
    };
    
    public static class_9139<class_9129, Vector2i> VEC2I_PACKED_CODEC = class_9139.method_56435(
      class_9135.field_49675, Vector2i::x,
      class_9135.field_49675, Vector2i::y,
      Vector2i::new
    );
    
    @SuppressWarnings("unchecked")
    public static <K, V> class_9139<class_9129, HashMap<K, V>> createMapCodec(Class<K> keyType, Class<V> valueType) {
        return class_9135.method_56377(HashMap::new, getAutoCodec(keyType), getAutoCodec(valueType));
    }
    
    public static class_9139<class_9129, class_243> VEC3D_PACKET_CODEC = new class_9139<>() {
        @Override
        public class_243 decode(class_9129 buf) {
            var x = buf.readDouble();
            var y = buf.readDouble();
            var z = buf.readDouble();
            return new class_243(x, y, z);
        }
        
        @Override
        public void encode(class_9129 buf, class_243 value) {
            buf.method_52940(value.field_1352);
            buf.method_52940(value.field_1351);
            buf.method_52940(value.field_1350);
        }
    };
    
}
