package org.gtreimagined.tesseract.api.eu;

import net.minecraft.core.BlockPos;
import net.minecraft.world.level.Explosion;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.Level.ExplosionInteraction;
import org.gtreimagined.tesseract.api.IConnectable;
import org.gtreimagined.tesseract.util.Utils;
import org.gtreimagined.tesseract.graph.IElement;

/**
 * An electric cable is the unit of interaction with electric inventories.
 */
public interface IEUCable extends IElement<IEUCable, IEUNode, EURoutingInfo, EUNetwork, EUGrid>, IConnectable, IEUEvent {

    /**
     * Returns the energy that this electrical component will permit to lost through or be received in a single tick.
     *
     * @return A positive double representing the loss energy per block, zero or negative indicates that this component doesn't have a loss. Note: partial amounts will round up to the nearest int when it reaches its destination
     */
    double getLoss();

    /**
     * Returns the maximum amount of packets that this electrical component will permit to pass through or be received in a single tick.
     *
     * @return A positive integer representing the maximum packets, zero or negative indicates that this component accepts no energy.
     */
    int getAmps();

    /**
     * Returns the maximum energy that this electrical component will permit to pass through or be received in a single packet.
     *
     * @return A positive integer representing the maximum accepted energy, zero or negative indicates that this component accepts no energy.
     */
    long getVoltage();

    /**
     *
     * @return if the cable is insulated
     */
    boolean insulated();

    /**
     * @param voltage  The current voltage.
     * @param amperage The current amperage.
     * @return Checks that the cable is able to handle single packet.
     */
    default EUStatus getHandler(long voltage, long amperage) {
        if (getVoltage() < voltage) return EUStatus.FAIL_VOLTAGE;
        else if (getAmps() < amperage) return EUStatus.FAIL_AMPERAGE;
        return EUStatus.SUCCESS;
    }

    long getHolder();

    void setHolder(long holder);

    @Override
    default void onNodeOverVoltage(Level w, long pos, long voltage) {
        Utils.createExplosion(w, BlockPos.of(pos), 4.0F, ExplosionInteraction.BLOCK);
    }

    @Override
    default void onCableOverAmperage(Level w, long pos, long amperage) {
        Utils.createFireAround(w, BlockPos.of(pos));
    }

    @Override
    default void onCableOverVoltage(Level w, long pos, long voltage) {
        Utils.createFireAround(w, BlockPos.of(pos));
    }
}

