package org.gtreimagined.gtlib.recipe.ingredient;

import com.google.gson.JsonObject;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import lombok.Getter;
import net.minecraft.core.registries.BuiltInRegistries;
import org.gtreimagined.gtlib.capability.machine.MachineFluidHandler;
import org.gtreimagined.gtlib.material.Material;
import org.gtreimagined.gtlib.recipe.RecipeUtil;
import org.gtreimagined.gtlib.util.TagUtils;
import org.gtreimagined.gtlib.util.Utils;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.tags.TagKey;
import net.minecraft.world.level.material.Fluid;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.capability.IFluidHandler;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import static net.minecraftforge.fluids.capability.IFluidHandler.FluidAction.EXECUTE;
import static net.minecraftforge.fluids.capability.IFluidHandler.FluidAction.SIMULATE;

public class FluidIngredient {
    private FluidStack[] stacks = new FluidStack[0];
    @Getter
    private TagKey<Fluid> tag;
    @Getter
    private int amount = 0;
    private boolean evaluated = false;

    public static final FluidIngredient EMPTY = new FluidIngredient();

    private FluidIngredient() {

    }

    public boolean matches(FluidStack fluidHolder){
        List<FluidStack> list = Arrays.stream(getStacks()).filter(f -> f.isFluidEqual(fluidHolder)).toList();
        return !list.isEmpty();
    }


    public FluidStack[] getStacks() {
        if (evaluated) return stacks;
        evaluated = true;
        if (tag != null) {
            List<FluidStack> list = new ObjectArrayList<>();
            BuiltInRegistries.FLUID.getTagOrEmpty(tag).iterator().forEachRemaining(t -> {
                if (!t.value().isSource(t.value().defaultFluidState())) return;
                FluidStack stack = new FluidStack(t.value(), getAmount());
                list.add(stack);
            });
            this.stacks = list.toArray(new FluidStack[0]);
        }
        return stacks;
    }

    public FluidIngredient copy(int amount) {
        FluidIngredient ing = new FluidIngredient();
        ing.stacks = Arrays.stream(stacks).map(t -> Utils.ca(amount, t)).toArray(FluidStack[]::new);
        ing.evaluated = this.evaluated;
        ing.amount = amount;
        ing.tag = this.tag;
        return ing;
    }

    public void write(FriendlyByteBuf buffer) {
        getStacks();
        buffer.writeVarInt(stacks.length);
        for (FluidStack stack : this.stacks) {
            buffer.writeFluidStack(stack);
        }
    }

    public JsonObject toJson(){
        JsonObject json = new JsonObject();
        if (tag != null){
            json.addProperty("fluidTag", true);
            json.addProperty("tag", tag.location().toString());
            json.addProperty("amount", amount);
        } else {
            json = RecipeUtil.fluidstackToJson(stacks[0]);
        }
        return json;
    }

    public static FluidIngredient read(FriendlyByteBuf buf) {
        int count = buf.readVarInt();
        FluidStack[] stacks = new FluidStack[count];
        for (int i = 0; i < count; i++) {
            stacks[i] = buf.readFluidStack();
        }
        FluidIngredient ing = new FluidIngredient();
        int amount = 0;
        for (FluidStack stack : stacks) {
            if (stack.getAmount() > amount){
                amount = stack.getAmount();
            }
        }
        ing.stacks = stacks;
        ing.evaluated = true;
        ing.amount = amount;
        return ing;
    }

    public static FluidIngredient of(ResourceLocation loc, int amount) {
        Objects.requireNonNull(loc);
        FluidIngredient ing = new FluidIngredient();
        ing.tag = TagUtils.getFluidTag(loc);
        ing.amount = amount;
        return ing;
    }

    public static FluidIngredient of(TagKey<Fluid> tag, int amount) {
        Objects.requireNonNull(tag);
        FluidIngredient ing = new FluidIngredient();
        ing.tag = tag;
        ing.amount = amount;
        return ing;
    }

    public static FluidIngredient of(Material mat, int amount) {
        return of(new ResourceLocation("forge", mat.getId()), amount);
    }


    public static FluidIngredient of(FluidStack stack) {
        Objects.requireNonNull(stack);
        FluidIngredient ing = new FluidIngredient();
        ing.stacks = new FluidStack[]{stack};
        ing.amount = stack.getAmount();
        return ing;
    }

    public List<FluidStack> drain(MachineFluidHandler<?> handler, boolean input, boolean simulate) {
        return drain(amount, handler, input, simulate);
    }

    public List<FluidStack> drain(int amount, MachineFluidHandler<?> handler, boolean input, boolean simulate) {
        int drained = amount;
        List<FluidStack> ret = new ObjectArrayList<>(1);
        for (FluidStack stack : getStacks()) {
            stack = stack.copy();
            stack.setAmount(drained);
            FluidStack drain = input ? handler.drainInput(stack, simulate ? SIMULATE : EXECUTE) : handler.drain(stack, simulate ? SIMULATE : EXECUTE);
            drained -= drain.getAmount();
            if (!drain.isEmpty()) {
                ret.add(drain);
            }
            if (drained == 0) break;
        }
        return ret;
    }

    public List<FluidStack> drain(IFluidHandler handler, boolean simulate) {
        return drain(amount, handler, simulate);
    }

    public List<FluidStack> drain(int amount, IFluidHandler handler, boolean simulate) {
        int drained = amount;
        List<FluidStack> ret = new ObjectArrayList<>(1);
        for (FluidStack stack : getStacks()) {
            stack = stack.copy();
            stack.setAmount(drained);
            FluidStack drain = handler.drain(stack, simulate ? SIMULATE : EXECUTE);
            drained -= drain.getAmount();
            if (!drain.isEmpty()) {
                ret.add(drain);
            }
            if (drained == 0) break;
        }
        return ret;
    }

    public int drainedAmount(int amount, MachineFluidHandler<?> handler, boolean input, boolean simulate) {
        return drain(amount, handler, input, simulate).stream().mapToInt(FluidStack::getAmount).sum();
    }
}
