package org.gtreimagined.gtlib.structure;

import org.gtreimagined.gtlib.GTLib;
import org.gtreimagined.gtlib.blockentity.multi.BlockEntityBasicMultiMachine;
import org.gtreimagined.gtlib.util.Dir;
import org.gtreimagined.gtlib.util.int3;
import net.minecraft.core.Direction;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;

public class StructureHandle<T extends BlockEntityBasicMultiMachine<T>> {
    private final boolean debug = false;
    private final BlockEntityBasicMultiMachine<?> source;
    private final List<int3> offsets;
    private final Consumer<T> onRemoval;
    private final Consumer<T> onAdd;
    private final Class<T> clazz;
    @Nullable
    private T object;

    public StructureHandle(Class<T> clazz, BlockEntityBasicMultiMachine<?> tile, int3 off, @Nullable Consumer<T> onRemoval, @Nullable Consumer<T> onAdd) {
        this(clazz, tile, Collections.singletonList(off), onRemoval, onAdd);
    }

    public StructureHandle(Class<T> clazz, BlockEntityBasicMultiMachine<?> tile, List<int3> off, @Nullable Consumer<T> onRemoval, @Nullable Consumer<T> onAdd) {
        this.source = tile;
        this.offsets = off;
        this.onRemoval = onRemoval;
        this.onAdd = onAdd;
        this.clazz = clazz;
        tile.addStructureHandle(this);
    }

    public BlockEntityBasicMultiMachine<?> getSource() {
        return source;
    }

    public void register() {
        BlockState state = source.getBlockState();
        boolean vertical = source.getMachineType().isVerticalFacingAllowed();
        Direction facing = vertical ? state.getValue(BlockStateProperties.FACING) : state.getValue(BlockStateProperties.HORIZONTAL_FACING);
        int3 newOff = new int3(facing);
        for (int3 offset : offsets) {
            newOff.set(source.getBlockPos()).offset(offset, Dir.RIGHT, Dir.UP, Dir.FORWARD);
            StructureCache.addListener(this, source.getLevel(), newOff);
        }
    }

    public void deregister() {
        BlockState state = source.getBlockState();
        boolean vertical = source.getMachineType().isVerticalFacingAllowed();
        Direction facing = vertical ? state.getValue(BlockStateProperties.FACING) : state.getValue(BlockStateProperties.HORIZONTAL_FACING);
        int3 newOff = new int3(facing);
        for (int3 offset : offsets) {
            newOff.set(source.getBlockPos()).offset(offset, Dir.RIGHT, Dir.UP, Dir.FORWARD);
            StructureCache.removeListener(this, source.getLevel(), newOff);
        }
    }

    public void structureCacheRemoval() {
        if (debug) GTLib.LOGGER.debug("removed structure handle");
        T obj = this.object;
        this.object = null;
        if (onRemoval != null && obj != null) onRemoval.accept(obj);
    }

    public void structureCacheAddition(BlockEntity t) {
        if (!clazz.isInstance(t)) return;
        if (debug) GTLib.LOGGER.debug("added to structure handle");
        this.object = (T) t;
        if (onAdd != null) onAdd.accept(this.object);
    }

    @Nullable
    public T get() {
        return object;
    }
}
