package org.gtreimagined.gtlib.gui.container;

import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import lombok.Getter;
import org.gtreimagined.gtlib.capability.IGuiHandler;
import org.gtreimagined.gtlib.capability.item.TrackedItemHandler;
import org.gtreimagined.gtlib.gui.GuiInstance;
import org.gtreimagined.gtlib.gui.slot.AbstractSlot;
import org.gtreimagined.gtlib.gui.slot.IClickableSlot;
import org.gtreimagined.gtlib.gui.slot.SlotFake;
import net.minecraft.CrashReport;
import net.minecraft.CrashReportCategory;
import net.minecraft.ReportedException;
import net.minecraft.core.NonNullList;
import net.minecraft.core.Registry;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.inventory.AbstractContainerMenu;
import net.minecraft.world.inventory.ClickType;
import net.minecraft.world.inventory.MenuType;
import net.minecraft.world.inventory.Slot;
import net.minecraft.world.item.ItemStack;
import net.minecraftforge.items.IItemHandler;

import java.util.Set;

public abstract class GTContainer extends AbstractContainerMenu implements IGTContainer {

    @Getter
    protected Inventory playerInv;
    protected int invSize;
    public final GuiInstance handler;
    public final Set<ServerPlayer> listeners = new ObjectOpenHashSet<>();

    public final NonNullList<Slot> playerSlots = NonNullList.create();
    private final MenuType<?> containerType;

    public GTContainer(IGuiHandler handler, MenuType<?> containerType, int windowId, Inventory playerInv, int invSize) {
        super(containerType, windowId);
        this.playerInv = playerInv;
        this.invSize = invSize;
        this.handler = new GuiInstance(handler, this, handler.isRemote());
        this.containerType = containerType;
    }

    @Override
    public Set<ServerPlayer> listeners() {
        return listeners;
    }

    protected void addPlayerSlots() {
        if (playerInv == null) return;
        for (int y = 0; y < 3; ++y) { //Inventory Slots
            for (int x = 0; x < 9; ++x) {
                Slot slot = new Slot(playerInv, x + y * 9 + 9, getXPlayerOffset() +8 + (x * 18), getYPlayerOffset() + 84 + (y * 18));
                this.addSlot(slot);
                playerSlots.add(slot);
            }
        }
        for (int x = 0; x < 9; ++x) { //HotBar Slots
            Slot slot = new Slot(playerInv, x, getXPlayerOffset() + 8 + (x * 18), getYPlayerOffset() + 142);
            this.addSlot(slot);
            playerSlots.add(slot);
        }
    }

    protected int getXPlayerOffset(){
        return 0;
    }

    protected int getYPlayerOffset(){
        return 0;
    }

    @Override
    public void broadcastChanges() {
        super.broadcastChanges();
        //NOTE: event to add player listener is fired after first sync, so check if the player exists.
        if (listeners().size() == 0) return;
        source().update();
    }

    @Override
    public void clicked(int slotId, int dragType, ClickType clickTypeIn, Player player) {
        if (slotId >= 0 && this.getSlot(slotId) instanceof IClickableSlot) {
            try {
                ((IClickableSlot) this.getSlot(slotId)).clickSlot(dragType, clickTypeIn, player, this);
                return;
            } catch (Exception exception) {
                CrashReport crashreport = CrashReport.forThrowable(exception, "Container click");
                CrashReportCategory crashreportcategory = crashreport.addCategory("Click info");
                crashreportcategory.setDetail("Menu Type", () -> {
                    return this.containerType != null ? Registry.MENU.getKey(this.containerType).toString() : "<no type>";
                });
                crashreportcategory.setDetail("Menu Class", () -> {
                    return this.getClass().getCanonicalName();
                });
                crashreportcategory.setDetail("Slot Count", this.slots.size());
                crashreportcategory.setDetail("Slot", slotId);
                crashreportcategory.setDetail("Button", dragType);
                crashreportcategory.setDetail("Type", clickTypeIn);
                throw new ReportedException(crashreport);
            }
        }
        super.clicked(slotId, dragType, clickTypeIn, player);
    }

    @Override
    public ItemStack quickMoveStack(Player player, int index) {
        ItemStack itemstack = ItemStack.EMPTY;
        Slot slot = this.slots.get(index);

        if (slot != null && slot.hasItem()) {
            ItemStack slotStack = slot.getItem();
            itemstack = slotStack.copy();

            if (index < invSize) {
                if (!this.moveItemStackTo(slotStack, invSize, this.slots.size(), true)) {
                    return ItemStack.EMPTY;
                }
            } else if (!this.moveItemStackTo(slotStack, 0, invSize, false)) {
                return ItemStack.EMPTY;
            }

            if (slotStack.getCount() == 0) {
                slot.set(ItemStack.EMPTY);
            } else {
                slot.setChanged();
            }
        }

        return itemstack;
    }

    //Because top level doesn't verify anything. Just a 1-1 copy but adds slot.isItemValid.
    @Override
    protected boolean moveItemStackTo(ItemStack stack, int startIndex, int endIndex, boolean reverseDirection) {
        boolean flag = false;
        int i = startIndex;
        if (reverseDirection) {
            i = endIndex - 1;
        }

        if (stack.isStackable()) {
            while (!stack.isEmpty()) {
                if (reverseDirection) {
                    if (i < startIndex) {
                        break;
                    }
                } else if (i >= endIndex) {
                    break;
                }

                Slot slot = this.slots.get(i);
                boolean continueLoop = false;
                if (slot instanceof SlotFake || !slot.mayPlace(stack)) {
                    continueLoop = true;
                }
                ItemStack itemstack = slot.getItem();
                if (!continueLoop && !itemstack.isEmpty() && ItemStack.isSameItemSameTags(stack, itemstack)) {
                    int j = itemstack.getCount() + stack.getCount();
                    int maxSize = Math.min(slot.getMaxStackSize(), stack.getMaxStackSize());
                    if (j <= maxSize) {
                        stack.setCount(0);
                        itemstack.setCount(j);
                        slot.setChanged();
                        if (slot instanceof AbstractSlot<?> abstractSlot) {
                            IItemHandler handle = abstractSlot.getContainer();
                            if (handle instanceof TrackedItemHandler<?> trackedItemHandler) {
                                trackedItemHandler.onContentsChanged(slot.index);
                            }
                        }
                        flag = true;
                    } else if (itemstack.getCount() < maxSize) {
                        stack.shrink(maxSize - itemstack.getCount());
                        itemstack.setCount(maxSize);
                        slot.setChanged();
                        flag = true;
                    }
                }

                if (reverseDirection) {
                    --i;
                } else {
                    ++i;
                }
            }
        }

        if (!stack.isEmpty()) {
            if (reverseDirection) {
                i = endIndex - 1;
            } else {
                i = startIndex;
            }

            while (true) {
                if (reverseDirection) {
                    if (i < startIndex) {
                        break;
                    }
                } else if (i >= endIndex) {
                    break;
                }

                Slot slot1 = this.slots.get(i);
                boolean continueLoop = false;
                if (slot1 instanceof SlotFake) {
                    continueLoop = true;
                }
                ItemStack itemstack1 = slot1.getItem();
                if (!continueLoop && itemstack1.isEmpty() && slot1.mayPlace(stack)) {
                    if (stack.getCount() > slot1.getMaxStackSize()) {
                        slot1.set(stack.split(slot1.getMaxStackSize()));
                    } else {
                        slot1.set(stack.split(stack.getCount()));
                    }

                    slot1.setChanged();
                    flag = true;
                    break;
                }

                if (reverseDirection) {
                    --i;
                } else {
                    ++i;
                }
            }
        }

        return flag;
    }

    @Override
    public GuiInstance source() {
        return handler;
    }
}
