package noobanidus.mods.lootr.common.impl;

import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import net.minecraft.core.BlockPos;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.entity.RandomizableContainerBlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.storage.loot.LootTable;
import noobanidus.mods.lootr.common.api.ILootrAPI;
import noobanidus.mods.lootr.common.api.ILootrBlockEntityConverter;
import noobanidus.mods.lootr.common.api.ILootrEntityConverter;
import noobanidus.mods.lootr.common.api.data.blockentity.ILootrBlockEntity;
import noobanidus.mods.lootr.common.api.data.entity.ILootrCart;
import noobanidus.mods.lootr.common.api.filter.ILootrFilter;
import noobanidus.mods.lootr.common.api.filter.ILootrFilterProvider;
import noobanidus.mods.lootr.common.api.postprocess.ILootrPostProcessor;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.function.Function;

public class LootrServiceRegistry {
  private static LootrServiceRegistry INSTANCE;

  private final Map<BlockEntityType<?>, Function<?, ?>> blockEntityConverterMap = new Object2ObjectOpenHashMap<>();
  private final Map<EntityType<?>, Function<?, ?>> entityConverterMap = new Object2ObjectOpenHashMap<>();
  private final List<ILootrFilter> filters = new ObjectArrayList<>();
  private final List<ILootrPostProcessor> postProcessors = new ObjectArrayList<>();

  @SuppressWarnings("rawtypes")
  public LootrServiceRegistry () {
    ClassLoader classLoader = ILootrAPI.class.getClassLoader();
    ServiceLoader<ILootrBlockEntityConverter> loader = ServiceLoader.load(ILootrBlockEntityConverter.class, classLoader);

    for (ILootrBlockEntityConverter<?> converter : loader) {
      blockEntityConverterMap.put(converter.getBlockEntityType(), converter);
    }

    ServiceLoader<ILootrEntityConverter> loader2 = ServiceLoader.load(ILootrEntityConverter.class ,classLoader);
    for (ILootrEntityConverter<?> converter2 : loader2) {
      entityConverterMap.put(converter2.getEntityType(), converter2);
    }

    ServiceLoader<ILootrFilterProvider> loader3 = ServiceLoader.load(ILootrFilterProvider.class, classLoader);
    for (ILootrFilterProvider provider : loader3) {
      filters.addAll(provider.getFilters());
    }
    filters.sort(Comparator.comparingInt(ILootrFilter::getPriority));

    ServiceLoader<ILootrPostProcessor> loader4 = ServiceLoader.load(ILootrPostProcessor.class, classLoader);
    for (ILootrPostProcessor processor : loader4) {
      postProcessors.add(processor);
    }
  }

  public static LootrServiceRegistry getInstance () {
    if (INSTANCE == null) {
      INSTANCE = new LootrServiceRegistry();
    }
    return INSTANCE;
  }

  @Nullable
  @SuppressWarnings("unchecked")
  private static <T> Function<T, ILootrBlockEntity> getBlockEntity(BlockEntityType<?> clazz) {
    return (Function<T, ILootrBlockEntity>) getInstance().blockEntityConverterMap.get(clazz);
  }

  @SuppressWarnings("unchecked")
  @Nullable
  private static <T> Function<T, ILootrCart> getEntity(EntityType<?> clazz) {
    return (Function<T, ILootrCart>) getInstance().entityConverterMap.get(clazz);
  }

  @Nullable
  public static <T extends BlockEntity> ILootrBlockEntity convertBlockEntity(T blockEntity) {
    if (blockEntity == null) {
      return null;
    }
    Function<T, ILootrBlockEntity> converter = getBlockEntity( blockEntity.getType());
    if (converter == null) {
      return null;
    }
    return converter.apply(blockEntity);
  }

  @Nullable
  public static <T extends Entity> ILootrCart convertEntity (T entity) {
    if (entity == null) {
      return null;
    }
    Function<T, ILootrCart> converter = getEntity(entity.getType());
    if (converter == null) {
      return null;
    }
    return converter.apply(entity);
  }

  public static List<ILootrFilter> getFilters () {
    return getInstance().filters;
  }

  public static void postProcess (ServerLevel level, BlockPos position, RandomizableContainerBlockEntity newBlockEntity, BlockState originalState, BlockState newState, ResourceKey<LootTable> lootTable) {
    for (ILootrPostProcessor processor : getInstance().postProcessors) {
      processor.process(level, position, newBlockEntity, originalState, newState, lootTable);
    }
  }
}
