package noobanidus.mods.lootr.common.block.entity;

import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Registry;
import net.minecraft.core.registries.Registries;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.entity.RandomizableContainerBlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.chunk.ChunkSource;
import net.minecraft.world.level.levelgen.structure.Structure;
import net.minecraft.world.level.storage.loot.LootTable;
import noobanidus.mods.lootr.common.api.DataToCopy;
import noobanidus.mods.lootr.common.api.LootrAPI;
import noobanidus.mods.lootr.common.api.LootrTags;
import noobanidus.mods.lootr.common.api.PlatformAPI;
import noobanidus.mods.lootr.common.api.data.blockentity.ILootrBlockEntity;
import noobanidus.mods.lootr.common.chunk.LoadedChunks;
import noobanidus.mods.lootr.common.impl.LootrServiceRegistry;
import org.jetbrains.annotations.Nullable;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public final class BlockEntityTicker {
  private static final Map<ResourceKey<Level>, BlockEntityTicker> TICKERS = new Object2ObjectOpenHashMap<>();

  private final ResourceKey<Level> levelKey;
  private final Map<ChunkPos, Entry> blockEntityEntries = new Object2ObjectOpenHashMap<>();
  private final Map<ChunkPos, Entry> pendingEntries = new Object2ObjectOpenHashMap<>();

  private BlockEntityTicker(ResourceKey<Level> levelKey) {
    this.levelKey = levelKey;
  }

  public static void addEntity(BlockEntity entity, Level level, ChunkPos chunkPos) {
    if (LootrAPI.isDisabled()) {
      return;
    }
    ResourceKey<Level> dimension = getServerDimensionIfValid(level);
    if (dimension == null) {
      return;
    }

    BlockEntityTicker ticker;
    synchronized (TICKERS) {
      ticker = TICKERS.computeIfAbsent(dimension, BlockEntityTicker::new);
    }

    ticker.addEntity(level, entity, chunkPos);
  }

  private void addEntity(Level level, BlockEntity entity, ChunkPos chunkPos) {
    if (!LootrAPI.isWorldBorderSafe(level, chunkPos)) {
      return;
    }

    if (!isValidEntity(entity)) {
      return;
    }

    synchronized (pendingEntries) {
      Entry previousEntry = pendingEntries.get(chunkPos);
      if (previousEntry != null) {
        previousEntry.entityPositions.add(entity.getBlockPos());
      } else {
        HashSet<BlockPos> entityPositions = new HashSet<>();
        entityPositions.add(entity.getBlockPos());
        Entry entry = new Entry(chunkPos, entityPositions);
        pendingEntries.put(chunkPos, entry);
      }
    }
  }

  private static boolean isValidEntity(BlockEntity entity) {
    if (!(entity instanceof RandomizableContainerBlockEntity validEntity)) {
      return false;
    }
    if (LootrAPI.resolveBlockEntity(validEntity) instanceof ILootrBlockEntity) {
      return false;
    }
    return validEntity.getLootTable() != null && !LootrAPI.isLootTableBlacklisted(validEntity.getLootTable());
  }

  public static void onServerTick(MinecraftServer server) {
    if (LootrAPI.isDisabled()) {
      return;
    }

    for (BlockEntityTicker ticker : TICKERS.values()) {
      ServerLevel level = server.getLevel(ticker.levelKey);
      if (level == null) {
        continue;
      }
      ticker.onServerLevelTick(level);
    }
  }

  private void onServerLevelTick(ServerLevel level) {
    Set<ChunkPos> loadedChunks = LoadedChunks.getLoadedChunks(level.dimension());
    Iterator<Entry> iterator = blockEntityEntries.values().iterator();
    while (iterator.hasNext()) {
      Entry entry = iterator.next();
      switch (entry.getChunkLoadStatus(level, loadedChunks)) {
        case UNLOADED -> {
          // the chunk has unloaded. this entry is no longer valid, and it will be added again if the chunk loads again.
          iterator.remove();
        }
        case NOT_FULLY_LOADED -> {
          // keep waiting for the chunk to be fully loaded
        }
        case SURROUNDING_CHUNKS_NOT_LOADED -> {
          // keep waiting for the surrounding chunks to load
        }
        case COMPLETE -> {
          replaceEntitiesInChunk(level, entry);
          iterator.remove();
        }
      }
    }

    synchronized (pendingEntries) {
      for (Entry entry : pendingEntries.values()) {
        blockEntityEntries.merge(entry.chunkPos, entry, (entry1, entry2) -> {
          entry1.entityPositions.addAll(entry2.entityPositions);
          return entry1;
        });
      }

      pendingEntries.clear();
    }
  }

  private static boolean checkStructureValidity(ServerLevel level, ChunkPos chunkPos, BlockPos position) {
    if (!level.getServer().getWorldData().worldGenOptions().generateStructures()) {
      return true;
    }
    Registry<Structure> registry = level.registryAccess().registryOrThrow(Registries.STRUCTURE);
    if (registry.getTag(LootrTags.Structure.STRUCTURE_BLACKLIST).filter(tag -> tag.size() != 0).isPresent()) {
      return !LootrAPI.isTaggedStructurePresent(level, chunkPos, LootrTags.Structure.STRUCTURE_BLACKLIST, position);
    } else if (registry.getTag(LootrTags.Structure.STRUCTURE_WHITELIST).filter(tag -> tag.size() != 0).isPresent()) {
      return LootrAPI.isTaggedStructurePresent(level, chunkPos, LootrTags.Structure.STRUCTURE_WHITELIST, position);
    }
    return true;
  }

  private static void replaceEntitiesInChunk(ServerLevel level, Entry entry) {
    for (BlockPos entityPos : entry.entityPositions()) {
      if (!checkStructureValidity(level, entry.chunkPos(), entityPos)) {
          continue;
      }
      BlockEntity blockEntity = level.getBlockEntity(entityPos);
      if (!(blockEntity instanceof RandomizableContainerBlockEntity be) || LootrAPI.resolveBlockEntity(blockEntity) instanceof ILootrBlockEntity) {
          continue;
      }
      ResourceKey<LootTable> table = be.getLootTable();
      if (table == null) {
        LootrAPI.LOG.warn("randomizable container \"{}\" has no loot table in {} ({})", be.getName(), level.dimension(), entityPos);
        continue;
      }
      if (LootrAPI.isLootTableBlacklisted(table)) {
        continue;
      }
      BlockState stateAt = level.getBlockState(entityPos);
      BlockState replacement = LootrAPI.replacementBlockState(stateAt);
      if (replacement == null) {
          continue;
      }

      replaceEntity(level, entityPos, be, stateAt, replacement, table);
    }
  }

  private static void replaceEntity(ServerLevel level, BlockPos entityPos, RandomizableContainerBlockEntity be, BlockState original, BlockState replacement, ResourceKey<LootTable> table) {
    // Save specific data. Currently, this includes the LockCode (all platforms), along with NeoForge's getPersistentData.
    DataToCopy data = PlatformAPI.copySpecificData(be);
    long seed = be.getLootTableSeed();
    // IMPORTANT: Clear loot table to prevent loot drop when container is destroyed
    be.setLootTable(null);
    level.setBlock(entityPos, replacement, Block.UPDATE_CLIENTS);
    BlockEntity newBlockEntity = level.getBlockEntity(entityPos);
    PlatformAPI.restoreSpecificData(data, newBlockEntity);
    if (LootrAPI.resolveBlockEntity(newBlockEntity) instanceof ILootrBlockEntity && newBlockEntity instanceof RandomizableContainerBlockEntity rbe) {
      rbe.setLootTable(table, seed);
      LootrServiceRegistry.postProcess(level, entityPos, rbe, original, replacement, table);
    } else {
      LootrAPI.LOG.error("replacement {} is not an ILootrBlockEntity {} at {}", replacement, level.dimension(), entityPos);
    }
  }

  @Nullable
  private static ResourceKey<Level> getServerDimensionIfValid(Level level) {
    if (LootrAPI.getServer() == null || level.isClientSide()) {
      return null;
    }
    ResourceKey<Level> dimension = level.dimension();
    if (LootrAPI.isDimensionBlocked(dimension)) {
      return null;
    }
    return dimension;
  }

  public record Entry(ChunkPos chunkPos, Set<BlockPos> entityPositions) {
    public ChunkLoadStatus getChunkLoadStatus(ServerLevel level, Set<ChunkPos> loadedChunks) {
      ChunkSource chunkSource = level.getChunkSource();
      if (!LootrAPI.isWorldBorderSafe(level, chunkPos) || !chunkSource.hasChunk(chunkPos.x, chunkPos.z)) {
        return ChunkLoadStatus.UNLOADED;
      }
      if (!loadedChunks.contains(chunkPos)) {
        return ChunkLoadStatus.NOT_FULLY_LOADED;
      }

      for (int x = chunkPos.x - 2; x <= chunkPos.x + 2; x++) {
        for (int z = chunkPos.z - 2; z <= chunkPos.z + 2; z++) {
          if (x == chunkPos.x && z == chunkPos.z) {
            // this case is already checked above
            continue;
          }
          ChunkPos pos = new ChunkPos(x, z);
          // This has the potential to force-load chunks on the main thread
          // by ignoring the loading state of chunks outside the world border.
          if (!LootrAPI.isWorldBorderSafe(level, pos)) {
            continue;
          }
          if (!loadedChunks.contains(pos)) {
            return ChunkLoadStatus.SURROUNDING_CHUNKS_NOT_LOADED;
          }
        }
      }
      return ChunkLoadStatus.COMPLETE;
    }
  }

  public enum ChunkLoadStatus {
    UNLOADED,
    SURROUNDING_CHUNKS_NOT_LOADED,
    NOT_FULLY_LOADED,
    COMPLETE
  }
}
