package commonnetwork.networking;

import commonnetwork.Constants;
import commonnetwork.networking.data.CommonPacketWrapper;
import commonnetwork.networking.data.PacketContainer;
import commonnetwork.networking.data.PacketContext;
import commonnetwork.networking.data.Side;
import commonnetwork.networking.exceptions.RegistrationException;
import net.minecraft.client.Minecraft;
import net.minecraft.network.Connection;
import net.minecraft.network.protocol.common.ClientboundCustomPayloadPacket;
import net.minecraft.network.protocol.common.ServerboundCustomPayloadPacket;
import net.minecraft.server.level.ServerPlayer;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.LogicalSide;
import net.neoforged.neoforge.network.event.RegisterPayloadHandlersEvent;
import net.neoforged.neoforge.network.handling.IPayloadHandler;
import org.jetbrains.annotations.Nullable;

import java.util.function.Consumer;


public class NeoForgeNetworkHandler extends PacketRegistrationHandler
{


    public NeoForgeNetworkHandler(Side side)
    {
        super(side);
    }

    @Override
    <T> void registerPacket(PacketContainer<T> container)
    {
        // not needed for neoforge
    }

    @SubscribeEvent
    @SuppressWarnings("unchecked")
    public void register(final RegisterPayloadHandlersEvent event)
    {
        if (!PACKET_MAP.isEmpty())
        {
            PACKET_MAP.forEach((type, container) -> {
                var registrar = event.registrar(container.getType().id().getNamespace()).optional();
                if (container.packetType() == PacketContainer.PacketType.PLAY)
                {
                    registrar.playBidirectional(container.getType(), container.getCodec(), buildHandler(container.handler()));
                }
                else
                {
                    registrar.configurationBidirectional(container.getType(), container.getCodec(), buildHandler(container.handler()));
                }
            });
        }
    }

    @SuppressWarnings("unchecked")
    public <T> void sendToServer(T packet, boolean ignoreCheck)
    {
        PacketContainer<T> container = (PacketContainer<T>) PACKET_MAP.get(packet.getClass());
        if (container != null)
        {
            if (ignoreCheck || Minecraft.getInstance().getConnection().hasChannel(container.getType()))
            {
                Minecraft.getInstance().getConnection().getConnection()
                        .send(new ServerboundCustomPayloadPacket(new CommonPacketWrapper<>(container, packet)));
            }
        }
        else
        {
            throw new RegistrationException(packet.getClass() + "{} packet not registered on the client, packets need to be registered on both sides!");
        }
    }

    @Override
    public <T> void send(T packet, Connection connection)
    {
        PacketContainer<T> container = (PacketContainer<T>) PACKET_MAP.get(packet.getClass());
        if (container != null)
        {
            if (this.side == Side.SERVER)
            {
                connection.send(new ClientboundCustomPayloadPacket(new CommonPacketWrapper<>(container, packet)));
            }
            else if (this.side == Side.CLIENT)
            {
                connection.send(new ServerboundCustomPayloadPacket(new CommonPacketWrapper<>(container, packet)));
            }
        }
    }

	 @Override
	 public <T> @Nullable ClientboundCustomPayloadPacket getRawClientboundPacket(T packet){
		 PacketContainer<T> container = (PacketContainer<T>) PACKET_MAP.get(packet.getClass());
		 if (container != null) {
			 return new ClientboundCustomPayloadPacket(new CommonPacketWrapper<>(container, packet));
		 }
		 return null;
	 }

	 @Override
	 public <T> @Nullable ServerboundCustomPayloadPacket getRawServerboundPacket(T packet){
		 PacketContainer<T> container = (PacketContainer<T>) PACKET_MAP.get(packet.getClass());
		 if (container != null) {
			 return new ServerboundCustomPayloadPacket(new CommonPacketWrapper<>(container, packet));
		 }
		 return null;
	 }

    @SuppressWarnings("unchecked")
    public <T> void sendToClient(T packet, ServerPlayer player, boolean ignoreCheck)
    {
        PacketContainer<T> container = (PacketContainer<T>) PACKET_MAP.get(packet.getClass());
        if (container != null)
        {
            if (ignoreCheck || player.connection.hasChannel(container.type()))
            {
                player.connection.getConnection().send(new ClientboundCustomPayloadPacket(new CommonPacketWrapper<>(container, packet)));
            }
        }
        else
        {
            throw new RegistrationException(packet.getClass() + "{} packet not registered on the server, packets need to be registered on both sides!");
        }
    }

    private <T, K extends
            CommonPacketWrapper<T>> IPayloadHandler<K> buildHandler(Consumer<PacketContext<T>> handler)
    {
        return (payload, ctx) -> {
            try
            {
                Side side = ctx.flow().getReceptionSide().equals(LogicalSide.SERVER) ? Side.SERVER : Side.CLIENT;
                if (Side.SERVER.equals(side))
                {
                    handler.accept(new PacketContext<>((ServerPlayer) ctx.player(), payload.packet(), side));
                }
                else
                {
                    handler.accept(new PacketContext<>(payload.packet(), side));
                }

            }
            catch (Throwable t)
            {
                Constants.LOG.error("Error handling packet: {} -> ", payload.packet().getClass(), t);
            }
        };
    }
}
