package mezz.jei.fabric.startup;

import com.mojang.blaze3d.platform.Window;
import mezz.jei.fabric.events.JeiCharTypedEvents;
import mezz.jei.fabric.events.JeiScreenEvents;
import mezz.jei.gui.events.GuiEventHandler;
import mezz.jei.gui.input.ClientInputHandler;
import mezz.jei.gui.input.InputType;
import mezz.jei.gui.input.UserInput;
import mezz.jei.gui.startup.JeiEventHandlers;
import net.fabricmc.fabric.api.client.screen.v1.ScreenEvents;
import net.fabricmc.fabric.api.client.screen.v1.ScreenKeyboardEvents;
import net.fabricmc.fabric.api.client.screen.v1.ScreenMouseEvents;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.client.gui.screens.inventory.AbstractContainerScreen;
import net.minecraft.client.input.CharacterEvent;
import net.minecraft.client.input.KeyEvent;
import net.minecraft.client.input.MouseButtonEvent;
import org.jspecify.annotations.Nullable;

public class EventRegistration {
	@Nullable
	private ClientInputHandler clientInputHandler;
	@Nullable
	private GuiEventHandler guiEventHandler;
	private boolean registered;

	public void setEventHandlers(JeiEventHandlers eventHandlers) {
		clientInputHandler = eventHandlers.clientInputHandler();
		guiEventHandler = eventHandlers.guiEventHandler();
		if (!registered) {
			registerEvents();
			registered = true;
		}
	}

	private void registerEvents() {
		ScreenEvents.BEFORE_INIT.register((client, screen, scaledWidth, scaledHeight) ->
			registerScreenEvents(screen)
		);
		JeiCharTypedEvents.BEFORE_CHAR_TYPED.register(this::beforeCharTyped);
		ScreenEvents.AFTER_INIT.register(this::afterInit);
		JeiScreenEvents.DRAW_FOREGROUND.register(this::drawForeground);
		JeiScreenEvents.DRAW_BACKGROUND.register(this::drawBackground);
	}

	private void registerScreenEvents(Screen screen) {
		if (guiEventHandler == null) {
			return;
		}

		ScreenKeyboardEvents.allowKeyPress(screen).register(this::allowKeyPress);
		ScreenMouseEvents.allowMouseClick(screen).register(this::allowMouseClick);
		ScreenMouseEvents.allowMouseRelease(screen).register(this::allowMouseRelease);
		ScreenMouseEvents.allowMouseScroll(screen).register(this::allowMouseScroll);
	}

	private boolean allowMouseClick(Screen screen, MouseButtonEvent event) {
		if (clientInputHandler == null) {
			return true;
		}
		return UserInput.fromVanilla(event, false, InputType.SIMULATE)
			.map(input -> !clientInputHandler.onGuiMouseClicked(screen, input))
			.orElse(true);
	}

	private boolean allowMouseRelease(Screen screen, MouseButtonEvent event) {
		if (clientInputHandler == null) {
			return true;
		}
		return UserInput.fromVanilla(event, false, InputType.EXECUTE)
			.map(input -> !clientInputHandler.onGuiMouseReleased(screen, input))
			.orElse(true);
	}

	private boolean allowKeyPress(Screen screen, KeyEvent keyEvent) {
		if (clientInputHandler == null) {
			return true;
		}
		UserInput userInput = UserInput.fromVanilla(keyEvent, InputType.IMMEDIATE);
		return !clientInputHandler.onKeyboardKeyPressedPre(screen, userInput);
	}

	private boolean allowMouseScroll(Screen screen, double mouseX, double mouseY, double horizontalAmount, double verticalAmount) {
		if (clientInputHandler == null) {
			return false;
		}
		return !clientInputHandler.onGuiMouseScroll(mouseX, mouseY, horizontalAmount, verticalAmount);
	}

	private boolean beforeCharTyped(long windowHandle, CharacterEvent event) {
		Minecraft minecraft = Minecraft.getInstance();
		Window window = minecraft.getWindow();
		if (window.handle() == windowHandle &&
			clientInputHandler != null &&
			minecraft.screen instanceof Screen screen &&
			minecraft.getOverlay() == null
		) {
			return clientInputHandler.onKeyboardCharTypedPre(screen, event);
		}
		return false;
	}

	private void afterInit(Minecraft client, Screen screen, int scaledWidth, int scaledHeight) {
		if (guiEventHandler != null) {
			guiEventHandler.onGuiInit(screen);
			guiEventHandler.onGuiOpen(screen);
		}
	}

	private void drawForeground(AbstractContainerScreen<?> screen, GuiGraphics guiGraphics, int mouseX, int mouseY) {
		if (guiEventHandler != null) {
			guiEventHandler.drawForContainerScreen(screen, guiGraphics, mouseX, mouseY);
		}
	}

	private void drawBackground(Screen screen, GuiGraphics guiGraphics, int mouseX, int mouseY, float tickDelta) {
		if (guiEventHandler != null) {
			guiEventHandler.drawForScreen(screen, guiGraphics, mouseX, mouseY);
		}
	}

	public void clear() {
		this.clientInputHandler = null;
		this.guiEventHandler = null;
	}
}
